# -*- coding: utf-8 -*-

from passport.backend.api.yasms import (
    grants,
    serializers,
)
from passport.backend.api.yasms.api import PROLONG_VALID_STATUS
from passport.backend.api.yasms.errors import prolong_valid as errors
from passport.backend.api.yasms.forms import (
    NormalizablePhoneNumberValidator,
    RequiredSenderForm,
)
from passport.backend.api.yasms.utils import get_account_by_uid
from passport.backend.core import validators
from passport.backend.core.builders.yasms.utils import normalize_phone_number
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.runner.context_managers import UPDATE

from .base import YasmsJsonView


class ProlongValidForm(RequiredSenderForm):
    uid = validators.Uid()
    number = NormalizablePhoneNumberValidator()


class ProlongValidView(YasmsJsonView):
    required_grants = (grants.PROLONG_VALID,)
    basic_form = ProlongValidForm
    sensitive_fields = tuple()

    _root_form = RequiredSenderForm

    _errors = errors

    def process_request(self):
        try:
            account = get_account_by_uid(self.form_values[u'uid'], self.blackbox)
        except UnknownUid:
            account = None

        if account is not None:
            phone_number = normalize_phone_number(self.form_values[u'number'])
            with UPDATE(
                account,
                self.request.env,
                {
                    u'action': u'prolong_valid',
                    u'consumer': self.form_values[u'sender'],
                },
            ):
                response = self.yasms.prolong_valid(account, phone_number)
        else:
            response = {
                u'status': PROLONG_VALID_STATUS.NO_PHONE,
                u'uid': self.form_values[u'uid'],
            }
        self.response_values.update(response)

    @classmethod
    def format_response(cls, response):
        formatted_response = build_prolong_valid_dict(
            response[u'uid'],
            response[u'status'],
        )
        return formatted_response


def build_prolong_valid_dict(uid, status):
    return {
        u'uid': serializers.number_to_str(uid),
        u'status': status.upper(),
    }
