# -*- coding: utf-8 -*-

from datetime import datetime

import elementflow
from passport.backend.api.yasms import (
    exceptions as yasms_exceptions,
    grants,
    serializers,
)
from passport.backend.api.yasms.errors import register as errors
from passport.backend.api.yasms.forms import (
    NormalizablePhoneNumberValidator,
    RequiredSenderForm,
    SmsLanguageValidator,
    UnixtimeValidator,
)
from passport.backend.api.yasms.utils import (
    get_account_by_uid,
    get_operation_id_by_phone_number,
)
from passport.backend.core import validators
from passport.backend.core.builders.yasms.utils import normalize_phone_number
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.yasms.unbinding import unbind_old_phone
from six import BytesIO

from .base import YasmsXmlView


class RegisterForm(RequiredSenderForm):
    uid = validators.Uid()
    number = NormalizablePhoneNumberValidator(strict=True)
    lang = SmsLanguageValidator()
    secure = validators.StringBool(if_empty=False, if_missing=False)
    withoutsms = validators.StringBool(if_empty=False, if_missing=False)
    ts = UnixtimeValidator(if_empty=None, if_missing=None)
    revalidate = validators.StringBool(if_empty=False, if_missing=False)
    ignore_bindlimit = validators.StringBool(if_empty=False, if_missing=False)


class RegisterView(YasmsXmlView):
    required_grants = (grants.REGISTRATOR,)
    basic_form = RegisterForm
    sensitive_fields = (u'number',)

    _root_form = RequiredSenderForm

    _errors = errors

    def process_request(self):
        if (self.form_values[u'withoutsms'] and
                not self.grant_is_issued(grants.AWESOME_REGISTRATOR)):
            raise self._errors.NoRights()

        try:
            account = get_account_by_uid(self.form_values[u'uid'], self.blackbox)
        except UnknownUid:
            raise self._errors.NoUser()

        phone_number = normalize_phone_number(self.form_values[u'number'])

        statbox = StatboxLogger(
            action=u'register',
            consumer=self.form_values[u'sender'],
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

        timestamp = datetime.now()

        with UPDATE(
            account,
            self.request.env,
            {
                u'action': u'register',
                u'consumer': self.form_values[u'sender'],
            },
            datetime_=timestamp,
        ):
            try:
                response = self.yasms.register(
                    account=account,
                    phone_number=phone_number,
                    language=self.form_values[u'lang'],
                    revalidate=self.form_values[u'revalidate'],
                    without_sms=self.form_values[u'withoutsms'],
                    ignore_bindlimit=self.form_values[u'ignore_bindlimit'],
                    timestamp=self.form_values[u'ts'],
                    secure=self.form_values[u'secure'],
                    consumer=self.form_values[u'sender'],
                    user_ip=self.client_ip,
                    statbox=statbox,
                    user_agent=self.user_agent,
                )
            except yasms_exceptions.YaSmsSecureNumberNotAllowed:
                statbox.dump_stashes()
                raise self._errors.CantHaveSecureNumber()
            except yasms_exceptions.YaSmsAlreadyVerified:
                statbox.dump_stashes()
                raise self._errors.NumberExists()
            except yasms_exceptions.YaSmsSecureNumberExists:
                statbox.dump_stashes()
                raise self._errors.SecureNumberExists()
            except yasms_exceptions.YaSmsCodeLimitError:
                statbox.dump_stashes()
                raise self._errors.ValidationLimit()
            except yasms_exceptions.YaSmsDeliveryError:
                statbox.dump_stashes()
                raise self._errors.NoRoute()
            except yasms_exceptions.YaSmsPermanentBlock:
                statbox.dump_stashes()
                raise self._errors.PhoneNumberBlockedPermanently()
            except yasms_exceptions.YaSmsTemporaryBlock:
                statbox.dump_stashes()
                raise self._errors.PhoneNumberBlockedTemporarily()

        if account.phones.all():
            # И напоследок выполним простое механическое действие!
            operation_id = get_operation_id_by_phone_number(
                account,
                phone_number,
            )
            statbox.dump_stashes(operation_id=operation_id)

            phone = account.phones.by_number(phone_number)
            if phone and phone.bound:
                # Вызов register завершился привязанным номером, попробуем
                # отвязать старые номера.
                unbind_old_phone(
                    subject_phone=phone,
                    blackbox_builder=self.blackbox,
                    statbox=statbox,
                    consumer=self.form_values[u'sender'],
                    event_timestamp=timestamp,
                    environment=self.request.env,
                )

        self.response_values = {
            u'id': response[u'id'],
            u'number': response[u'number'],
            u'is_number_revalidated': response[u'is_revalidated'],
            u'uid': response[u'uid'],
        }

    @classmethod
    def format_response(cls, response):
        return build_register_xml(
            response[u'id'],
            response[u'number'],
            response[u'uid'],
            response[u'is_number_revalidated'],
        )


def build_register_xml(phone_id, phone_number, uid, is_phone_number_revalidated):
    out = BytesIO()
    with elementflow.xml(out, u'doc') as doc:
        attrs = {
            u'id': serializers.number_to_str(phone_id),
            u'number': phone_number,
            u'uid': serializers.number_to_str(uid),
        }
        if is_phone_number_revalidated:
            attrs[u'revalidated'] = u'1'
        else:
            attrs[u'added'] = u'1'
        doc.element(u'phone', attrs=attrs)
    xml = out.getvalue()
    return xml
