# -*- coding: utf-8 -*-

from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.api.yasms import (
    exceptions as yasms_exceptions,
    grants,
)
from passport.backend.api.yasms.errors import remove_user_phones as errors
from passport.backend.api.yasms.forms import RequiredSenderForm
from passport.backend.api.yasms.utils import (
    adapt_to_bundle_view,
    get_account_by_uid,
)
from passport.backend.core import validators
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE

from .base import (
    YasmsJsonResponse,
    YasmsJsonView,
)


class RemoveUserPhonesForm(RequiredSenderForm):
    uid = validators.Uid(if_missing=None, if_invalid=None)
    block = validators.StringBool(if_missing=False)


class RemoveUserPhonesErrorResponse(YasmsJsonResponse):
    def __init__(self, code, message, status_code=200):
        super(RemoveUserPhonesErrorResponse, self).__init__({
            u'status': u'Error',
            u'error': message,
        })
        self.status_code = status_code


@adapt_to_bundle_view
class RemoveUserPhonesView(YasmsJsonView, BundleAccountGetterMixin):
    required_grants = (grants.REMOVE_USER_PHONES,)
    basic_form = RemoveUserPhonesForm
    sensitive_fields = tuple()

    _root_form = RequiredSenderForm

    _errors = errors
    _error_response_class = RemoveUserPhonesErrorResponse

    def process_request(self):
        if self.form_values[u'uid'] is None:
            uid = self._get_uid_from_session()
        else:
            uid = self.form_values[u'uid']
        try:
            account = get_account_by_uid(uid, self.blackbox)
        except UnknownUid:
            self.response_values.update({u'status': u'OK'})
            return

        with UPDATE(
            account,
            self.request.env,
            {
                u'action': u'remove_user_phones',
                u'consumer': self.form_values[u'sender'],
            },
        ):
            try:
                self.yasms.remove_userphones(
                    account,
                    block=self.form_values[u'block'],
                    statbox=StatboxLogger(
                        action=u'remove_user_phones',
                        consumer=self.form_values[u'sender'],
                        ip=self.client_ip,
                    ),
                    consumer=self.form_values[u'sender'],
                )
            except yasms_exceptions.YaSmsOperationInapplicable:
                # Ручка не умеет удалять телефоны у 2ФАщиков, но
                # такой ошибки в её интерфейсе не предусмотрено.
                raise self._errors.InternalError()
        self.response_values.update({u'status': u'OK'})

    @classmethod
    def format_response(self, response):
        return response

    def _get_uid_from_session(self):
        session = self.check_session_cookie()
        if (session.response is None):
            raise self._errors.NoUid()
        return session.response[u'uid']
