# -*- coding: utf-8 -*-

import elementflow
from passport.backend.api.common.decorators import get_request_values
from passport.backend.api.yasms import (
    forms,
    grants,
    serializers,
)
from passport.backend.api.yasms.forms import (
    OptionalSenderForm,
    ResponseFormatValidator,
)
from passport.backend.api.yasms.utils import (
    get_account_by_uid,
    old_mask_phone_number,
)
from passport.backend.core import validators
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.types.phone_number.phone_number import mask_phone_number
from six import BytesIO

from .base import (
    BaseYasmsView,
    YasmsJsonView,
    YasmsXmlView,
)


_PHONE_ATTR_TO_SERIALIZE = {
    u'id': serializers.number_to_str,
    u'active': serializers.bool_to_onezero,
    u'secure': serializers.bool_to_onezero,
    u'cyrillic': serializers.bool_to_onezero,
    u'validation_date': serializers.datetime_to_str,
    u'validations_left': serializers.number_to_str,
    u'autoblocked': serializers.bool_to_onezero,
    u'permblocked': serializers.bool_to_onezero,
    u'blocked': serializers.bool_to_onezero,
    u'is_removing': serializers.bool_to_onezero,
}


class UserPhonesForm(OptionalSenderForm):
    uid = validators.Uid(min=1)
    format = ResponseFormatValidator()


class UserPhonesView(BaseYasmsView):
    def __init__(self, *args, **kwargs):
        super(UserPhonesView, self).__init__(*args, **kwargs)
        self._xml_view = _UserPhonesXmlView()
        self._json_view = _UserPhonesJsonView()

    def dispatch_request(self):
        unsafe_params = get_request_values()
        try:
            format_ = ResponseFormatValidator().to_python(unsafe_params.get(u'format'))
        except forms.Invalid:
            response = self._xml_view.dispatch_request()
        else:
            if format_ == u'json':
                response = self._json_view.dispatch_request()
            else:
                response = self._xml_view.dispatch_request()
        return response


class _BaseUserPhonesView(object):
    required_grants = (grants.USER_PHONES,)
    basic_form = UserPhonesForm
    sensitive_fields = (u'phones[].number', u'phones[].masked_number')

    _root_form = OptionalSenderForm

    def process_request(self):
        try:
            account = get_account_by_uid(self.form_values[u'uid'], self.blackbox)
        except UnknownUid:
            account = None

        if account is not None:
            user_phones = self.yasms.userphones(account)
        else:
            user_phones = []

        is_allowed_return_full_phone = self.grant_is_issued(grants.RETURN_FULL_PHONE)
        out_phones = []
        for phone in user_phones:
            out_phone = {
                u'id': phone[u'id'],
                u'active': phone[u'active'],
                u'secure': phone[u'secure'],
                u'cyrillic': phone[u'cyrillic'],
                u'valid': phone[u'valid'],
                u'validation_date': phone[u'validation_date'],
                u'validations_left': phone[u'validations_left'],
                u'autoblocked': phone[u'autoblocked'],
                u'permblocked': phone[u'permblocked'],
                u'blocked': phone[u'blocked'],
            }
            if is_allowed_return_full_phone:
                out_phone.update({
                    u'number': phone[u'number'],
                    u'masked_number': mask_phone_number(phone[u'number']),
                })
            else:
                out_phone.update({
                    u'number': old_mask_phone_number(phone[u'number']),
                })
            out_phones.append(out_phone)

        self.response_values = {
            u'uid': self.form_values[u'uid'],
            u'phones': out_phones,
        }


class _UserPhonesXmlView(_BaseUserPhonesView, YasmsXmlView):
    @classmethod
    def format_response(cls, response):
        return build_user_phones_xml(response[u'uid'], response[u'phones'])


def build_user_phones_xml(uid, phones):
    out = BytesIO()
    with elementflow.xml(out, u'doc') as doc:
        doc.element(u'uid', text=str(uid))
        for phone_attrs in phones:
            xml_attrs = serializers.stringify_attr_dict(
                phone_attrs,
                _PHONE_ATTR_TO_SERIALIZE,
            )
            doc.element(u'phone', attrs=xml_attrs)
    xml = out.getvalue()
    return xml


class _UserPhonesJsonView(_BaseUserPhonesView, YasmsJsonView):
    @classmethod
    def format_response(cls, response):
        return build_user_phones_dict(response[u'uid'], response[u'phones'])


def build_user_phones_dict(uid, phones):
    phones = [dict(
        p,
        validation_date=serializers.datetime_to_str(p[u'validation_date']),
    ) for p in phones]
    return {
        u'uid': uid,
        u'phones': phones,
    }
