# -*- coding: utf-8 -*-

from functools import partial

import elementflow
from passport.backend.api.yasms import (
    grants,
    serializers,
)
from passport.backend.api.yasms.forms import OptionalSenderForm
from passport.backend.api.yasms.utils import (
    get_account_by_uid,
    old_mask_phone_number,
)
from passport.backend.core import validators
from passport.backend.core.exceptions import UnknownUid
from six import BytesIO

from .base import YasmsXmlView


_VALIDATION_ATTR_TO_SERIALIZE = {
    u'validations_number': serializers.number_to_str,
    u'confirmed_date': serializers.datetime_to_str,
    u'other_accounts': serializers.number_to_str,
}


class ValidationsNumberOfUserPhonesForm(OptionalSenderForm):
    uid = validators.Uid()


class ValidationsNumberOfUserPhonesView(YasmsXmlView):
    required_grants = (grants.VALIDATIONS_NUMBER_OF_USER_PHONES,)
    basic_form = ValidationsNumberOfUserPhonesForm
    sensitive_fields = (u'validations[].number',)

    _root_form = OptionalSenderForm

    def process_request(self):
        try:
            account = get_account_by_uid(self.form_values[u'uid'], self.blackbox)
        except UnknownUid:
            account = None

        if account is not None:
            validations = self.yasms.validations_number_of_user_phones(
                account,
                partial(get_account_by_uid, blackbox_builder=self.blackbox),
            )
        else:
            validations = []

        if not self.grant_is_issued(grants.RETURN_FULL_PHONE):
            for phone in validations:
                phone[u'number'] = old_mask_phone_number(phone[u'number'])

        self.response_values[u'validations'] = validations

    @classmethod
    def format_response(cls, response):
        validations = response[u'validations']
        return build_validations_number_of_user_phones_xml(validations)


def build_validations_number_of_user_phones_xml(validations):
    out = BytesIO()
    with elementflow.xml(out, u'doc') as doc:
        for validation_attrs in validations:
            xml_attrs = serializers.stringify_attr_dict(
                validation_attrs,
                _VALIDATION_ATTR_TO_SERIALIZE,
            )
            doc.element(u'phone', attrs=xml_attrs)

    xml = out.getvalue()
    return xml
