# -*- coding: utf-8 -*-

from datetime import datetime
import logging

from passport.backend.core import validators
from passport.backend.core.builders.yasms.utils import normalize_phone_number
from passport.backend.core.types.phone_number.phone_number import (
    InvalidPhoneNumber,
    PhoneNumber,
)


log = logging.getLogger('passport.backend.api.yasms')

Invalid = validators.Invalid

_sender_regexp = u'^[-_A-z0-9.]+$'
_code_regexp = u'^0.+$|^[^0].*$'


class RequiredSenderValidator(validators.Regex):
    strip = True
    not_empty = True
    regex = _sender_regexp


class OptionalSenderValidator(validators.Regex):
    if_empty = None
    if_missing = None
    strip = True
    regex = _sender_regexp


class PhoneValidator(validators.FancyValidator):
    not_empty = True
    strip = True


class ConfirmationCodeValidator(validators.Regex):
    not_empty = True
    strip = True
    regex = _code_regexp


class NormalizablePhoneNumberValidator(validators.FancyValidator):
    not_empty = True
    strip = True

    messages = {
        u'badPhone': u'Phone number is in wrong format',
    }

    def __init__(self, strict=False, *args, **kwargs):
        super(NormalizablePhoneNumberValidator, self).__init__(*args, **kwargs)
        self._strict = strict

    def validate_python(self, phone_number, state):
        try:
            normalized = normalize_phone_number(phone_number)
            if self._strict:
                PhoneNumber.parse(normalized)
        except InvalidPhoneNumber:
            raise Invalid(self.message(u'badPhone', state), phone_number, state)


class PhoneIdValidator(validators.Int):
    not_empty = True
    strip = True
    min = 0


class UnixtimeValidator(validators.Int):
    not_empty = False
    strip = True

    messages = {
        u'tooLow': u'Unix time is too low',
        u'tooHigh': u'Unix time is too high',
    }

    def validate_python(self, unixtime, state):
        unixtime = super(UnixtimeValidator, self)._to_python(unixtime, state)
        try:
            datetime.fromtimestamp(unixtime)
        except (ValueError, OSError):
            if unixtime > 0:
                error_code = u'tooHigh'
            else:
                error_code = u'tooLow'
            raise Invalid(self.message(error_code, state), unixtime, state)


class SmsLanguageValidator(validators.DisplayLanguage):
    default_language = u'en'
    not_empty = False
    if_empty = if_missing = u'ru'


class OptionalSenderForm(validators.Schema):
    sender = OptionalSenderValidator()


class RequiredSenderForm(validators.Schema):
    sender = RequiredSenderValidator()


class ResponseFormatValidator(validators.Regex):
    strip = True
    regex = u'^xml$|^json$'
    if_empty = if_missing = u'xml'


def form_encode_invalid_to_field_and_code(exception):
    field = list(exception.error_dict.keys())[0]
    code = exception.error_dict[field].code
    return field, code
