# -*- coding: utf-8 -*-

from datetime import datetime
import logging
from operator import itemgetter

from flask import request  # noqa
from passport.backend.api.common.phone import (
    CONFIRM_METHOD_BY_SMS,
    PhoneAntifraudFeatures,
)
from passport.backend.api.common.yasms import build_route_advice
from passport.backend.core.builders.antifraud import get_antifraud_api
from passport.backend.core.builders.blackbox import get_blackbox
from passport.backend.core.builders.yasms.utils import normalize_phone_number
from passport.backend.core.conf import settings
from passport.backend.core.counters import (
    sms_per_ip,
    sms_per_ip_for_consumer,
)
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.logging_utils.loggers import DummyLogger
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.phone_number.phone_number import (
    InvalidPhoneNumber,
    PhoneNumber,
)
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.yasms.phonenumber_alias import Aliasification
from passport.backend.core.yasms.utils import get_many_accounts_with_phones_by_uids
from passport.backend.utils.common import (
    format_code_by_3,
    generate_random_code,
)
from passport.backend.utils.time import datetime_to_unixtime as to_unixtime
from six import string_types

from . import exceptions


log = logging.getLogger(u'passport.backend.api.yasms')


def get_passport_consumer_from_yasms_consumer(consumer):
    return u'old_yasms_grants_' + consumer


def old_mask_phone_number(phone):
    if phone[-4:].isdigit():
        return u'*******' + phone[-4:]
    return phone


def adapt_to_bundle_view(cls):
    def setup(self):
        super(cls, self).setup()
        self.host = self.request.env.host
        self.cookies = self.request.env.cookies
        self.client_ip = self.request.env.user_ip
        self.account = None

    @cached_property
    def blackbox(self):
        return get_blackbox()

    setattr(cls, u'setup', setup)
    setattr(cls, u'blackbox', blackbox)
    return cls


def get_account_by_uid(uid, blackbox_builder):
    """
    Исключения
        BlackboxInvalidResponseError
        BlackboxTemporaryError
        BlackboxUnknownError
        UnknownUid
    """
    accounts, unknown_uids = get_many_accounts_with_phones_by_uids([uid], blackbox_builder)
    if unknown_uids:
        raise UnknownUid()
    return accounts[0]


class Restricter(object):
    def __init__(
        self,
        confirmation_info,
        env,
        statbox,
        consumer,
        phone_number,
        track=None,
        can_handle_captcha=False,
        login_id=None,
    ):
        self._config = {
            u'ip': False,
            u'reconfirmation': False,
            u'rate': False,
        }
        self._confirmation_info = confirmation_info
        self._consumer = consumer
        self._env = env
        self._phone_number = phone_number
        self._statbox = statbox
        self._track = track
        self._user_ip = env.user_ip
        self._can_handle_captcha = can_handle_captcha
        self._login_id = login_id

    def restrict_ip(self):
        self._config[u'ip'] = True

    def restrict_reconfirmation(self):
        self._config[u'reconfirmation'] = True

    def restrict_rate(self):
        self._config[u'rate'] = True

    @property
    def resend_timeout(self):
        return settings.SMS_VALIDATION_RESEND_TIMEOUT

    def check(self):
        if self._config[u'ip']:
            if sms_per_ip_for_consumer.exists(self._consumer):
                if sms_per_ip_for_consumer.get_counter(self._consumer).hit_limit_by_ip(self._user_ip):
                    self._statbox.stash(error=u'sms_limit.exceeded', reason=u'ip_and_consumer_limit')
                    log.debug(u'Unable to send code because of IP and consumer limit')
                    raise exceptions.YaSmsIpLimitExceeded()
            elif sms_per_ip.get_counter(self._user_ip).hit_limit_by_ip(self._user_ip):
                self._statbox.stash(error=u'sms_limit.exceeded', reason=u'ip_limit')
                log.debug(u'Unable to send code because of IP limit')
                raise exceptions.YaSmsIpLimitExceeded()

        if self._config[u'reconfirmation'] and self._confirmation_info.code_confirmed:
            self._statbox.stash(error=u'phone.confirmed')
            log.debug(u'Unable to send code because it is confirmed already')
            raise exceptions.YaSmsAlreadyVerified()

        if self._config[u'rate'] and self._confirmation_info.code_last_sent:
            seconds_since_code_last_sent = (datetime.now() - self._confirmation_info.code_last_sent).total_seconds()
            if seconds_since_code_last_sent < self.resend_timeout:
                self._statbox.stash(error=u'sms_limit.exceeded', reason=u'rate_limit')
                log.debug(u'Unable to send code because it was sent recently (Passport)')
                raise exceptions.YaSmsTemporaryBlock()

        self.score_phone()

    def update(self):
        self._confirmation_info.code_send_count += 1
        self._confirmation_info.code_last_sent = datetime.now()
        if sms_per_ip_for_consumer.exists(self._consumer):
            sms_per_ip_for_consumer.get_counter(self._consumer).incr(self._user_ip)
        else:
            sms_per_ip.get_counter(self._user_ip).incr(self._user_ip)

    def score_phone(self):
        if not settings.PHONE_CONFIRM_CHECK_ANTIFRAUD_SCORE:
            return

        phone_features = PhoneAntifraudFeatures.default(
            sub_channel=self._consumer,
            user_phone_number=self._phone_number,
        )
        phone_features.phone_confirmation_method = CONFIRM_METHOD_BY_SMS
        phone_features.add_environment_features(self._env)
        if self._track:
            phone_features.add_track_features(self._track)
        if self._login_id:
            phone_features.login_id = self._login_id

        phone_features.score(
            antifraud=get_antifraud_api(),
            consumer=self._consumer,
            error_class=exceptions.YaSmsIpLimitExceeded,
            captcha_class=exceptions.YaSmsCaptchaRequiredError if self._track and self._can_handle_captcha else None,
            statbox=self._statbox,
        )


def build_default_restricter(
    confirmation_info,
    env,
    phone_number,
    consumer=None,
    statbox=None,
    login_id=None,
):
    restricter = Restricter(
        confirmation_info=confirmation_info,
        consumer=consumer,
        env=env,
        phone_number=phone_number,
        statbox=statbox,
        login_id=login_id,
    )
    restricter.restrict_ip()
    restricter.restrict_rate()
    restricter.restrict_reconfirmation()
    return restricter


class build_send_confirmation_code(object):
    """
    Строит функтор, умеющий высылать СМС с кодом, чтобы подтвердить обладание
    телефонным номером.

    Замечание: модуль ничего не записывает в статбокс, чтобы сбросить записи об
    ошибках или о хорошем завершении, нужно вызвать statbox.dump_stashes.
    """
    def __init__(
        self,
        phone_number,
        confirmation_info,
        account,
        env,
        yasms_builder,
        consumer=None,
        statbox=None,
        language=None,
        restricter=None,
        code_length=None,
        code_checks_limit=None,
        sms_template_builder=None,
        force_new_code=False,
        can_format_for_sms_retriever=None,
        gps_package_name=None,
        code_format=None,
        login_id=None,
    ):
        self._account = account
        self._can_format_for_sms_retriever = can_format_for_sms_retriever
        self._gps_package_name = gps_package_name
        self._user_ip = str(env.user_ip)
        self._user_agent = env.user_agent
        if isinstance(phone_number, string_types):
            phone_number = PhoneNumber.parse(phone_number)
        self._phone_number = phone_number
        self._yasms_builder = yasms_builder
        self._consumer = consumer
        self._statbox = statbox or DummyLogger()
        self._confirmation_info = confirmation_info
        self._code_length = code_length or settings.SMS_VALIDATION_CODE_LENGTH
        self._code_checks_limit = code_checks_limit or settings.SMS_VALIDATION_MAX_CHECKS_COUNT
        self._force_new_code = force_new_code
        self._code_format = code_format
        self._login_id = login_id

        # default language
        if language is None and self._account:
            self._language = get_preferred_language(self._account)
        else:
            self._language = language

        sms_template_builder = sms_template_builder or self._build_sms_template
        self._sms_template = sms_template_builder(language=self._language)

        if restricter is None:
            restricter = build_default_restricter(
                self._confirmation_info,
                env,
                phone_number,
                self._consumer,
                self._statbox,
                self._login_id,
            )
        self._restricter = restricter

    def __call__(self):
        """
        Высылает код.

        Выходные параметры
            Высланный код.

        Исключения
            YaSmsCodeLimitError
                С данного IP-адреса отправлено слишком много кодов для
                подтверждения обладения телефоном.
            YaSmsTemporaryBlock
                Коды отправляются слишком часто в рамках данного процесса
                подтверждения.
            YaSmsLimitExceeded
                На данный номер отправлено слишком много кодов для
                подтверждения обладения телефоном.
            YaSmsUidLimitExceeded
                Данный пользователь отправляет слишком много кодов для
                подтверждения обладения телефоном.
            YaSmsAlreadyVerified
                Пользователь уже подтвердил обладание номером в рамках данного
                процесса подтверждения (confirmation_info).
            YaSmsPermenentBlock
                Номер заблокирован
        """
        statbox_ctx = {
            u'number': self._phone_number.masked_format_for_statbox,
            u'action': self._statbox.Link(u'send_confirmation_code'),
        }
        if self._account and self._account.uid:
            statbox_ctx[u'uid'] = self._account.uid

        with self._statbox.make_context(**statbox_ctx):
            self._restricter.check()
            code = self._generate_code()
            sms_id = self._send_sms(code)
            self._restricter.update()
            self._statbox.stash(**self._build_code_sent_statline(sms_id))
            return code

    def _build_code_sent_statline(self, sms_id):
        line = dict(
            action=self._statbox.Link(u'code_sent'),
            ip=self._user_ip,
            sms_count=self._confirmation_info.code_send_count,
            sms_id=sms_id,
        )
        if self._gps_package_name is not None:
            line.update(
                gps_package_name=self._gps_package_name,
                sms_retriever=self._can_format_for_sms_retriever,
            )
        return line

    def _build_sms_template(self, language):
        return settings.translations.SMS[language][u'APPROVE_CODE']

    def _send_sms(self, code):
        kwargs = {u'identity': self._statbox.all_values[u'action']}
        if self._account and self._account.uid:
            kwargs[u'from_uid'] = self._account.uid
        if self._consumer is not None:
            kwargs[u'caller'] = self._consumer

        try:
            response = self._yasms_builder.send_sms(
                self._phone_number.e164,
                self._sms_template,
                text_template_params={'code': str(code)},
                route=build_route_advice(self._phone_number, self._consumer),
                used_gate_ids=self._confirmation_info.yasms_used_gate_ids,
                client_ip=self._user_ip,
                user_agent=self._user_agent,
                **kwargs
            )
            self._confirmation_info.yasms_used_gate_ids = response.get('used_gate_ids')
            return response['id']
        except exceptions.YaSmsLimitExceeded:
            self._statbox.stash(error=u'sms_limit.exceeded', reason=u'yasms_phone_limit')
            log.debug(u'Unable to send code because of phone number limit')
            raise
        except exceptions.YaSmsUidLimitExceeded:
            self._statbox.stash(error=u'sms_limit.exceeded', reason=u'yasms_uid_limit')
            log.debug(u'Unable to send code because of UID limit')
            raise
        except exceptions.YaSmsTemporaryBlock:
            self._statbox.stash(error=u'sms_limit.exceeded', reason=u'yasms_rate_limit')
            log.debug(u'Unable to send code because it was sent recently (send_sms)')
            raise
        except exceptions.YaSmsPermanentBlock:
            log.debug(u'Unable to send code because it is blocked permanently')
            self._statbox.stash(error=u'phone.blocked')
            raise
        except exceptions.YaSmsError:
            self._statbox.stash(error=u'sms.isnt_sent')
            raise

    def _generate_code(self):
        if (self._force_new_code or
                not self._confirmation_info.code_value or
                self._confirmation_info.code_checks_count >= self._code_checks_limit):
            self._confirmation_info.code_value = generate_random_code(self._code_length)
            self._confirmation_info.code_checks_count = 0
            if self._code_format == 'by_3_dash':
                self._confirmation_info.code_value = format_code_by_3(self._confirmation_info.code_value, delimiter='-')
        return self._confirmation_info.code_value


class build_legacy_send_confirmation_code(build_send_confirmation_code):
    def __init__(
        self,
        phone_number,
        confirmation_info,
        account,
        env,
        yasms_builder,
        consumer=None,
        statbox=None,
        language=None,
        restricter=None,
        login_id=None,
    ):
        super(build_legacy_send_confirmation_code, self).__init__(
            phone_number,
            confirmation_info,
            account,
            env,
            yasms_builder,
            consumer,
            statbox,
            language,
            restricter,
            settings.SMS_LEGACY_VALIDATION_CODE_LENGTH,
            login_id,
        )


def get_operation_id_by_phone_number(account, phone_number):
    """
    Возвращает id операции, которая происходит над телефоном.

    Когда нет операции, или телефона, или телефонов возвращается None.
    """
    phone_number = normalize_phone_number(phone_number)

    try:
        phone_on_account = account.phones.by_number(phone_number)
    except InvalidPhoneNumber:
        return

    if phone_on_account is None:
        return

    # Статбокс не нужен, т.к. никаких действий с операцией мы делать не будем.
    logical_op = phone_on_account.get_logical_operation(statbox=None)
    if not logical_op:
        return
    return logical_op.id


def does_binding_allow_washing(account, phone_number, binding_time, should_ignore_binding_limit, bindings_history):
    """
    Вычисляет высказывание "можно ли обелить учётную запись account, после
    того как к ней привязали телефон phone_number в момент binding_time с
    флагом should_ignore_binding_limit".

    В bindings_history должна быть история привязок данного номера. Но могут
    быть и истории привязок других номеров.

    Входные параметры:
        phone_number
            Номер телефона (строка), который привязали к учётной записи.
        binding_time
            Время (Datetime), когда номер привязали

    Замечания
        bindings_history -- это результат вызова phone_bindings_history,
        но не результат вызова phone_bindings (они отличаются по форме).
    """
    if account.karma.prefix == settings.KARMA_PREFIX_SPAMMER:
        # PASSP-15566
        # Аккаунт в ручном режиме разметили как спамера -- им можно обелиться
        # только через службу поддержки.
        return False

    if account.karma.is_washed():
        return False

    bindings_history = bindings_history[u'history']
    bindings_history = [b for b in bindings_history if b[u'phone'] == phone_number]

    bindings_count = len(bindings_history)

    # ЧЯ может не знать о связи учётной записи с номером, если
    # транзакция ещё не завершилась.
    account_binding = {
        u'uid': account.uid,
        u'phone': phone_number,
        u'ts': to_unixtime(binding_time),
    }
    if not should_ignore_binding_limit and account_binding not in bindings_history:
        bindings_count += 1

    # Проверим, что номер "чистый".
    return bindings_count <= settings.YASMS_VALIDATION_LIMIT


def aliasify(account, phone_number, blackbox, statbox, consumer, action, language=None):
    """
    Создаёт телефонный алиас на данной учётной записи, удаляет телефонный
    алиас с прежней учётной записи и извещает пользователей.

    Входные параметры
        account
            Учётная запись для которой создаётся алиас
        phone_number
            Номер телефона (объект PhoneNumber) из которого образуется алиас
        language
            Язык который понимает пользователь учётной записи account

    Выходные параметры
        Учётная запись бывшего владельца алиаса или None.

    Исключения
        BlackboxInvalidResponseError
        BlackboxTemporaryError
        BlackboxUnknownError
        DBError

    Замечания
        Возможны гонки в результате которых алиас на учётной записи будет, но
        программа выбросит исключение DbError (смотрите тесты).
    """
    global request

    try:
        aliasification = Aliasification(account, phone_number, consumer, blackbox, DummyLogger(), language)
    except Aliasification.AliasNotAllowed:
        return

    prev_owner = aliasification.get_owner()

    datetime_ = datetime.now()
    events = {u'action': action, u'consumer': consumer}

    with UPDATE(
        account,
        request.env,
        events,
        datetime_=datetime_,
        initiator_uid=account.uid,
    ):
        aliasification.take_old_alias_away()

    if prev_owner is not None:
        with UPDATE(
            prev_owner,
            request.env,
            events,
            datetime_=datetime_,
            initiator_uid=account.uid,
        ):
            aliasification.take_away()

    with UPDATE(
        account,
        request.env,
        events,
        datetime_=datetime_,
        initiator_uid=account.uid,
    ):
        aliasification.give_out()

    aliasification.notify()

    return prev_owner


def remove_equal_phone_bindings(history):
    """
    Удаляет одинаковые вехи из телефонной истории.

    Телефонная история -- это список словарей с ключами
        uid
            Идентификатор пользователя

        phone
            Телефонный номер

        ts
            Unixtime
    """
    def timestamps_equal(t1, t2):
        return abs(t1 - t2) <= 1

    clean_history = []
    timestamp_table = {}
    for landmark in sorted(history, key=itemgetter(u'ts')):
        uid = landmark[u'uid']
        phone_number = landmark[u'phone']

        if (uid, phone_number) in timestamp_table:
            timestamps = timestamp_table[(uid, phone_number)]
            timestamp = landmark[u'ts']
            if not timestamps_equal(timestamps[-1], timestamp):
                landmark_is_copy = False
                timestamps.append(timestamp)
            else:
                landmark_is_copy = True
        else:
            timestamp_table[(uid, phone_number)] = [landmark[u'ts']]
            landmark_is_copy = False

        if not landmark_is_copy:
            clean_history.append(landmark)

    return clean_history
