# -*- coding: utf-8 -*-
from datetime import timedelta
import logging
import re
import subprocess

from passport.backend.ch_stat_loader.ch_stat_loader import settings


log = logging.getLogger(__name__)


def send_juggler_event(service, status, description=''):
    """
        Правило в juggler:
            https://juggler.yandex-team.ru/notification_rules/?query=rule_id=5b9273370ac06800972e6b4f
        Правило в juggler для дебага:
            https://juggler.yandex-team.ru/notification_rules/?query=rule_id=5b992055cee02a00950b6c12
    """
    log.debug('Sending juggler event: service="{service}", status="{status}", description="{desc}"'.format(
        service=service,
        status=status,
        desc=description,
    ))
    assert status in ['OK', 'WARN', 'CRIT'], 'Incorrect status \'{}\''.format(status)
    assert re.match('^[a-zA-Z0-9_-]*$', service), 'Bad service name: "{}"'.format(service)
    assert len(description) < 1000, 'Too long descriptionfor juggler event: "{}"'.format(description)

    cmd = [
        'juggler_queue_event',
        '--host', 'AM-NOTIFICATION',
        '--service', service,
        '--status', status,
        '--tags', 'am_notification' if not settings.DEBUG else 'am_notification_test',
        '--description', description,
    ]
    proc = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    proc.wait()
    juggler_output = ''.join(list(proc.stderr))
    assert 'Accepted events: 1' in juggler_output, 'Sending juggler event failed.\nArguments:{}.\nError:{}'.format(cmd, juggler_output)


class AMNotificator(object):
    def __init__(self, statface_client):
        self._notify_rules = dict()
        self._active_notifications = None
        self.statface_client = statface_client

    def set_active_notifications(self, notifications):
        self._active_notifications = notifications

    def is_notification_active(self, name):
        return self._active_notifications is None or name in self._active_notifications

    def add_notify_rule(self, name, func):
        """
        name - строка, латиница без пробелов
        func - функция с параметром date, вычисляющая description для события для даты date.
                Должна возвращать None если уведомление не нужно
        """
        assert name not in self._notify_rules, 'Duplicate rule name \'{}\''.format(name)
        self._notify_rules[name] = func

    def reset_notifications(self):
        for name in self._notify_rules:
            if not self.is_notification_active(name):
                continue
            send_juggler_event(name, 'OK')

    def send_notifications(self, date_start, date_end):
        for name in self._notify_rules:
            if not self.is_notification_active(name):
                continue
            descs = []
            for offset in range((date_end - date_start).days):
                target_date = date_start + timedelta(days=offset)
                desc = self._notify_rules[name](self.statface_client, target_date)
                if desc is not None:
                    descs.append(desc)
            if len(descs) > 0:
                send_juggler_event(name, 'WARN', '\n' + '\n\n'.join(descs))


def get_report_data_for_date(client, path, date):
    logging.disable(logging.DEBUG)
    report = client.get_report(path)
    data = report.download_data('d', fielddate=date)
    logging.disable(logging.NOTSET)
    assert len(data) > 0, 'No data in report {} for {}'.format(path, date)
    return data


def am_max_version_change(statface_client, target_date):
    target_data = get_report_data_for_date(statface_client, 'Passport.All/own/am/system/max_am_versions', target_date)
    target_versions = {'%s (%s)' % (r['app_id'], r['app_platform']): (r['max_am_ver'], r['event_count']) for r in target_data}
    prev_data = get_report_data_for_date(statface_client, 'Passport.All/own/am/system/max_am_versions', target_date - timedelta(days=1))
    prev_versions = {'%s (%s)' % (r['app_id'], r['app_platform']): (r['max_am_ver'], r['event_count']) for r in prev_data}

    def is_less(ver1, ver2):
        if ver1 == ver2:
            return False
        if ver2 is None or ver2[0] == '<':
            return False
        if ver1 is None or ver1[0] == '<':
            return True
        ver1 = list(map(int, ver1.split('.')))
        ver2 = list(map(int, ver2.split('.')))
        for v1, v2 in zip(ver1, ver2):
            if v1 < v2:
                return True
            if v1 > v2:
                return False
        return len(ver1) < len(ver2)

    result = []
    for app in target_versions:
        curr_ver, curr_count = target_versions.get(app, (None, 0))
        prev_ver, prev_count = prev_versions.get(app, (None, 0))
        if curr_ver == prev_ver or is_less(curr_ver, prev_ver):
            continue
        if prev_ver is None and curr_ver == '<4.70':
            continue
        if curr_count + prev_count < 200:
            continue
        result.append('{app}: {prev} -> {curr}'.format(
            app=app.encode('utf-8'),
            prev=prev_ver,
            curr=curr_ver,
        ))
    if len(result) > 0:
        return '\n'.join([str(target_date)] + result)


def autologin_from_smartlock(statface_client, target_date):
    def get_app_ids_for_date(date):
        data = get_report_data_for_date(statface_client, 'Passport.All/own/am/notify_info/autologin_from_smartlock', date)
        return {r['app_id'] for r in data}
    target_apps = get_app_ids_for_date(target_date)
    prev_apps = set().union(*[get_app_ids_for_date(target_date - timedelta(days=d)) for d in range(1, 8)])
    result = []
    for app in target_apps:
        if app not in prev_apps:
            result.append('New app: {app}'.format(
                app=app,
            ))
    if len(result) > 0:
        return '\n'.join([str(target_date)] + result)


def get_AMnotificator(statface_client):
    notificator = AMNotificator(statface_client)

    notificator.add_notify_rule('am-max-version-change', am_max_version_change)
    notificator.add_notify_rule('autologin-from-smartlock', autologin_from_smartlock)

    return notificator
