# -*- coding: utf-8 -*-

from passport.backend.ch_stat_loader.ch_stat_loader import query
from passport.backend.ch_stat_loader.ch_stat_loader.utils import escape_name


def build_generic_count_per_unique_device_report(description, group_by_app, group_by_am_version,
                                                 titles, measures=None, source_events=None, conditions_names=None,
                                                 extra_dimensions=None,
                                                 query=query.UniqueDeviceEventCountQuery,
                                                 **kwargs):
    name = 'unique_events_per_dev'
    dimensions = [
        'fielddate',
        'app_platform',
        'os_version',
    ]
    if group_by_app:
        name += '_per_app_id'
        dimensions.append('app_id')
    if group_by_am_version:
        name += '_per_am_version'
        dimensions.append('am_version')
    if extra_dimensions:
        dimensions += extra_dimensions

    query_pos_args = (conditions_names, source_events) if conditions_names and source_events else tuple()
    additional_kwargs = kwargs
    return {
        'types': ['custom'],
        'description': description,
        'name': name,
        'dimensions': dimensions,
        'measures': [escape_name(event) for event in source_events or measures],
        'titles': titles,
        'query': lambda *args, **kwargs: query(
            group_by_app=group_by_app,
            group_by_am_version=group_by_am_version,
            *(query_pos_args + args),
            **dict(kwargs, **additional_kwargs)
        ),
    }


def build_count_per_unique_device_report_configs(titles, source_events=None, query=None,
                                                 extra_dimensions=None,
                                                 conditions_names=None, measures=None,
                                                 **kwargs):
    base_kwargs = dict(
        titles=titles,
        **kwargs
    )
    if source_events and conditions_names:
        base_kwargs.update(
            source_events=source_events,
            conditions_names=conditions_names,
        )
    else:
        base_kwargs['measures'] = measures
    if query:
        base_kwargs['query'] = query
    if extra_dimensions:
        base_kwargs['extra_dimensions'] = extra_dimensions
    configs = []
    for description, group_by_app, group_by_am_version in (
            (u'События по уникальным устройствам (per AM, per AppID)', True, True),
            (u'События по уникальным устройствам (per AM)', False, True),
            (u'События по уникальным устройствам (per AppID)', True, False),
            (u'События по уникальным устройствам', False, False),
    ):
        configs.append(
            build_generic_count_per_unique_device_report(
                description=description,
                group_by_app=group_by_app,
                group_by_am_version=group_by_am_version,
                **base_kwargs
            ),
        )
    return configs


def build_swc_count_per_unique_device_report_configs():
    source_events = [
        'auth.swc.set_mode',
        'auth.swc.import_try',
        'auth.swc.import_error',
        'auth.swc.import_cancel',
        'auth.swc.import_success',
        'auth.swc.save_success',
        'auth.swc.save_fail',
    ]
    conditions_names = []
    for event in source_events:
        conditions_names.append(
            (
                'EventName=\'%s\'' % event,
                escape_name(event),
            ),
        )
    titles = {
        'auth_swc_set_mode': u'Установок mode',
        'auth_swc_import_try': u'Попыток импорта',
        'auth_swc_import_error': u'Ошибок импорта',
        'auth_swc_import_cancel': u'Отказов от импорта',
        'auth_swc_import_success': u'Успешных импортов',
        'auth_swc_save_success': u'Успешных сохранений',
        'auth_swc_save_fail': u'Ошибок при сохранении',
    }
    return build_count_per_unique_device_report_configs(
        source_events=source_events,
        conditions_names=conditions_names,
        titles=titles,
    )


def build_smartlock_count_per_unique_device_report_configs():
    source_events = [
        'auth.smartlock.set_mode',
        'auth.smartlock.import_try',
        'auth.smartlock.import_error',
        'auth.smartlock.import_cancel',
        'auth.smartlock.import_success',
        'auth.smartlock.save_success',
        'auth.smartlock.save_fail',
    ]
    conditions_names = []
    for event in source_events:
        conditions_names.append(
            (
                'EventName=\'%s\'' % event,
                escape_name(event),
            ),
        )
    titles = {
        'auth_smartlock_set_mode': u'Установок mode',
        'auth_smartlock_import_try': u'Попыток импорта',
        'auth_smartlock_import_error': u'Ошибок импорта',
        'auth_smartlock_import_cancel': u'Отказов от импорта',
        'auth_smartlock_import_success': u'Успешных импортов',
        'auth_smartlock_save_success': u'Успешных сохранений',
        'auth_smartlock_save_fail': u'Ошибок при сохранении',
    }
    return build_count_per_unique_device_report_configs(
        source_events=source_events,
        conditions_names=conditions_names,
        titles=titles,
    )

AM_REPORTS = [
    {
        'name': 'system',
        'description': u'Системные события',
        'reports': [
            {
                'types': ['custom'],
                'name': 'sectoral',
                'description': u'Версии АМ и слейвы',
                'dimensions': [
                    'fielddate',
                    'app_platform',
                    'os_version',
                    'am_version',
                ],
                'measures': [
                    'devices_count',
                    # 10.11 нет нужных событий, андроид не реализован
                ],
                'query': query.AMVersionsQuery,
                'titles': {
                    'am_version': u'Версия AM',
                    'devices_count': u'Число устройств',
                },
            },
            {
                'types': ['custom'],
                'name': 'max_am_versions',
                'description': u'Максимальные версии АМ по приложениям (по метрике)',
                'dimensions': [
                    'fielddate',
                    'app_platform',
                    'app_id',
                ],
                'measures': [
                    'max_am_ver',
                    'event_count',
                ],
                'titles': {
                    'max_am_ver': u'Максимальная версия АМ',
                    'event_count': u'Число событий с заданным App ID',
                },
                'custom_types': {
                    'max_am_ver': 'string',
                },
                'query': query.MaxAMVersionPerAppIdQuery,
            },
        ],
    },
    {
        'name': 'auth',
        'description': u'Авторизация',
        'reports': [
            {
                'types': [
                    'generic_daily',
                    'generic_hourly',
                ],
                'queries': {
                    'generic_daily': query.NoPhonishDailyEventCountQuery,
                    'generic_hourly': query.NoPhonishHouryEventCountQuery,
                },
                'measures': [
                    'auth.launch',
                    'auth.auth_try',
                    'auth.auth_success',
                    'auth.auth_fail',
                    'auth.forget_login',
                    'auth.registration',
                    'auth.phonish.get_phone_code',
                    'auth.phonish.send_code',
                    'auth.cancel',
                ],
                'titles': {
                    'auth.launch': u'Число открытий экрана авторизации',
                    'auth.auth_try': u'Число попыток авторизации',
                    'auth.auth_success': u'Число успешных авторизаций',
                    'auth.auth_fail': u'Число неуспешных авторизаций',
                    'auth.forget_login': u'Кликов по кнопке "забыл пароль"',
                    'auth.registration': u'Кликов по кнопке "регистрация"',
                    'auth.phonish.get_phone_code': u'Кликов по кнопке "получить код"',
                    'auth.phonish.send_code': u'Кликов по кнопке "далее" при получении кода',
                    'auth.cancel': u'Кликов по кнопке "отмена"',
                },
            },
            {
                'types': ['custom'],
                'name': 'conversion_per_device',
                'description': u'Конверсия авторизаций (по уникальным устройствам)',
                'dimensions': [
                    'fielddate',
                    'app_platform',
                    'os_version',
                ],
                'measures': [
                    'dev_with_launches',
                    'dev_with_successes',
                    'conversion',
                ],
                'titles': {
                    'dev_with_launches': u'Число устройств с открытиями экрана авторизации',
                    'dev_with_successes': u'Число устройств с успешным прохождением авторизации',
                    'conversion': u'Конверсия',
                },
                'query': lambda *args, **kwargs: query.NoPhonishConversionQuery(
                    launch_event='auth.launch',
                    success_event='auth.auth_success',
                    group_by_app=False,
                    *args,
                    **kwargs
                ),
            },
            {
                'types': ['custom'],
                'description': u'Конверсия авторизаций по приложениям (по уникальным устройствам)',
                'name': 'conversion_per_device_per_app_id',
                'dimensions': [
                    'fielddate',
                    'app_platform',
                    'os_version',
                    'app_id',
                ],
                'measures': [
                    'dev_with_launches',
                    'dev_with_successes',
                    'conversion',
                ],
                'titles': {
                    'dev_with_launches': u'Число устройств с открытиями экрана авторизации',
                    'dev_with_successes': u'Число устройств с успешным прохождением авторизации',
                    'conversion': u'Конверсия',
                },
                'query': lambda *args, **kwargs: query.NoPhonishConversionQuery(
                    launch_event='auth.launch',
                    success_event='auth.auth_success',
                    group_by_app=True,
                    *args,
                    **kwargs
                ),
            },
            {
                'types': ['custom'],
                'description': u'Конверсия авторизаций по версиям АМ и по приложениям (по уникальным устройствам)',
                'name': 'conversion_per_device_per_app_id_per_am_version',
                'dimensions': [
                    'fielddate',
                    'app_platform',
                    'os_version',
                    'app_id',
                    'am_version',
                ],
                'measures': [
                    'dev_with_launches',
                    'dev_with_successes',
                    'conversion',
                ],
                'titles': {
                    'dev_with_launches': u'Число устройств с открытиями экрана авторизации',
                    'dev_with_successes': u'Число устройств с успешным прохождением авторизации',
                    'conversion': u'Конверсия',
                },
                'query': lambda *args, **kwargs: query.NoPhonishConversionQuery(
                    launch_event='auth.launch',
                    success_event='auth.auth_success',
                    group_by_app=True,
                    group_by_am_version=True,
                    *args,
                    **kwargs
                ),
            },
            {
                'types': ['custom'],
                'description': u'Авторизации по типам (без группировки)',
                'name': 'auths_by_type',
                'dimensions': [
                    'fielddate',
                    'app_platform',
                    'os_version',
                    'am_version',
                    'app_id',
                    'subtype',
                ],
                'measures': [
                    'attempts_count',
                    'success_count',
                    'conversion',
                ],
                'titles': {
                    'subtype': u'Тип авторизации',
                    'attempts_count': u'Число попыток авторизоваться',
                    'success_count': u'Число успешных авторизаций',
                    'conversion': u'Конверсия',
                },
                'query': query.NoPhonishAuthTypesQuery,
            },
        ],
    },
    {
        'name': 'carousel',
        'description': u'Карусель',
        'reports': [
            {
                'types': [
                    'generic_daily',
                ],
                'titles': {
                    'carousel.launch': u'Число открытий экрана',
                    'carousel.swipe': u'Число свайпов',
                    'carousel.choose_account': u'Число выборов аккаунта для входа',
                    'carousel.add_account_try': u'Число кликов по кнопке "добавить аккаунт"',
                    'carousel.cancel': u'Число кликов по кнопке "отмена"',
                },
                'measures': [
                    'carousel.launch',
                    'carousel.swipe',
                    'carousel.choose_account',
                    'carousel.add_account_try',
                    'carousel.cancel',
                ],
            },
            {
                'types': ['custom'],
                'description': u'Среднее число аккаунтов в карусели (по устройствам за неделю)',
                'name': 'avg_accounts_per_device',
                'dimensions': [
                    'fielddate',
                    'app_platform',
                    'os_version',
                ],
                'measures': [
                    'avg',
                    'sum',
                    'count',
                ],
                'titles': {
                    'sum': u'Суммарное число аккаунтов',
                    'avg': u'Среднее число аккаунтов на устройстве',
                    'count': u'Число устройств за неделю',
                },
                'query': query.AvgCarouselAccountsPerDeviceQuery,
            },
        ],
    },
    {
        'name': 'register',
        'description': u'Регистрация',
        'reports': [
            {
                'types': [
                    'generic_daily',
                ],
                'queries': {
                    'generic_daily': query.RegDailyEventCountQuery,
                },
                'measures': [
                    'reg.launch',
                    'reg.error.login_exist',
                    'reg.error.login_prohibited_symbols',
                    'reg.error.short_password',
                    'reg.error.password_prohibited_symbols',
                    'reg.error.password_confirm_incorrect',
                    'reg.error.incorrect_phone',
                    'reg.success',
                    'reg.phone.cancel',
                    'reg.choose_type',
                    'reg.phone.try',
                    # Кликов по кнопке "отмена" - нет события
                    # События с дополнительными ограничениями:
                    'reg_success_phone',
                    'reg_success_question',
                ],
                'titles': {
                    'reg.launch': u'Число открытий экрана регистрации',
                    'reg.error.login_exist': u'Число ошибок: логин существует',
                    'reg.error.login_prohibited_symbols': u'Число ошибок: запрещенные символы в логине',
                    'reg.error.short_password': u'Число ошибок: короткий пароль',
                    'reg.error.password_prohibited_symbols': u'Число ошибок: запрещенные символы в пароле',
                    'reg.error.password_confirm_incorrect': u'Число ошибок: повтор пароля не совпадает',
                    'reg.error.incorrect_phone': u'Число ошибок: неверный номер телефона',
                    'reg.success': u'Число успешных регистраций',
                    'reg_success_phone': u'Число успешных регистраций (phone)',
                    'reg_success_question': u'Число успешных регистраций (question)',
                    'reg.phone.cancel': u'Число возвратов с экрана ввода кода',
                    'reg.choose_type': u'Кликов по кнопкам выбора типа регистрации',
                    'reg.phone.try': u'Кликов по кнопке "Далее" при получении кода',
                },
            },
            {
                'types': ['custom'],
                'name': 'conversion_per_device',
                'description': u'Конверсия регистраций (по уникальным устройствам)',
                'dimensions': [
                    'fielddate',
                    'app_platform',
                    'os_version',
                ],
                'measures': [
                    'dev_with_launches',
                    'dev_with_successes',
                    'conversion',
                ],
                'titles': {
                    'dev_with_launches': u'Число устройств с открытиями экрана регистрации',
                    'dev_with_successes': u'Число устройств с успешным прохождением регистрации',
                    'conversion': u'Конверсия',
                },
                'query': lambda *args, **kwargs: query.NoPhonishConversionQuery(
                    launch_event='reg.launch',
                    success_event='reg.success',
                    group_by_app=False,
                    *args,
                    **kwargs
                ),
            },
            {
                'types': ['custom'],
                'description': u'Конверсия регистраций по приложениям (по уникальным устройствам)',
                'name': 'conversion_per_device_per_app_id',
                'dimensions': [
                    'fielddate',
                    'app_platform',
                    'os_version',
                    'app_id',
                ],
                'measures': [
                    'dev_with_launches',
                    'dev_with_successes',
                    'conversion',
                ],
                'titles': {
                    'dev_with_launches': u'Число устройств с открытиями экрана регистрации',
                    'dev_with_successes': u'Число устройств с успешным прохождением регистрации',
                    'conversion': u'Конверсия',
                },
                'query': lambda *args, **kwargs: query.NoPhonishConversionQuery(
                    launch_event='reg.launch',
                    success_event='reg.success',
                    group_by_app=True,
                    *args,
                    **kwargs
                ),
            },
            {
                'types': ['custom'],
                'description': u'Конверсия регистраций по версиям АМ (по уникальным устройствам)',
                'name': 'conversion_per_device_per_am_version',
                'dimensions': [
                    'fielddate',
                    'app_platform',
                    'os_version',
                    'am_version',
                ],
                'measures': [
                    'dev_with_launches',
                    'dev_with_successes',
                    'conversion',
                ],
                'titles': {
                    'dev_with_launches': u'Число устройств с открытиями экрана регистрации',
                    'dev_with_successes': u'Число устройств с успешным прохождением регистрации',
                    'conversion': u'Конверсия',
                },
                'query': lambda *args, **kwargs: query.NoPhonishConversionQuery(
                    launch_event='reg.launch',
                    success_event='reg.success',
                    group_by_am_version=True,
                    *args,
                    **kwargs
                ),
            },
            {
                'types': ['custom'],
                'description': u'Конверсия регистраций по версиям АМ и по приложениям (по уникальным устройствам)',
                'name': 'conversion_per_device_per_app_id_per_am_version',
                'dimensions': [
                    'fielddate',
                    'app_platform',
                    'os_version',
                    'app_id',
                    'am_version',
                ],
                'measures': [
                    'dev_with_launches',
                    'dev_with_successes',
                    'conversion',
                ],
                'titles': {
                    'dev_with_launches': u'Число устройств с открытиями экрана регистрации',
                    'dev_with_successes': u'Число устройств с успешным прохождением регистрации',
                    'conversion': u'Конверсия',
                },
                'query': lambda *args, **kwargs: query.NoPhonishConversionQuery(
                    launch_event='reg.launch',
                    success_event='reg.success',
                    group_by_app=True,
                    group_by_am_version=True,
                    *args,
                    **kwargs
                ),
            },
            {
                'types': ['custom'],
                'description': u'Регистрации по типам (без группировки)',
                'name': 'regs_by_type',
                'dimensions': [
                    'fielddate',
                    'app_platform',
                    'os_version',
                    'am_version',
                    'app_id',
                    'subtype',
                ],
                'measures': [
                    'attempts_count',
                    'success_count',
                    'conversion',
                ],
                'titles': {
                    'subtype': u'Тип регистрации',
                    'attempts_count': u'Число открытий экрана регистрации (общее для разных типов)',
                    'success_count': u'Число успешных регистраций',
                    'conversion': u'Конверсия',
                },
                'query': query.NoPhonishRegTypesQuery,
            },
        ],
    },
    {
        'name': 'swc',
        'description': u'SWC',
        'reports': [
            {
                'types': [
                    'generic_daily',
                    'generic_hourly',
                ],
                'measures': [
                    'auth.swc.set_mode',
                    'auth.swc.import_try',
                    'auth.swc.import_error',
                    'auth.swc.import_cancel',
                    'auth.swc.import_success',
                    'auth.swc.save_success',
                    'auth.swc.save_fail',
                ],
                'titles': {
                    'auth.swc.set_mode': u'Установок mode',
                    'auth.swc.import_try': u'Попыток импорта',
                    'auth.swc.import_error': u'Ошибок импорта',
                    'auth.swc.import_cancel': u'Отказов от импорта',
                    'auth.swc.import_success': u'Успешных импортов',
                    'auth.swc.save_success': u'Успешных сохранений',
                    'auth.swc.save_fail': u'Ошибок при сохранении',
                },
            },
        ] + build_swc_count_per_unique_device_report_configs(),
    },
    {
        'name': 'smartlock',
        'description': u'Smartlock',
        'reports': [
            {
                'types': [
                    'generic_daily',
                    'generic_hourly',
                ],
                'measures': [
                    'auth.smartlock.set_mode',
                    'auth.smartlock.import_try',
                    'auth.smartlock.import_error',
                    'auth.smartlock.import_cancel',
                    'auth.smartlock.import_success',
                    'auth.smartlock.save_success',
                    'auth.smartlock.save_fail',
                ],
                'titles': {
                    'auth.smartlock.set_mode': u'Установок mode',
                    'auth.smartlock.import_try': u'Попыток импорта',
                    'auth.smartlock.import_error': u'Ошибок импорта',
                    'auth.smartlock.import_cancel': u'Отказов от импорта',
                    'auth.smartlock.import_success': u'Успешных импортов',
                    'auth.smartlock.save_success': u'Успешных сохранений',
                    'auth.smartlock.save_fail': u'Ошибок при сохранении',
                },
            },
        ] + build_smartlock_count_per_unique_device_report_configs(),
    },
    {
        'name': 'auth_sdk',
        'description': u'Auth SDK',
        'reports': [
            {
                'types': ['custom'],
                'description': u'Auth SDK по AM',
                'name': 'auth_sdk_am',
                'dimensions': [
                    'fielddate',
                    'app_platform',
                    'os_version',
                    'app_id',
                    'reporter',
                ],
                'measures': [
                    'loginsdk_show_scopes',
                    'loginsdk_accept',
                    'loginsdk_decline',
                    'loginsdk_account_added',
                ],
                'titles': {
                    'reporter': 'reporter',
                    'loginsdk_show_scopes': 'loginsdk_show_scopes',
                    'loginsdk_accept': 'loginsdk_accept',
                    'loginsdk_decline': 'loginsdk_decline',
                    'loginsdk_account_added': 'loginsdk_account_added',
                },
                'query': lambda *args, **kwargs: query.LoginSDKUniqueDeviceEventCountQuery(
                    *args, **kwargs
                ),
            },
        ],
    },
    {
        'name': 'new_flow',
        'description': u'Новый флоу',
        'reports': [
            {
                'types': ['custom'],
                'name': 'authorizations_by_xtoken',
                'priority': 20,
                'description': u'авторизации по xtoken',
                'dimensions': [
                    'fielddate',
                    'app_id',
                    'app_platform',
                    'source',
                ],
                'measures': [
                    'count',
                ],
                'titles': {
                    'source': u'from',
                    'count': u'count',
                },
                'query': lambda *args, **kwargs: query.AuthorizationsByXtoken(
                    *args,
                    **kwargs
                ),
            },
        ],
    },
    {
        'name': 'quality',
        'description': 'Качество',
        'reports': [
            {
                'types': ['custom'],
                'name': 'broadcast',
                'description': u'broadcast',
                'dimensions': [
                    'fielddate',
                ],
                'measures': [
                    'sent_events',
                    'received_events',
                    'sent_devices',
                    'received_devices',
                ],
                'titles': {
                    'sent_events': u'sent_events',
                    'received_events': u'received_events',
                    'sent_devices': u'sent_devices',
                    'received_devices': u'received_devices',
                },
                'query': lambda *args, **kwargs: query.BroadcastQuery(
                    *args,
                    **kwargs
                ),
            },
        ],
    },
    {
        'name': 'notify_info',
        'description': 'Данные для уведомлений',
        'reports': [
            {
                'types': ['custom'],
                'name': 'autologin_from_smartlock',
                'description': u'autologin_from_smartlock',
                'dimensions': [
                    'fielddate',
                    'app_id',
                ],
                'measures': [
                    'count',
                ],
                'titles': {
                    'count': u'count',
                },
                'query': lambda *args, **kwargs: query.AutologinFromSmartlockQuery(
                    *args,
                    **kwargs
                ),
            },
        ],
    },
    {
        'name': 'live',
        'description': 'live',
        'reports': [
            {
                'types': ['custom'],
                'name': 'sso',
                'description': u'sso',
                'dimensions': [
                    'fielddate',
                    'app_id',
                    'event_name',
                    'timestamp',
                    'am_version',
                ],
                'measures': [
                    'count',
                ],
                'titles': {
                    'event_name': u'event_name',
                    'timestamp': u'timestamp',
                    'count': u'count',
                },
                'query': lambda *args, **kwargs: query.SSOEventsQuery(
                    *args,
                    **kwargs
                ),
            },
        ],
    },
]
