# -*- coding: utf-8 -*-
import copy
from datetime import timedelta
import httplib
import json
import logging
import re

from passport.backend.ch_stat_loader.ch_stat_loader import settings
from passport.backend.ch_stat_loader.ch_stat_loader.utils import (
    cached,
    escape_name,
    retriable_n,
)
import requests


httplib._MAXHEADERS = 1000
log = logging.getLogger('ch_stat_loader.query')

API_KEYS = set()


def set_api_keys(api_keys):
    API_KEYS.update(api_keys)


CLIENT_ID_MAPPING_GETTER = None


def set_client_id_mapping_getter(getter):
    global CLIENT_ID_MAPPING_GETTER
    CLIENT_ID_MAPPING_GETTER = getter


@cached
def get_client_id_mapping(target_date):
    assert CLIENT_ID_MAPPING_GETTER is not None
    return CLIENT_ID_MAPPING_GETTER(target_date)


@cached
def get_login_sdk_frequent_reporters(target_date):
    query = LoginSDKFrequentReportersQuery(date_start=target_date, date_end=target_date + timedelta(days=1),
                                           report=None)
    result = query.execute()
    row = result.get_slice(['reporters'])[0]
    frequent_reporters = set(row['reporters'])
    return frequent_reporters


class RetriableClickhouseException(ValueError):
    pass


class QuotaExceededClickhouseException(ValueError):
    pass


@retriable_n(10, 10, exceptions=(RetriableClickhouseException,))
def get_clickhouse_data(query, connection_timeout=1500):
    r = requests.post(
        settings.CLICKHOUSE_HOST,
        auth=(settings.CLICKHOUSE_USER, settings.CLICKHOUSE_PASSWORD),
        params={
            'distributed_aggregation_memory_efficient': 1,
        },
        data=re.sub(' +', ' ', query).encode('utf-8'),
        timeout=connection_timeout,
    )
    if r.status_code == 200:
        return r.text
    else:
        def get_exception_class():
            # коды из https://github.com/ClickHouse/ClickHouse/blob/master/dbms/src/Common/ErrorCodes.cpp
            # 62 - синтаксическая ошибка
            # Code: 160, e.displayText() = DB::Exception: Estimated query execution time is too long.
            # 201 - квота исчерпана
            # 307 - Limit for (uncompressed) bytes to read exceeded
            for code in [62, 160, 201, 307]:
                prefix = 'Code: %s' % code
                if r.text[:len(prefix)] == prefix:
                    if code == 201:
                        return QuotaExceededClickhouseException
                    return ValueError
            return RetriableClickhouseException
        raise get_exception_class()('code: %s\nmessage: %s\nquery: %s' % (r.status_code, r.text, query))


def make_clickhouse_request(raw_query):
    log.debug('CH request: %s', raw_query)
    data = get_clickhouse_data(raw_query)
    return CHResponse(data)


def is_value_empty(value):
    return not value or value == '0' or value == 0


class DataResponse(object):
    def get_slice(self, names, skip_if_all_empty=None):
        raise NotImplementedError()


class CHResponse(DataResponse):
    def __init__(self, raw):
        self.parsed = json.loads(raw)
        self.index = {item['name']: index for (index, item) in enumerate(self.parsed['meta'])}

    def get_slice(self, names, skip_if_all_empty=None):
        skip_if_all_empty = skip_if_all_empty or []
        result = []
        for item in self.parsed['data']:
            values = {escape_name(name): item[self.index[escape_name(name)]] for name in names}
            if skip_if_all_empty and all(is_value_empty(values[escape_name(name)]) for name in skip_if_all_empty):
                continue
            result.append(values)
        return result


def make_source_events_filter(source_events):
    return '(%s)' % ' OR '.join([
        'EventName IN (%s)' % ','.join(
            ['\'%s\'' % event for event in source_events if '%' not in event]),
    ] + [
        'EventName LIKE \'%s\'' % event for event in source_events if '%' in event
    ])


def make_am_version_comparator(extractor1, c, extractor2):
    return 'arrayMap(x -> toUInt64OrZero(x), splitByChar(\'.\', %s)) %s arrayMap(x -> toUInt64OrZero(x), splitByChar(\'.\', %s))' % (extractor1, c, extractor2)


class Query(object):
    can_be_joined = False
    extra_filtering_clauses = None

    def __init__(self, date_start, date_end, report, events=None):
        if events:
            self.events_map = {event: {report} for event in events}
        self.date_start = date_start
        self.date_end = date_end
        self.report = report

    def build(self):
        raise NotImplementedError()

    def join(self, query):
        raise NotImplementedError()

    def execute(self):
        raw_query = self.build()
        return make_clickhouse_request(raw_query)

    def notify_reports(self, execute_results):
        reports_map = {}
        if hasattr(self, 'events_map'):
            for event, reports in self.events_map.iteritems():
                for report in reports:
                    reports_map.setdefault(report, []).append(event)
        else:
            for measure in self.report.measures:
                assert isinstance(measure, str), (measure, self.report.path)
            reports_map[self.report] = self.report.measures
        for report, measures in reports_map.iteritems():
            data = execute_results.get_slice(report.dimensions + measures)
            report.set_data(data)

    @property
    def priority(self):
        if hasattr(self, 'events_map'):
            all_reports = set()
            for _, reports in self.events_map.iteritems():
                all_reports |= reports
            return max([r.priority for r in all_reports])
        else:
            return self.report.priority


def fix_broken_ParsedParams(cl):
    """
    Другой способ отправки некоторых событий в АМ для андроида
        core.activation с версии 7.1
        auth.autologin.finish и auth.autologin.start в версиях [7.1 ... 7.3)
    """
    oldbuild = cl.build

    def newbuild(self):
        def make_key_checker(key):
            return '(%s or %s)' % (
                'has(ParsedParams.Key1, \'%s\')' % key,
                'visitParamHas(EventValue, \'%s\')' % key,
            )

        def make_key_getter(key):
            return 'if(%s, %s, %s)' % (
                'has(ParsedParams.Key1, \'%s\')' % key,
                'arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, \'%s\'))' % key,
                'visitParamExtractString(EventValue, \'%s\')' % key,
            )

        request = oldbuild(self)
        request = re.sub(
            r"has\(ParsedParams.Key1, '(\w+)'\)",
            lambda m: make_key_checker(m.group(1)),
            request
        )
        request = re.sub(
            r"arrayElement\(ParsedParams.Key2, indexOf\(ParsedParams.Key1, '(\w+)'\)\)",
            lambda m: make_key_getter(m.group(1)),
            request
        )
        return request

    cl.build = newbuild
    return cl


class APIKeysQuery(Query):
    def build(self):
        return ur"""
            SELECT
                groupUniqArray(APIKey) as api_keys
            FROM %(table)s
            WHERE (StartDate >= toDate('%(date_start)s'))
                AND (StartDate < toDate('%(date_end)s'))
                AND (EventDate >= toDate('%(date_start)s'))
                AND (EventDate <= toDate('%(date_end)s'))
                AND EventType=4
                AND EventName='auth.launch'
                AND AppID not in (%(blacklisted_app_ids)s)
            FORMAT JSONCompact
        """ % dict(
            table=settings.EVENTS_TABLE,
            date_start=self.date_start,
            date_end=self.date_end,
            blacklisted_app_ids=(
                ','.join(map(lambda app_id: '\'%s\'' % app_id, settings.BLACKLISTED_APP_IDS))
            ) if settings.BLACKLISTED_APP_IDS else '',

        )


class EventCountQuery(Query):
    can_be_joined = True

    def __init__(self, *args, **kwargs):
        super(EventCountQuery, self).__init__(*args, **kwargs)
        self.extra_conditions_names = []
        self.custom_measures = set()
        self.hidden_measures = set()
        self.extra_select_clauses = []
        self.extra_group_fields = []

    @property
    def join_key(self):
        return '%s.%s.%s' % (self.__class__.__name__, self.date_start, self.date_end)

    def join(self, query):
        if self.join_key != query.join_key:
            raise ValueError('Join key mismatch')
        for event, reports in query.events_map.iteritems():
            self.events_map.setdefault(event, set()).update(reports)

    def build(self):
        counted_event_names = set(self.events_map.keys()) - set(self.custom_measures)
        event_names = counted_event_names.union(self.hidden_measures)
        select_clauses = [
            'countIf(EventName=\'%s\') as %s' % (event, escape_name(event))
            for event in counted_event_names
        ] + [
            'countIf(%s) as %s' % (condition, name)
            for (condition, name) in self.extra_conditions_names
        ] + self.extra_select_clauses
        return ur"""
            SELECT
                'group0' as group,
                StartDate as fielddate,
                AppPlatform as app_platform,
                AppID as app_id,
                %(select_clauses)s
            FROM %(table)s
            WHERE (StartDate >= toDate('%(date_start)s'))
                AND (StartDate < toDate('%(date_end)s'))
                AND (EventDate >= toDate('%(date_start)s'))
                AND (EventDate <= toDate('%(date_end)s'))
                AND APIKey IN (%(api_keys)s)
                AND EventType=4
                AND has(ParsedParams.Key1, 'am_version')
                AND EventName IN (%(event_names)s)
                AND AppID not in (%(blacklisted_app_ids)s)
                AND OSVersion != '' AND AppPlatform != ''
                %(extra_filtering_clauses)s
            GROUP BY StartDate, AppPlatform, AppID%(extra_group_fields)s
            FORMAT JSONCompact
            """ % dict(
            table=settings.EVENTS_TABLE,
            select_clauses=',\n                '.join(sorted(select_clauses)),
            date_start=self.date_start,
            date_end=self.date_end,
            api_keys=', '.join(map(str, API_KEYS)),
            event_names=', '.join(map(lambda event: '\'%s\'' % event, event_names)),
            blacklisted_app_ids=(
                ','.join(map(lambda app_id: '\'%s\'' % app_id, settings.BLACKLISTED_APP_IDS))
            ) if settings.BLACKLISTED_APP_IDS else '',
            extra_filtering_clauses=self.extra_filtering_clauses or '',
            extra_group_fields=(', '.join([''] + sorted(self.extra_group_fields))) if len(self.extra_group_fields) > 0 else '',
        )


class DailyEventCountQuery(EventCountQuery,):
    def __init__(self, *args, **kwargs):
        super(DailyEventCountQuery, self).__init__(*args, **kwargs)
        self.extra_select_clauses.extend([
            'arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, \'am_version\')) as am_version',
            'OSVersion as os_version',
        ])
        self.extra_group_fields.extend([
            'am_version',
            'os_version',
        ])


class HourlyEventCountQuery(EventCountQuery):
    def __init__(self, *args, **kwargs):
        super(HourlyEventCountQuery, self).__init__(*args, **kwargs)
        self.extra_select_clauses.extend([
            'arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, \'am_version\')) as am_version',
            'OSVersion as os_version',
            'toHour(toDateTime(StartTimestamp + StartTimeZone + EventTimeOffset)) as hour',
        ])
        self.extra_group_fields.extend([
            'am_version',
            'os_version',
            'hour',
        ])


NO_PHONISH_FILTERING_CLAUSES = 'AND AppID NOT IN (%s)' % ','.join(map(lambda s: "'%s'" % s, settings.PHONISH_APP_IDS))


class NoPhonishAppsFilteringMixin(Query):
    extra_filtering_clauses = NO_PHONISH_FILTERING_CLAUSES


class RegEventCountQueryMixin(EventCountQuery):
    def __init__(self, *args, **kwargs):
        super(RegEventCountQueryMixin, self).__init__(*args, **kwargs)
        self.custom_measures = {
            'reg_success_phone',
            'reg_success_question',
        }
        type_getter = 'arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, \'type\'))'
        for reg_type in ['phone', 'question']:
            self.extra_conditions_names.append(
                (
                    'EventName=\'reg.success\' AND %s=\'%s\'' % (type_getter, reg_type),
                    'reg_success_%s' % reg_type,
                ),
            )


class PhonishEventCountQueryMixin(EventCountQuery):
    def __init__(self, *args, **kwargs):
        super(PhonishEventCountQueryMixin, self).__init__(*args, **kwargs)
        self.custom_measures = {
            'auth_launch_aggregate',
            'auth_success_aggregate',
            'auth_cancel_step_1',
            'auth_cancel_step_2',
        }
        self.hidden_measures = {
            'auth.cancel',
            'auth.launch',
            'reg.launch',
            'auth.auth_success',
            'reg.success',
        }
        step_getter = 'arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, \'step\'))'
        for step in ['1', '2']:
            self.extra_conditions_names.append(
                (
                    'EventName=\'auth.cancel\' AND %s=\'%s\'' % (step_getter, step),
                    'auth_cancel_step_%s' % step,
                ),
            )
        self.extra_conditions_names.extend([
            (
                'EventName=\'auth.launch\' '
                'AND arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, \'is_back\')) IN (\'false\', \'0\') '
                'OR EventName=\'reg.launch\'',
                'auth_launch_aggregate',
            ),
            (
                'EventName=\'auth.auth_success\' OR EventName=\'reg.success\'',
                'auth_success_aggregate',
            ),
        ])


class PhonishExpEventCountQueryMixin(PhonishEventCountQueryMixin):
    def __init__(self, *args, **kwargs):
        super(PhonishExpEventCountQueryMixin, self).__init__(*args, **kwargs)
        self.extra_group_fields = ['phone_exp']
        self.extra_select_clauses = ['arrayElement([\'base\', \'experiment\'], toInt8(arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, \'exp_taxiphonish\')))) as phone_exp']
        self.extra_filtering_clauses = (self.extra_filtering_clauses or '') + """
            AND has(ParsedParams.Key1, 'exp_taxiphonish')
            AND arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'exp_taxiphonish')) IN ('1', '2')
        """


class NoPhonishDailyEventCountQuery(DailyEventCountQuery, NoPhonishAppsFilteringMixin):
    pass


class NoPhonishHouryEventCountQuery(HourlyEventCountQuery, NoPhonishAppsFilteringMixin):
    pass


class RegDailyEventCountQuery(DailyEventCountQuery, RegEventCountQueryMixin, NoPhonishAppsFilteringMixin):
    pass


class AMVersionsQuery(Query):
    def build(self):
        date_start = self.date_end - timedelta(days=7)
        return ur"""
            SELECT
                '%(target_date)s' as fielddate,
                AppPlatform as app_platform,
                OSVersion as os_version,
                arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'am_version')) as am_version,
                uniq(DeviceIDHash) as devices_count
            FROM %(table)s
            WHERE (StartDate >= toDate('%(date_start)s'))
                AND (StartDate <= toDate('%(target_date)s'))
                AND (EventDate >= toDate('%(date_start)s'))
                AND (EventDate <= toDate('%(target_date)s'))
                AND APIKey IN (%(api_keys)s)
                AND EventType=4
                AND has(ParsedParams.Key1, 'am_version')
                AND EventName IN ('auth.launch', 'domik.identifier.open')
                AND AppID not in (%(blacklisted_app_ids)s)
                AND OSVersion != '' AND AppPlatform != ''
            GROUP BY AppPlatform, OSVersion, am_version
            FORMAT JSONCompact
        """ % dict(
            table=settings.EVENTS_TABLE,
            date_start=date_start,
            target_date=self.date_end - timedelta(days=1),
            api_keys=', '.join(map(str, API_KEYS)),
            blacklisted_app_ids=(
                ','.join(map(lambda app_id: '\'%s\'' % app_id, settings.BLACKLISTED_APP_IDS))
            ) if settings.BLACKLISTED_APP_IDS else '',
        )


class MaxAMVersionPerAppIdQuery(Query):
    def build(self):
        date_start = self.date_end - timedelta(days=3)
        return ur"""
            SELECT
                '%(target_date)s' as fielddate,
                AppPlatform as app_platform,
                AppID as app_id,
                max(
                  arrayMap(x -> toUInt64OrZero(x), splitByChar('.',
                    has(ParsedParams.Key1, 'am_version') ? arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'am_version')): '0'
                  ))
                ) as tmp,
                length(tmp)=1 and tmp[1]=0? '<4.70':arrayStringConcat(arrayMap(x -> toString(x), tmp), '.') as max_am_ver,
                count(*) as event_count
            FROM %(table)s
            WHERE (StartDate >= toDate('%(date_start)s'))
                AND (StartDate <= toDate('%(target_date)s'))
                AND (EventDate >= toDate('%(date_start)s'))
                AND (EventDate <= toDate('%(target_date)s'))
                AND APIKey IN (%(api_keys)s)
                AND EventType=4
                AND AppID not in (%(blacklisted_app_ids)s)
                AND OSVersion != '' AND AppPlatform != ''
            GROUP BY AppPlatform, AppID
            HAVING event_count >= 100
            ORDER BY AppPlatform, AppID
            FORMAT JSONCompact
        """ % dict(
            table=settings.EVENTS_TABLE,
            date_start=date_start,
            target_date=self.date_end - timedelta(days=1),
            api_keys=', '.join(map(str, API_KEYS)),
            blacklisted_app_ids=(
                ','.join(map(lambda app_id: '\'%s\'' % app_id, settings.BLACKLISTED_APP_IDS))
            ) if settings.BLACKLISTED_APP_IDS else '',
        )


class GenericUniqueDeviceQuery(Query):
    def __init__(self, *args, **kwargs):
        super(GenericUniqueDeviceQuery, self).__init__(*args, **kwargs)
        self.internal_select_clauses = set()
        self.external_select_clauses = set()
        self.extra_internal_group_fields = set()
        self.external_group_fields = set()
        self.source_events = set()
        self.days = 1
        self.sample_ratio = 1
        self.internal_having_condition = None

    def build(self):
        date_start = self.date_end - timedelta(days=self.days)
        assert len(self.internal_select_clauses) > 0
        return ur"""
            SELECT
                '%(target_date)s' as fielddate,
                %(external_select_clauses)s
            FROM (
                SELECT
                    any(AppPlatform) as AppPlatform1,
                    any(OSVersion) as OSVersion1,
                    any(arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'am_version'))) as am_version,
                    %(internal_select_clauses)s
                FROM %(table)s
                %(sampling_clause)s
                WHERE (StartDate >= toDate('%(date_start)s'))
                    AND (StartDate <= toDate('%(target_date)s'))
                    AND (EventDate >= toDate('%(date_start)s'))
                    AND (EventDate <= toDate('%(target_date)s'))
                    AND APIKey IN (%(api_keys)s)
                    AND EventType=4
                    AND has(ParsedParams.Key1, 'am_version')
                    AND %(source_events_filter)s
                    AND AppID not in (%(blacklisted_app_ids)s)
                    AND OSVersion != '' AND AppPlatform != ''
                    %(extra_filtering_clauses)s
                GROUP BY DeviceIDHash%(extra_internal_group_fields)s
                %(internal_having_clause)s
            )
            %(external_group_clause)s
            FORMAT JSONCompact
        """ % dict(
            table=settings.EVENTS_TABLE,
            date_start=date_start,
            target_date=self.date_end - timedelta(days=1),
            api_keys=', '.join(map(str, API_KEYS)),
            source_events_filter=make_source_events_filter(self.source_events),
            extra_filtering_clauses=self.extra_filtering_clauses or '',
            blacklisted_app_ids=(
                ','.join(map(lambda app_id: '\'%s\'' % app_id, settings.BLACKLISTED_APP_IDS))
            ) if settings.BLACKLISTED_APP_IDS else '',
            internal_select_clauses=',\n                    '.join(sorted(self.internal_select_clauses)),
            external_select_clauses=',\n                '.join(sorted(self.external_select_clauses)),
            extra_internal_group_fields=', '.join([''] + sorted(self.extra_internal_group_fields)) if len(
                self.extra_internal_group_fields) > 0 else '',
            external_group_clause=('GROUP BY %s' % ', '.join(sorted(self.external_group_fields))) if len(
                self.external_group_fields) > 0 else '',
            sampling_clause='' if self.sample_ratio == 1 else 'SAMPLE {}'.format(self.sample_ratio),
            internal_having_clause=('HAVING %s' % self.internal_having_condition) if self.internal_having_condition else '',
        )


NEW_FLOW_AM_VERSION_FILTERING_CLAUSE = """
    AND (
        (AppPlatform = 'android' AND %(am_version_filter_android)s)
        OR
        (AppPlatform = 'iOS' AND %(am_version_filter_ios)s)
    )
""" % dict(
    am_version_filter_android=make_am_version_comparator(
        'arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, \'am_version\'))', '>=', '\'6.5.0\''),
    am_version_filter_ios=make_am_version_comparator(
        'arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, \'am_version\'))', '>=', '\'4.220\''),
)


class AuthorizationsByXtoken(GenericUniqueDeviceQuery):
    def __init__(self, *args, **kwargs):
        super(AuthorizationsByXtoken, self).__init__(*args, **kwargs)
        self.internal_select_clauses.update([
            'arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, \'from\')) as source',
            'AppID as AppID1',
            'count(*) as count',
        ])
        self.external_select_clauses.update([
            'AppPlatform1 as app_platform',
            'AppID1 as app_id',
            'source',
            'countIf(count > 0) as count',
        ])
        self.extra_internal_group_fields.update(['AppID', 'source'])
        self.external_group_fields.update(['app_platform', 'app_id', 'source'])
        self.source_events.add('core.get_xtoken')
        self.extra_filtering_clauses = """
            AND arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'from')) != ''
        """ + NEW_FLOW_AM_VERSION_FILTERING_CLAUSE


class AvgCarouselAccountsPerDeviceQuery(GenericUniqueDeviceQuery):
    def __init__(self, *args, **kwargs):
        super(AvgCarouselAccountsPerDeviceQuery, self).__init__(*args, **kwargs)
        self.internal_select_clauses.add('max(toUInt64(arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, \'existing_accounts_count\')))) as acc_count')
        self.external_select_clauses.update([
            'AppPlatform1 as app_platform',
            'OSVersion1 as os_version',
            'sum(acc_count) as sum',
            'avg(acc_count) as avg',
            'count() as count',
        ])
        self.external_group_fields.update(['AppPlatform1', 'OSVersion1'])
        self.source_events.add('carousel.launch')
        self.days = 7
        self.extra_filtering_clauses = (self.extra_filtering_clauses or '') + 'AND has(ParsedParams.Key1, \'existing_accounts_count\')'


class ActionDurationQuery(Query):
    def __init__(self, points, date_start, date_end, report):
        super(ActionDurationQuery, self).__init__(
            date_start=date_start,
            date_end=date_end,
            report=report,
        )
        self.points_map = {}
        points = copy.deepcopy(points)
        for point in points:
            point.setdefault('reports', set()).add(report)
            self.points_map[self.get_point_key(point)] = point

    @staticmethod
    def get_point_key(point):
        left = point['left_event']
        is_first_left = point['is_first_left']
        right = point['right_event']
        is_first_right = point['is_first_right']
        key = '%s_%s_b_%s_%s' % (
            escape_name(left),
            int(is_first_left),
            escape_name(right),
            int(is_first_right),
        )
        if 'deny_intermediate_events' in point:
            deny = '_'.join(map(escape_name, sorted(point['deny_intermediate_events'])))
            key += '_d_%s' % deny
        return key

    def build(self):
        points = self.points_map.values()
        event_names = set([point['left_event'] for point in points] + [point['right_event'] for point in points])
        outer_select_clauses = []
        regex_select_clauses = []
        having_clauses = []
        for point in points:
            left = point['left_event']
            is_first_left = point['is_first_left']
            right = point['right_event']
            is_first_right = point['is_first_right']
            name_suffix = self.get_point_key(point)
            filtered_events = [left, right]
            if 'deny_intermediate_events' in point:
                filtered_events.extend(list(point['deny_intermediate_events']))

            def build_filter_func(include):
                position_clauses = ['position(x, \'%sN\')=%d' % (event, int(include)) for event in filtered_events]
                joiner = ' or ' if include else ' and '
                return joiner.join(position_clauses)

            expressions = [
                # Отделить из общего массива заданные записи
                'arrayFilter(x-> %(initial_filter_func)s, groupedArray) as grouped_%(sfx)s',
                # Объединить записи в строку для дальнейшей обработки Regexp-ами
                'arrayStringConcat(grouped_%(sfx)s, \' \') as str1_%(sfx)s',
            ]
            # Схлопнуть последовательные одинаковые события в одно, оставив первое или последнее вхождение
            fold_first = 'replaceRegexpAll(%(from)s, \'(%(ev)sN\\\\d+)( %(ev)sN\\\\d+)*\', \'\\\\1\') as %(to)s'
            fold_last = 'replaceRegexpAll(%(from)s, \'(%(ev)sN\\\\d+ )*(%(ev)sN\\\\d+)\', \'\\\\2\') as %(to)s'
            left_expr = fold_first if is_first_left else fold_last
            expressions.append(left_expr % {
                'from': 'str1_%(sfx)s',
                'ev': '%(left_reg_esc)s',
                'to': 'str2_%(sfx)s',
            })
            right_expr = fold_first if is_first_right else fold_last
            expressions.append(right_expr % {
                'from': 'str2_%(sfx)s',
                'ev': '%(right_reg_esc)s',
                'to': 'str3_%(sfx)s',
            })
            expressions.extend([
                # Схлопнуть соседние пары событий в запись разности
                'replaceRegexpAll(str3_%(sfx)s, \'%(left_reg_esc)sN(\\\\d+) %(right_reg_esc)sN(\\\\d+)\', \'\\\\2-\\\\1\') as str4_%(sfx)s',
                # Распакуем строку в массив
                'splitByChar(\' \', str4_%(sfx)s) as splitted_%(sfx)s',
                # Могли остаться события без пары - нужно удалить их
                'arrayFilter(x-> match(x, \'\\\\d+-\\\\d+\'), splitted_%(sfx)s) as filtered_%(sfx)s',
                # Подсчитать разности, записать их в массив
                'arrayMap(item -> toFloat64(splitByChar(\'-\', item)[1]) - toFloat64(splitByChar(\'-\', item)[2]), filtered_%(sfx)s) as final_array_%(sfx)s',
                # Подсчитаем сумму значений и общее число значений в сессии
                'arraySum(final_array_%(sfx)s) as sess_sum_%(sfx)s',
                'length(final_array_%(sfx)s) as sess_count_%(sfx)s',
            ])

            expressions = ',\n'.join(expressions) % dict(
                initial_filter_func=build_filter_func(include=True),
                left=left,
                left_reg_esc=left.replace('.', r'\.'),
                right=right,
                right_reg_esc=right.replace('.', r'\.'),
                sfx=name_suffix,
            )
            regex_select_clauses.append(expressions)
            outer_select_clauses.append(
                # Считаем сумму всех значений, общее число всех значений и среднее
                'sum(sess_sum_%(sfx)s) as sum_%(sfx)s, '
                'sum(sess_count_%(sfx)s) as count_%(sfx)s, '
                'sum_%(sfx)s / count_%(sfx)s as avg_%(sfx)s' % dict(sfx=name_suffix),
            )
            having_clauses.append('(count_%(sfx)s > 0)' % dict(sfx=name_suffix))

        return ur"""
            SELECT
                '%(date_start)s' as fielddate,
                AppPlatform as app_platform,
                OSVersion as os_version,
                am_version,
                AppID as app_id,
                group,
                %(outer_select_clauses)s
            FROM (
                SELECT
                    any(AppPlatform) as AppPlatform,
                    any(OSVersion) as OSVersion,
                    any(am_version) as am_version,
                    AppID,
                    'group0' as group,
                    groupArray(EventTimestamp) as TsArray,
                    groupArray(EventName) as EventArray,
                    arrayMap(event, index -> concat(concat(event, 'N'), toString(index)), EventArray, TsArray) as groupedArray,
                    %(regex_select_clauses)s
                FROM (
                    SELECT
                        DeviceIDSessionIDHash,
                        AppPlatform,
                        OSVersion,
                        arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'am_version')) as am_version,
                        AppID,
                        StartTimestamp + EventTimeOffset as EventTimestamp,
                        EventName
                    FROM %(table)s
                    WHERE (StartDate >= toDate('%(date_start)s'))
                        AND (StartDate < toDate('%(date_end)s'))
                        AND (EventDate >= toDate('%(date_start)s'))
                        AND (EventDate <= toDate('%(date_end)s'))
                        AND APIKey IN (%(api_keys)s)
                        AND EventType=4
                        AND has(ParsedParams.Key1, 'am_version')
                        AND EventName IN (%(event_names)s)
                        AND AppID not in (%(blacklisted_app_ids)s)
                        AND OSVersion != '' AND AppPlatform != ''
                        %(extra_filtering_clauses)s
                    ORDER BY EventTimestamp
                )
                GROUP BY DeviceIDSessionIDHash, AppID
            )
            GROUP BY AppPlatform, OSVersion, am_version, AppID, group
            HAVING %(having_clause)s
            FORMAT JSONCompact
        """ % dict(
            table=settings.EVENTS_TABLE,
            outer_select_clauses=',\n'.join(outer_select_clauses),
            regex_select_clauses=',\n'.join(regex_select_clauses),
            having_clause=' OR '.join(having_clauses),
            date_start=self.date_start,
            date_end=self.date_end,
            api_keys=', '.join(map(str, API_KEYS)),
            event_names=', '.join(map(lambda event: '\'%s\'' % event, event_names)),
            extra_filtering_clauses=self.extra_filtering_clauses or '',
            blacklisted_app_ids=(
                ','.join(map(lambda app_id: '\'%s\'' % app_id, settings.BLACKLISTED_APP_IDS))
            ) if settings.BLACKLISTED_APP_IDS else '',
        )

    can_be_joined = True

    @property
    def join_key(self):
        return '%s.%s.%s' % (self.__class__.__name__, self.date_start, self.date_end)

    def join(self, query):
        if self.join_key != query.join_key:
            raise ValueError('Join key mismatch')
        for point_key, point in query.points_map.iteritems():
            self.points_map.setdefault(point_key, point)['reports'].update(point['reports'])

    def notify_reports(self, execute_results):
        reports_map = {}
        for point_key, point in self.points_map.iteritems():
            for report in point['reports']:
                reports_map.setdefault(report, []).append(point)

        for report, points in reports_map.iteritems():
            names = []
            for point in points:
                sfx = self.get_point_key(point)
                names += ['avg_%(sfx)s' % dict(sfx=sfx), 'count_%(sfx)s' % dict(sfx=sfx)]

            data = execute_results.get_slice(report.dimensions + names, skip_if_all_empty=names)
            log.debug('notify %s (%s)', report.path, report.date_start)
            report.set_data(data)


class RegPhoneCodeDurationQuery(NoPhonishAppsFilteringMixin):
    can_be_joined = False

    def build(self):
        event_names = {'reg.success', 'reg.phone.launch', 'reg.choose_type'}
        return ur"""
            SELECT
                '%(date_start)s' as fielddate,
                AppPlatform as app_platform,
                OSVersion as os_version,
                am_version,
                AppID as app_id,
                group,
                sum(sess_sum) as sum_%(key)s,
                sum(sess_count) as count_%(key)s,
                sum_%(key)s / count_%(key)s as avg_%(key)s
            FROM (
                SELECT
                    any(AppPlatform) as AppPlatform,
                    any(OSVersion) as OSVersion,
                    any(am_version) as am_version,
                    AppID,
                    'group0' as group,
                    groupArray(EventTimestamp) as TsArray,
                    groupArray(event_name_w_type) as EventArray,
                    arrayMap(event, index -> concat(concat(event, 'N'), toString(index)), EventArray, TsArray) as groupedArray,
                    arrayStringConcat(groupedArray, ' ') as str1,
                    replaceRegexpAll(str1, '(reg\.phone\.launchN\\d+)( reg\.phone\.launchN\\d+)*', '\\1') as str2,
                    replaceRegexpAll(str2, '(reg\.successN\\d+)( reg\.successN\\d+)*', '\\1') as str3,
                    replaceRegexpAll(str3, 'reg\.phone\.launchN(\\d+) reg\.successN(\\d+)', '\\2-\\1') as str4,
                    splitByChar(' ', str4) as splitted,
                    arrayFilter(x-> match(x, '\\d+-\\d+'), splitted) as filtered,
                    arrayMap(item -> toFloat64(splitByChar('-', item)[1]) - toFloat64(splitByChar('-', item)[2]), filtered) as final_array,
                    arraySum(final_array) as sess_sum,
                    length(final_array) as sess_count
                FROM (
                    SELECT
                        DeviceIDSessionIDHash,
                        AppPlatform,
                        OSVersion,
                        arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'am_version')) as am_version,
                        AppID,
                        StartTimestamp + EventTimeOffset as EventTimestamp,
                        EventName,
                        if(indexOf(ParsedParams.Key1, 'type') > 0, arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'type')), '') as reg_type,
                        if(EventName='reg.choose_type', concat(EventName, '_', reg_type), EventName) as event_name_w_type
                    FROM %(table)s
                    WHERE (StartDate >= toDate('%(date_start)s'))
                        AND (StartDate < toDate('%(date_end)s'))
                        AND (EventDate >= toDate('%(date_start)s'))
                        AND (EventDate <= toDate('%(date_end)s'))
                        AND APIKey IN (%(api_keys)s)
                        AND EventType=4
                        AND has(ParsedParams.Key1, 'am_version')
                        AND EventName IN (%(event_names)s)
                        AND AppID not in (%(blacklisted_app_ids)s)
                        AND OSVersion != '' AND AppPlatform != ''
                        %(extra_filtering_clauses)s
                    ORDER BY EventTimestamp
                )
                GROUP BY DeviceIDSessionIDHash, AppID
            )
            GROUP BY AppPlatform, OSVersion, am_version, AppID, group
            HAVING (count_%(key)s > 0)
            FORMAT JSONCompact
        """ % dict(
            table=settings.EVENTS_TABLE,
            key=self.get_point_key(),
            date_start=self.date_start,
            date_end=self.date_end,
            api_keys=', '.join(map(str, API_KEYS)),
            event_names=', '.join(map(lambda event: '\'%s\'' % event, event_names)),
            extra_filtering_clauses=self.extra_filtering_clauses or '',
            blacklisted_app_ids=(
                ','.join(map(lambda app_id: '\'%s\'' % app_id, settings.BLACKLISTED_APP_IDS))
            ) if settings.BLACKLISTED_APP_IDS else '',
        )

    @staticmethod
    def get_point_key(point=None):
        return 'reg_phone_code_duration_query'

    def notify_reports(self, execute_results):
        names = map(lambda name: '%s_%s' % (name, self.get_point_key()), ['sum', 'count', 'avg'])
        data = execute_results.get_slice(self.report.dimensions + names)
        self.report.set_data(data)


class ConversionQuery(GenericUniqueDeviceQuery):
    def __init__(self, launch_event=None, success_event=None, group_by_app=False, group_by_am_version=False,
                 days=7, source_events=None, launch_condition=None, success_condition=None,
                 *args, **kwargs):
        super(ConversionQuery, self).__init__(*args, **kwargs)
        self.days = days
        self.source_events = source_events or {launch_event, success_event}
        self.internal_select_clauses.update([
            'countIf(%s) > 0 as has_launches' % (launch_condition or ('EventName=\'%s\'' % launch_event)),
            'countIf(%s) > 0 as has_successes' % (success_condition or ('EventName=\'%s\'' % success_event)),
            'AppID as AppID1',
        ])
        self.external_select_clauses.update([
            'AppPlatform1 as app_platform',
            'OSVersion1 as os_version',
            'sum(toUInt64(has_launches)) as dev_with_launches',
            'sum(toUInt64(has_successes)) as dev_with_successes',
            'dev_with_successes / dev_with_launches as conversion',
        ])
        self.external_group_fields.update(['AppPlatform1', 'OSVersion1'])
        self.extra_internal_group_fields.update(['AppID'])
        self.internal_having_condition = 'has_launches'

        if group_by_app:
            self.external_group_fields.add('AppID1')
            self.external_select_clauses.add('AppID1 as app_id')
        if group_by_am_version:
            self.external_group_fields.add('am_version')
            self.external_select_clauses.add('am_version')


class NoPhonishConversionQuery(ConversionQuery, NoPhonishAppsFilteringMixin):
    pass


class UniqueDeviceEventCountQuery(Query):
    def __init__(self, conditions_names, source_events, days=1,
                 group_by_os_version=True, group_by_app=False, group_by_am_version=False,
                 extra_internal_select_clauses=[], extra_external_select_clauses=[], extra_group_fields=[],
                 sample_ratio=1,
                 *args, **kwargs):
        super(UniqueDeviceEventCountQuery, self).__init__(*args, **kwargs)
        self.days = days
        self.group_by_app = group_by_app
        self.group_by_am_version = group_by_am_version
        self.group_by_os_version = group_by_os_version
        self.source_events = source_events
        self.conditions_names = conditions_names
        self.extra_internal_select_clauses = extra_internal_select_clauses
        self.extra_external_select_clauses = extra_external_select_clauses
        self.extra_group_fields = extra_group_fields
        self.sample_ratio = sample_ratio

    def build(self):
        date_start = self.date_end - timedelta(days=self.days)
        app_id_select_clause = 'AppID1 as app_id,' if self.group_by_app else ''
        app_id_group_clause = ', AppID1' if self.group_by_app else ''
        am_ver_select_clause = 'am_version,' if self.group_by_am_version else ''
        am_ver_group_clause = ', am_version' if self.group_by_am_version else ''
        os_ver_select_clause = 'OSVersion1 as os_version,' if self.group_by_os_version else ''
        os_ver_group_clause = ', OSVersion1' if self.group_by_os_version else ''

        sum_clauses = []
        count_clauses = []
        for condition, name in self.conditions_names:
            sum_clauses.append('toUInt64(sum(toUInt64(has_%s)) / %s) as %s' % (name, self.sample_ratio, name))
            count_clauses.append('countIf(%s) > 0 as has_%s' % (condition, name))

        return ur"""
        SELECT
            '%(target_date)s' as fielddate,
            AppPlatform1 as app_platform,
            %(os_ver_select_clause)s
            %(am_ver_select_clause)s
            %(app_id_select_clause)s
            %(sum_clauses)s
            %(extra_external_select_clauses)s
        FROM (
            SELECT
                any(AppPlatform) as AppPlatform1,
                any(OSVersion) as OSVersion1,
                any(arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'am_version'))) as am_version,
                AppID as AppID1,
                %(count_clauses)s
                %(extra_internal_select_clauses)s
            FROM %(table)s
            %(sampling_clause)s
            WHERE (StartDate >= toDate('%(date_start)s'))
                AND (StartDate <= toDate('%(target_date)s'))
                AND (EventDate >= toDate('%(date_start)s'))
                AND (EventDate <= toDate('%(target_date)s'))
                AND APIKey IN (%(api_keys)s)
                AND EventType=4
                AND has(ParsedParams.Key1, 'am_version')
                AND %(source_events_filter)s
                AND AppID not in (%(blacklisted_app_ids)s)
                AND OSVersion != '' AND AppPlatform != ''
                %(extra_filtering_clauses)s
            GROUP BY DeviceIDHash, AppID
        )
        GROUP BY AppPlatform1 %(os_ver_group_clause)s %(am_ver_group_clause)s %(app_id_group_clause)s %(extra_group_fields)s
            FORMAT JSONCompact
        """ % dict(
            table=settings.EVENTS_TABLE,
            am_ver_select_clause=am_ver_select_clause,
            am_ver_group_clause=am_ver_group_clause,
            os_ver_select_clause=os_ver_select_clause,
            os_ver_group_clause=os_ver_group_clause,
            app_id_select_clause=app_id_select_clause,
            app_id_group_clause=app_id_group_clause,
            date_start=date_start,
            target_date=self.date_end - timedelta(days=1),
            api_keys=', '.join(map(str, API_KEYS)),
            count_clauses=', '.join(count_clauses),
            sum_clauses=', '.join(sum_clauses),
            source_events_filter=make_source_events_filter(self.source_events),
            extra_filtering_clauses=self.extra_filtering_clauses or '',
            blacklisted_app_ids=(
                ','.join(map(lambda app_id: '\'%s\'' % app_id, settings.BLACKLISTED_APP_IDS))
            ) if settings.BLACKLISTED_APP_IDS else '',
            extra_internal_select_clauses=', '.join([''] + self.extra_internal_select_clauses) if len(
                self.extra_internal_select_clauses) > 0 else '',
            extra_external_select_clauses=', '.join([''] + self.extra_external_select_clauses) if len(
                self.extra_external_select_clauses) > 0 else '',
            extra_group_fields=', '.join([''] + self.extra_group_fields) if len(
                self.extra_group_fields) > 0 else '',
            sampling_clause='' if self.sample_ratio == 1 else 'SAMPLE {}'.format(self.sample_ratio),
        )


class AuthTypesQuery(Query):
    def build(self):
        return ur"""
            SELECT
                StartDate as fielddate,
                AppPlatform as app_platform,
                OSVersion as os_version,
                AppID as app_id,
                arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'am_version')) as am_version,
                arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'subtype')) as subtype,
                countIf(EventName='auth.auth_try') as attempts_count,
                countIf(EventName='auth.auth_success') as success_count,
                if(attempts_count>0, success_count/attempts_count, 0) as conversion
            FROM %(table)s
            WHERE (StartDate >= toDate('%(date_start)s'))
                AND (StartDate < toDate('%(date_end)s'))
                AND (EventDate >= toDate('%(date_start)s'))
                AND (EventDate <= toDate('%(date_end)s'))
                AND APIKey IN (%(api_keys)s)
                AND EventType=4
                AND has(ParsedParams.Key1, 'am_version')
                AND has(ParsedParams.Key1, 'subtype')
                AND EventName IN ('auth.auth_try', 'auth.auth_success')
                AND AppID not in (%(blacklisted_app_ids)s)
                AND OSVersion != '' AND AppPlatform != ''
                %(extra_filtering_clauses)s
            GROUP BY StartDate, AppPlatform, OSVersion, am_version, AppID, subtype
            HAVING attempts_count>0
            FORMAT JSONCompact
        """ % dict(
            table=settings.EVENTS_TABLE,
            date_start=self.date_start,
            date_end=self.date_end,
            api_keys=', '.join(map(str, API_KEYS)),
            extra_filtering_clauses=self.extra_filtering_clauses or '',
            blacklisted_app_ids=(
                ','.join(map(lambda app_id: '\'%s\'' % app_id, settings.BLACKLISTED_APP_IDS))
            ) if settings.BLACKLISTED_APP_IDS else '',
        )


class NoPhonishAuthTypesQuery(AuthTypesQuery, NoPhonishAppsFilteringMixin):
    pass


class RegTypesQuery(Query):
    def build(self):
        return ur"""
            SELECT
              fielddate,
              app_platform,
              os_version,
              am_version,
              app_id,
              values.2 as subtype,
              values.1 as success_count,
              attempts_count,
              success_count/attempts_count as conversion
            FROM (
                SELECT
                    StartDate as fielddate,
                    AppPlatform as app_platform,
                    OSVersion as os_version,
                    am_version,
                    AppID as app_id,
                    sum(phone_success_count) as phone_success_count,
                    sum(question_success_count) as question_success_count,
                    sum(success_count) as success_count,
                    sum(attempts_count) as attempts_count,
                    arrayJoin([
                        (phone_success_count, 'phone'),
                        (question_success_count, 'question'),
                        (success_count, 'total')
                    ] as measures) as values
                FROM (
                    SELECT
                        StartDate,
                        AppPlatform,
                        OSVersion,
                        AppID,
                        arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'am_version')) as am_version,
                        if(EventName='reg.success', arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'type')), '') as subtype,
                        if(EventName='reg.success', 1, 0) as success_count,
                        if(subtype='phone', 1, 0) as phone_success_count,
                        if(subtype='question', 1, 0) as question_success_count,
                        if(EventName='reg.launch', 1, 0) as attempts_count
                    FROM %(table)s
                    WHERE (StartDate >= toDate('%(date_start)s'))
                        AND (StartDate < toDate('%(date_end)s'))
                        AND (EventDate >= toDate('%(date_start)s'))
                        AND (EventDate <= toDate('%(date_end)s'))
                        AND APIKey IN (%(api_keys)s)
                        AND EventType=4
                        AND has(ParsedParams.Key1, 'am_version')
                        AND EventName IN ('reg.launch', 'reg.success')
                        AND AppID not in (%(blacklisted_app_ids)s)
                        AND OSVersion != '' AND AppPlatform != ''
                        %(extra_filtering_clauses)s
                )
                GROUP BY StartDate, AppPlatform, OSVersion, am_version, AppID
                HAVING attempts_count>0
            )
            FORMAT JSONCompact
        """ % dict(
            table=settings.EVENTS_TABLE,
            date_start=self.date_start,
            date_end=self.date_end,
            api_keys=', '.join(map(str, API_KEYS)),
            extra_filtering_clauses=self.extra_filtering_clauses or '',
            blacklisted_app_ids=(
                ','.join(map(lambda app_id: '\'%s\'' % app_id, settings.BLACKLISTED_APP_IDS))
            ) if settings.BLACKLISTED_APP_IDS else '',
        )


class NoPhonishRegTypesQuery(RegTypesQuery, NoPhonishAppsFilteringMixin):
    pass


LOGINSDK_SOURCE_EVENTS = [
    'loginsdk.show_scope',
    'loginsdk.show_scopes',
    'loginsdk.accept',
    'loginsdk.decline',
    'loginsdk.account_added',
    'AM_loginsdk.show_scope',
    'AM_loginsdk.accept',
    'AM_loginsdk.decline',
    'AM_loginsdk.account_added',
]


class LoginSDKFrequentReportersQuery(Query):
    def build(self):
        log.debug('LoginSDKFrequentReportersQuery %s %s' % (self.date_start, self.date_end))
        return ur"""
            SELECT
                groupUniqArray(reporter) as reporters
            FROM (
                SELECT
                    reporter,
                    count(*) * 2 as count
                FROM (
                    SELECT
                        any(
                          has(ParsedParams.Key1, 'reporter')
                          ? arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'reporter'))
                          : arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'requester'))
                        ) as reporter
                    FROM
                        %(table)s
                    SAMPLE 1/2
                    WHERE
                        (StartDate >= toDate('%(date_start)s')) AND (StartDate < toDate('%(date_end)s'))
                        AND (EventDate >= toDate('%(date_start)s')) AND (EventDate <= toDate('%(date_end)s'))
                        AND EventType=4
                        AND has(ParsedParams.Key1, 'am_version')
                        AND EventName IN (%(loginsdk_events)s)
                        AND AppID not in (%(blacklisted_app_ids)s)
                        AND OSVersion != '' AND AppPlatform != ''
                    GROUP BY DeviceIDHash, AppID
                )
                GROUP BY
                    reporter
                order by
                    reporter
            )
            WHERE
                count >= 10
            FORMAT JSONCompact
        """ % dict(
            table=settings.EVENTS_TABLE,
            date_start=self.date_start,
            date_end=self.date_end,
            loginsdk_events=', '.join(map(lambda x: '\'%s\'' % x, LOGINSDK_SOURCE_EVENTS)),
            blacklisted_app_ids=(
                ','.join(map(lambda app_id: '\'%s\'' % app_id, settings.BLACKLISTED_APP_IDS))
            ) if settings.BLACKLISTED_APP_IDS else '',

        )


class LoginSDKUniqueDeviceEventCountQuery(UniqueDeviceEventCountQuery):

    def __init__(self, date_start, *args, **kwargs):
        CLIENT_ID_MAPPING = get_client_id_mapping(date_start)
        client_ids = list(CLIENT_ID_MAPPING)
        client_id_names = [CLIENT_ID_MAPPING[clid] for clid in client_ids]
        client_id_transformer = u'[%s][indexOf([%s], reporter)]' % (
            u', '.join(map(lambda x: u'\'%s\'' % x.decode('utf-8'), client_id_names)),
            u', '.join(map(lambda x: u'\'%s\'' % x, client_ids)),
        )
        super(LoginSDKUniqueDeviceEventCountQuery, self).__init__(
            conditions_names=[
                ('EventName=\'AM_loginsdk.show_scope\' OR EventName=\'loginsdk.show_scopes\' OR EventName=\'loginsdk.show_scope\'', 'loginsdk_show_scopes'),
                ('EventName=\'AM_loginsdk.accept\' OR EventName=\'loginsdk.accept\'', 'loginsdk_accept'),
                ('EventName=\'AM_loginsdk.decline\' OR EventName=\'loginsdk.decline\'', 'loginsdk_decline'),
                ('EventName=\'AM_loginsdk.account_added\' OR EventName=\'loginsdk.account_added\'',
                 'loginsdk_account_added'),
            ],
            source_events=LOGINSDK_SOURCE_EVENTS,
            extra_internal_select_clauses=[
                'any(has(ParsedParams.Key1, \'reporter\')'
                '?  arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, \'reporter\'))'
                ': arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, \'requester\'))) as reporter',
            ],
            extra_group_fields=['reporter'],
            extra_external_select_clauses=[
                '%s as reporter' % client_id_transformer,
            ],
            group_by_app=True,
            date_start=date_start,
            *args, **kwargs
        )
        self.extra_filtering_clauses = (self.extra_filtering_clauses or '') + ur"""
            AND (has(ParsedParams.Key1, 'reporter')
                 ? arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'reporter'))
                 : arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'requester'))
            ) in (%(frequent_reporters)s)
        """ % dict(
            frequent_reporters=', '.join(map(lambda x: '\'%s\'' % x, get_login_sdk_frequent_reporters(date_start))),
        )


class AutologinFromSmartlockQuery(Query):
    def build(self):
        return ur"""
            SELECT
                StartDate as fielddate,
                count(*) as count,
                AppID as app_id
            FROM %(table)s
            WHERE (StartDate >= toDate('%(date_start)s'))
                AND (StartDate < toDate('%(date_end)s'))
                AND (EventDate >= toDate('%(date_start)s'))
                AND (EventDate <= toDate('%(date_end)s'))
                AND APIKey IN (%(api_keys)s)
                AND EventType=4
                AND has(ParsedParams.Key1, 'am_version')
                AND AppID not in (%(blacklisted_app_ids)s)
                AND AppPlatform = 'android'
                AND arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'from')) = 'autologin'
                AND (EventName = 'core.get_xtoken')
            GROUP BY StartDate, AppID
            FORMAT JSONCompact
        """ % dict(
            table=settings.EVENTS_TABLE,
            date_start=self.date_start,
            date_end=self.date_end,
            api_keys=', '.join(map(str, API_KEYS)),
            blacklisted_app_ids=(
                ','.join(map(lambda app_id: '\'%s\'' % app_id, settings.BLACKLISTED_APP_IDS))
            ) if settings.BLACKLISTED_APP_IDS else '',
        )


def _make_new_flow_source_extractor():
    res = 'arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, \'source\'))'
    res = 'if(has(ParsedParams.Key1, \'source\'), %s, \'null\')' % res
    res = 'replaceRegexpOne(%s, \'^web_to_am\\.socialserp.*\', \'web_to_am.socialserp\')' % res
    res = 'if(%s = \'\', \'null\', %s)' % (res, res)
    return res


NEW_FLOW_SOURCE_EXTRACTOR = _make_new_flow_source_extractor()


class BroadcastQuery(GenericUniqueDeviceQuery):
    def __init__(self, *args, **kwargs):
        super(BroadcastQuery, self).__init__(*args, **kwargs)
        self.source_events = {'core.announcement_sent', 'core.announcement_received'}
        self.internal_select_clauses.update([
            'countIf(EventName = \'core.announcement_sent\') as sent',
            'countIf(EventName = \'core.announcement_received\') as received',
        ])
        self.external_select_clauses.update([
            'sum(sent) as sent_events',
            'sum(received) as received_events',
            'countIf(sent > 0) as sent_devices',
            'countIf(received > 0) as received_devices',
        ])


SSO_EVENTS = [
    'sso.content_provider_client_error',
    'sso.insert_accounts_in_backup',
    'sso.insert_accounts_in_bootstrap',
    'sso.insert_accounts_failed',
    'sso.insert_accounts_finish',
    'sso.insert_accounts_start',
    'sso.is_trusted_error',
    'sso.fetch_accounts',
    'sso.give_accounts',
    'sso.receive_accounts',
    'sso.send_broadcast_in_backup',
    'sso.send_broadcast_in_bootstrap',
    'sso.sync_accounts',
]


class SSOEventsQuery(Query):
    def build(self):
        return ur"""
            SELECT
                '%(target_date)s' as fielddate,
                AppID as app_id,
                arrayElement(ParsedParams.Key2, indexOf(ParsedParams.Key1, 'am_version')) as am_version,
                EventName as event_name,
                (StartTimestamp + EventTimeOffset) - (StartTimestamp + EventTimeOffset) %% 600 as timestamp,
                count(*) * 2 as count
            FROM %(table)s
            SAMPLE 1/2
            WHERE (StartDate == toDate('%(target_date)s'))
                AND (EventDate >= toDate('%(target_date)s'))
                AND (EventDate <= toDate('%(target_date)s'))
                AND %(event_filter)s
                AND AppID not in (%(blacklisted_app_ids)s)
                AND has(ParsedParams.Key1, 'am_version')
                AND OSVersion != '' AND AppPlatform != ''
            GROUP BY am_version, app_id, timestamp, event_name
            FORMAT JSONCompact
        """ % dict(
            table=settings.EVENTS_TABLE,
            target_date=self.date_end - timedelta(days=1),
            end_date=self.date_end,
            api_keys=', '.join(map(str, API_KEYS)),
            event_filter=make_source_events_filter(SSO_EVENTS),
            blacklisted_app_ids=(
                ','.join(map(lambda app_id: '\'%s\'' % app_id, settings.BLACKLISTED_APP_IDS))
            ) if settings.BLACKLISTED_APP_IDS else '',
        )
