# -*- coding: utf-8 -*-
import logging
import sys
import warnings

from passport.backend.ch_stat_loader.ch_stat_loader.query import (
    ActionDurationQuery,
    DailyEventCountQuery,
    HourlyEventCountQuery,
)
from passport.backend.ch_stat_loader.ch_stat_loader.utils import (
    escape_name,
    retriable_n,
)
from statface_client import StatfaceHttpResponseError


log = logging.getLogger('ch_stat_loader.report')


DIM_TITLES = {
    'fielddate': u'Дата',
    'app_platform': u'Платформа',
    'os_version': u'Версия ОС',
    'group': u'Группа пользователей',
    'app_id': u'ID приложения',
    'hour': u'Час',
    'am_version': u'Версия AM',
}


DIM_TYPES = {
    'fielddate': 'date',
    'app_platform': 'string',
    'os_version': 'string',
    'group': 'string',
    'app_id': 'string',
    'hour': 'number',
    'am_version': 'string',
    'max_am_ver': 'string',
}


TYPE_TITLES = {
    'generic_daily': u'число событий по дням',
    'generic_hourly': u'среднее число событий по часам',
    'generic_duration': u'время между событиями',
}


TYPE_DIMENSIONS = {
    'generic_daily': [
        'fielddate',
        'app_platform',
        'os_version',
        'group',
        'am_version',
        'app_id',
    ],
    'generic_hourly': [
        'fielddate',
        'app_platform',
        'os_version',
        'group',
        'am_version',
        'app_id',
        'hour',
    ],
    'generic_duration': [
        'fielddate',
        'app_platform',
        'os_version',
        'group',
        'am_version',
        'app_id',
    ],
}


def get_dimensions_by_type(type_):
    return TYPE_DIMENSIONS.get(type_, [])


@retriable_n(retry_count=10, time_sleep=10, args_for_log_transformer=lambda args: args[:3] + ('<cut>',))
def publish_report_to_statface(client, path, config, values):
    report = client.get_report(path)
    report.upload_config(config)
    report.upload_data(scale='daily', data=values, replace_mask=['fielddate'])


def is_report_published(client, path, target_date):
    try:
        logging.disable(logging.DEBUG)
        with warnings.catch_warnings():
            warnings.simplefilter("ignore")
            report = client.get_report(path)
            missing_dates = report.fetch_missing_dates(
                scale='daily',
                date_min=target_date,
                date_max=target_date,
            )
        if len(missing_dates['missing']) == 0:
            return True
    except StatfaceHttpResponseError as e:
        # такая ошибка означает, что отчет не существует
        # https://wiki.yandex-team.ru/statbox/statface/externalreports/statface-python-client/#vozmozhnyeoshibki
        if ('Unable to init xreport with name %s' % path) not in str(e):
            raise
    finally:
        logging.disable(logging.NOTSET)
    return False


class Report(object):
    def __init__(self, path, type_, config, date_start, date_end, group_title, custom_types, statface_client):
        self.path = path
        self.type_ = type_
        self.config = config
        self.dimensions = (config.get('dimensions') or get_dimensions_by_type(type_)) + (config.get('extra_dimensions') or [])
        self.measures = config['measures']
        self.base_title = config.get('description', group_title)
        self.subtitle = config.get('title')
        self.titles = config.get('titles', {})
        self.custom_types = custom_types
        self.date_start = date_start
        self.date_end = date_end
        self.data = {}
        self.priority = config.get('priority', 1)
        self.number_of_queries = 0
        self.received_data_from_queries = 0
        self.statface_client = statface_client

    def get_queries(self):
        if self.type_ == 'custom':
            query = self.config['query']
            for q in (query if isinstance(query, list) else [query]):
                self.number_of_queries += 1
                yield q(
                    date_start=self.date_start,
                    date_end=self.date_end,
                    report=self,
                )
        for measure in self.measures:
            if self.type_ == 'generic_duration':
                if 'query' in measure:
                    query = measure['query'](
                        date_start=self.date_start,
                        date_end=self.date_end,
                        report=self,
                    )
                    self.number_of_queries += 1
                    yield query
                else:
                    query = self.config.get('queries', {}).get(self.type_) or ActionDurationQuery
                    self.number_of_queries += 1
                    yield query(
                        date_start=self.date_start,
                        date_end=self.date_end,
                        points=[measure['point']],
                        report=self,
                    )
            if self.type_ == 'generic_daily':
                query = self.config.get('queries', {}).get(self.type_) or DailyEventCountQuery
                self.number_of_queries += 1
                yield query(
                    date_start=self.date_start,
                    date_end=self.date_end,
                    events=[measure],
                    report=self,
                )
            if self.type_ == 'generic_hourly':
                query = self.config.get('queries', {}).get(self.type_) or HourlyEventCountQuery
                self.number_of_queries += 1
                yield query(
                    date_start=self.date_start,
                    date_end=self.date_end,
                    events=[measure],
                    report=self,
                )

    def set_data(self, data):
        assert self.received_data_from_queries < self.number_of_queries
        for columns in data:
            key = tuple(columns[dim] for dim in self.dimensions)
            values = self.data.setdefault(key, {})
            for new_key, new_value in columns.iteritems():
                if new_key in self.dimensions:
                    values[new_key] = new_value
                    continue
                if new_key in values:
                    raise ValueError('Value of %s already present for dimensions key %s (old %s new %s) in report %s' % (
                        new_key,
                        key,
                        values,
                        columns,
                        self.path,
                    ))
                values[new_key] = new_value
        size = sys.getsizeof(self.data)
        for values in self.data.values():
            size += sys.getsizeof(values)
            size += sum(map(sys.getsizeof, values.keys()))
            size += sum(map(sys.getsizeof, values.values()))
        log.debug('REPORT {} SIZE OF DATA: {} bytes'.format(self.path, size))
        self.received_data_from_queries += 1
        if self.received_data_from_queries == self.number_of_queries:
            self.publish()

    def describe_measures(self):
        for measure in self.measures:
            if self.type_ == 'generic_duration':
                if 'point' in measure:
                    query = ActionDurationQuery
                else:
                    query = measure['query']
                point_key = query.get_point_key(measure.get('point'))
                yield 'avg_' + point_key, measure['description'], 'number'
                yield 'count_' + point_key, u'Количество событий (%s)' % measure['description'].lower(), 'number'

            else:
                title = self.titles.get(measure, self.titles.get(escape_name(measure)))
                if not title:
                    raise ValueError('Failed to find title for %s' % measure)
                custom_types = self.custom_types or {}
                yield escape_name(measure), title, custom_types.get(measure, 'number')

    def generate_statface_config(self):
        titles = {}
        measures = []
        for dim in self.dimensions:
            titles[dim] = self.titles.get(dim) or DIM_TITLES[dim]
        for name, description, type_ in self.describe_measures():
            titles[name] = description
            measures.append({name: type_})
        if self.type_ in TYPE_TITLES:
            title = '%s (%s)' % (self.base_title, TYPE_TITLES[self.type_])
        elif self.subtitle:
            title = '%s (%s)' % (self.base_title, self.subtitle)
        else:
            title = self.base_title
        config = {
            'dimensions': [
                {dim: dict(DIM_TYPES, **(self.custom_types or {})).get(dim, 'string')} for dim in self.dimensions
            ],
            'measures': measures,
            'title': title,
            'titles': titles,
        }
        return config

    def publish(self):
        if self.data:
            config = self.generate_statface_config()
            log.debug('Publish %s (%s)', self.path, self.date_start)
            publish_report_to_statface(self.statface_client, self.path, config, self.data.values())
            self.data = {}
        else:
            log.warning('No data for path %s (%s)', self.path, self.date_start)
