# -*- coding: utf-8 -*-
import datetime
from functools import wraps
import logging
import time
import traceback


log = logging.getLogger('ch_stat_loader.utils')


DATE_FORMAT = '%Y-%m-%d'


def from_str_to_date(date):
    return datetime.datetime.strptime(date, DATE_FORMAT)


NAME_TO_CH_NAME = {}
CH_NAME_TO_NAME = {}


def escape_name(name):
    if name in NAME_TO_CH_NAME:
        return NAME_TO_CH_NAME[name]
    output = name.replace('.', '_')
    if output in CH_NAME_TO_NAME:
        log.warning('Conflict for %s and %s' % (CH_NAME_TO_NAME[output], name))
    CH_NAME_TO_NAME[output] = name
    NAME_TO_CH_NAME[name] = output
    return output


def retriable_n(retry_count=3, time_sleep=0.2, exceptions=(Exception,), args_for_log_transformer=None):
    def retriable_n_deco(func):
        @wraps(func)
        def wrapper(*args, **kw):
            for i in range(retry_count - 1):
                start = time.time()
                try:
                    return func(*args, **kw)
                except Exception as e:
                    if isinstance(e, exceptions):
                        args_for_log = args_for_log_transformer(args) if args_for_log_transformer else args
                        log.warning('%s(*%s, **%s) try %i failed, retrying: %s' % (func.__name__, args_for_log, kw, i, e))
                        log.info('%s(*%s, **%s) try %i failed, execution_time=%s, retrying' %
                                 (func.__name__, args_for_log, kw, i, start - time.time()))
                        log.debug(traceback.format_exc())
                        print('%s(*%s, **%s) try %i failed, retrying: %s' % (func.__name__, args_for_log, kw, i, e))  # noqa
                        print(  # noqa
                            '%s(*%s, **%s) try %i failed, execution_time=%s, retrying' % (
                                func.__name__, args_for_log, kw, i, start - time.time()
                            )
                        )
                        print(traceback.format_exc())  # noqa
                        time.sleep(time_sleep)
                    else:
                        raise
            else:
                return func(*args, **kw)
        return wrapper
    return retriable_n_deco


def cached(func):
    cache = {}

    def wrapper(*args):
        if args not in cache:
            cache[args] = func(*args)
        return cache[args]
    return wrapper
