from functools import wraps
import logging
import multiprocessing
import time

from passport.backend.ch_stat_loader.ch_stat_loader.settings import get_yt
from yt.wrapper.errors import YtHttpResponseError


log = logging.getLogger('passport_profile.utils.yt')
yt = get_yt()


class YtLockException(Exception):
    pass


class YtLock(object):
    def __init__(self, lock_path, yt, log=None, transaction_id_to_ping=None):
        self.lock_path = lock_path
        self.yt = yt
        self.transaction_id_to_ping = transaction_id_to_ping
        self.pinger = None
        self.log = log
        if not yt.exists(self.lock_path):
            self.yt.create('document', self.lock_path, recursive=True)

    def __enter__(self):
        try:
            lock_id = self.yt.lock(self.lock_path)
            if lock_id is None:
                raise YtLockException()
            if self.transaction_id_to_ping is not None:
                def pinger(delay=60):
                    local_thread_yt = get_yt()
                    while True:
                        time.sleep(delay)
                        local_thread_yt.ping_transaction(self.transaction_id_to_ping)
                self.pinger = multiprocessing.Process(target=pinger)
                self.pinger.start()
        except Exception as e:
            if self.pinger is not None:
                self.pinger.terminate()
            if self.log is not None:
                self.log.error(str(e))
            raise YtLockException()

    def __exit__(self, exc_type, exc_value, traceback):
        if self.pinger is not None:
            self.pinger.terminate()
        if isinstance(exc_type, YtLockException) and self.log is not None:
            self.log.info("%s can't be acquired" % self.lock_path)
        if self.yt.exists(self.lock_path):
            self.yt.remove(self.lock_path)


def run_exclusively(lock_path):
    def run_exclusively_wrapper(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            with yt.Transaction():
                try:
                    with YtLock(lock_path=lock_path, yt=yt, log=log):
                        func(*args, **kwargs)
                except YtHttpResponseError as e:
                    # code from: https://github.yandex-team.ru/yt/yt/blob/archive-stable-seneca/19.2/yt/ytlib/cypress_client/public.h#L50
                    if e.error['code'] != 402:
                        raise
        return wrapper
    return run_exclusively_wrapper
