import os

from passport.backend.ci.common import run_with_prompt
from passport.backend.library.configurator import Configurator
from passport.backend.library.packaging import Packaging
from passport.backend.library.packaging.exceptions import CantFindProjectDirectoryError
from passport.backend.library.repo import (
    get_repository_info,
    get_repository_manager,
)
from passport.backend.utils.time import unixtime_to_datetime
from six.moves import input


def configure_changelog_args(commander):
    commander.add_argument(
        '-cf', '--changelog',
        help='Path to changelog file',
        dest='changelog',
    ).add_argument(
        '-v', '--version',
        help='Point version for the new changelog entry',
        dest='release_version',
    ).add_argument(
        '-f', '--force-changelog',
        action='store_true',
        default=False,
        help='Create new changelog entry even without changes',
        dest='force_changelog',
    ).add_argument(
        '-on', '--omit-names',
        action='store_true',
        default=False,
        help='do not write usernames in changelog',
        dest='omit_names',
    ).add_argument(
        '-ol', '--omit-lines',
        nargs='+',
        help='Regular expression for omitting lines in changelog',
        dest='omit_lines',
        default=[],
    ).add_argument(
        '-lr', '--last-release',
        help='Explicitly point last release commit hash to grab all changes since it',
        dest='last_release_commit_hash',
        default=None,
    ).add_argument(
        '-b', '--bump',
        help='Point which part of semantic version to bump (<major>.<minor>.<patch>-pre.<prerelease>+build.<build>)',
        choices=['major', 'minor', 'patch', 'prerelease', 'build'],
        dest='bump',
        default=None,
    ).add_argument(
        '-y', '--yes',
        help='Answer "yes" for all interactive questions',
        action='store_true',
        dest='always_yes',
        default=False,
    )


def configure_changelog_command(commander):
    subcommander = commander.add_command(
        'changelog',
        run_with_prompt(run_changelog_command),
        expand_kwargs=False,
    )
    configure_changelog_args(subcommander)


def run_changelog_command(parsed_args, path='.', config=None):
    if config is None:
        config = Configurator('passport-ci', configs=['~/.ci.yaml?', os.path.join(path, '.ci.yaml?')])

    repository_manager = get_repository_manager()
    try:
        packaging = Packaging(base_path=path, deb_path=config.get('deb_path'))
    except CantFindProjectDirectoryError:
        raise RuntimeError(
            "Could not find package files. "
            "Run `ci {cmd}` from project folder or from `deb` folder. "
            "Another way is to configure ./.ci.yaml",
        )

    changelog, package, version = packaging.ensure()

    print('')
    print('Changelog file: %s' % packaging.changelog_path)
    print('Package directory: %s' % packaging.base_path)

    if repository_manager.is_touched(packaging.changelog_path):
        if parsed_args.always_yes or input('Warning! Changelog has uncommited changes. Use them? (y/n) ') == 'y':
            return changelog, package, version

        repository_manager.revert(packaging.changelog_path)
        changelog, package, version = packaging.ensure()

    search_result = repository_manager.get_commits_since(
        directory=packaging.base_path,
        commit_hash=parsed_args.last_release_commit_hash,
        message=['release', '%s %s' % (package, version)],
    )

    if config.get('changelog', {}).get('extra_directories'):
        limit_unixtime = unixtime_to_datetime(
            search_result.found_commit.unixtime,
        ) if search_result.found_commit else None
        extra_directories = config['changelog']['extra_directories']
        if extra_directories:
            print('Extra directories:')
            for directory in extra_directories:
                directory_path = os.path.join(get_repository_info(packaging.base_path)['path'], directory)
                print('- %s' % directory_path)
                extra_search_result = repository_manager.get_commits_since(
                    directory=directory_path,
                    target_date=limit_unixtime,
                )
                search_result = search_result.extend(extra_search_result)

    if not search_result.commits_since and not parsed_args.force_changelog:
        print('No changes found since release %s' % version)
        return changelog, None, None

    return packaging.increase_version(
        search_result.commits_since,
        new_version=parsed_args.release_version,
        omit_names=parsed_args.omit_names or config.get('changelog', {}).get('omit_names', False),
        omit_lines=parsed_args.omit_lines or config.get('changelog', {}).get('omit_lines', False),
        version_increase=parsed_args.bump or config.get('changelog', {}).get('default_version_increase'),
        always_yes=parsed_args.always_yes,
    )
