import os

from passport.backend.ci.common import run_with_prompt
from passport.backend.clients.conductor import ConductorClient
from passport.backend.library.configurator import Configurator
from passport.backend.library.packaging import Packaging


def configure_conductor_command(commander):
    commander.add_command(
        'conductor',
        run_with_prompt(run_conductor_command),
    )


def configure_conductor_args(commander):
    pass


def get_config(path='.'):
    try:
        config = Configurator('passport-ci', configs=['~/.ci.yaml', os.path.join(path, '.ci.yaml?')])
        if not config.has('conductor', 'token'):
            raise ValueError('No Conductor token')
        return config
    except (IOError, ValueError):
        raise RuntimeError('Run `ci config` to acquire Conductor token')


def run_conductor_command(path='.', package_names_with_versions=None, comment=None, release_issue=None, config=None):
    if config is None:
        config = get_config(path)

    if not package_names_with_versions:
        packaging = Packaging(base_path=path, deb_path=config.get('deb_path'))
        changelog, package_name, version = packaging.ensure()
        package_names_with_versions = [(package_name, version)]

    if not comment:
        packaging = Packaging(base_path=path, deb_path=config.get('deb_path'))
        comment = packaging.get_last_changelog()

    if release_issue:
        comment = (
            config['conductor'].get(
                'release_issue_template', 'Release issue: https://st.yandex-team.ru/{release_issue}'
            ).format(release_issue=release_issue)
        ) + '\n\n' + comment

    conductor_client = ConductorClient(token=config['conductor']['token'])
    ticket_url = conductor_client.create_ticket(
        packages_with_versions=[
            {
                'package': package_name_,
                'version': version_,
            }
            for package_name_, version_ in package_names_with_versions
        ],
        comment=comment,
        branch=config['conductor'].get('branch', 'testing'),
    )
    print('\n-----===== Conductor Ticket =====-----')
    print(ticket_url)
