import getpass

from passport.backend.ci.common import run_with_prompt
from passport.backend.library.configurator import Configurator
from passport.backend.library.packaging import Packaging
from startrek_client import Startrek


def configure_issue_args(commander):
    commander.add_argument(
        '-q', '--queue',
        help='Queue name',
        dest='queue',
        default='',
    )
    commander.add_argument(
        '--issue-type',
        help='Issue type (Release, Task etc.)',
        dest='issue_type',
        default='',
    )


def configure_issue_command(commander):
    subcommander = commander.add_command(
        'issue',
        run_with_prompt(run_issue_command),
        expand_kwargs=False,
    )
    configure_issue_args(subcommander)


def get_config():
    try:
        config = Configurator('passport-ci', configs=['~/.ci.yaml', './.ci.yaml?'])
        if not config.has('startrek', 'token'):
            raise ValueError('No Startrek token')
        return config
    except (IOError, ValueError):
        raise RuntimeError('Run `ci config` to acquire Startrek token')


def get_startrek_client(config):
    config = config or get_config()
    return Startrek(
        useragent='Passport CI Tool',
        base_url='https://st-api.yandex-team.ru',
        token=config['startrek']['token'],
    )


def run_issue_command(parsed_args, path='.', summary=None, description=None, config=None):
    if config is None:
        config = get_config()

    if not summary:
        packaging = Packaging(base_path=path, deb_path=config.get('deb_path'))
        _, package_name, version = packaging.ensure()
        summary = 'Release {package_name} {version}'.format(package_name=package_name, version=version)

    if not description:
        packaging = Packaging(base_path=path, deb_path=config.get('deb_path'))
        description = '\n'.join(packaging.get_last_block_commits())

    issue_config = config.get('issue', {})
    startrek_client = get_startrek_client(config)
    response = startrek_client.issues.create(
        assignee=getpass.getuser(),
        queue=parsed_args.queue or issue_config.get('queue', 'PASSP'),
        summary=summary,
        type={'name': parsed_args.issue_type or issue_config.get('type', 'Release')},
        description=description,
        components=issue_config.get('components', []),
    )
    print('\n-----===== StarTrek Issue =====-----')
    print('https://st.yandex-team.ru/%s' % response.key)
    return response.key


def post_comment(issue, comment, config=None):
    startrek_client = get_startrek_client(config)
    startrek_client.issues[issue].comments.create(text=comment)


def get_issue_info(issue, config=None):
    startrek_client = get_startrek_client(config)
    issue_info = startrek_client.issues[issue]
    return {
        'status': issue_info.status.key,
        'summary': issue_info.summary,
        'description': issue_info.description,
    }


def update_issue(issue, summary, description, config=None):
    startrek_client = get_startrek_client(config)
    startrek_client.issues[issue].update(summary=summary, description=description)
