import os
import re
from time import sleep
from uuid import uuid4

from passport.backend.ci.changelog import (
    configure_changelog_args,
    run_changelog_command,
)
from passport.backend.ci.common import run_with_prompt
from passport.backend.ci.conductor import (
    configure_conductor_args,
    run_conductor_command,
)
from passport.backend.ci.issue import (
    configure_issue_args,
    get_issue_info,
    post_comment,
    run_issue_command,
    update_issue,
)
from passport.backend.ci.sandbox import (
    configure_sandbox_args,
    run_sandbox_command,
)
from passport.backend.library.configurator import Configurator
from passport.backend.library.packaging import Packaging
from passport.backend.library.repo import get_repository_manager
from passport.backend.utils.system import (
    run_command,
    RunCommandException,
)
import semver


RELEASE_COMMIT_MESSAGE_RE = re.compile(r'(?P<ticket>[^ ]+ )?release (.*)')
SECTION_TITLE_RE = re.compile(r'\[(?P<package_name>.+)\].*')
REVIEW_URL_RE = re.compile(r'https://a.yandex-team.ru/review/([0-9]+)')
MAX_BRANCH_NAME_LENGTH = 120


def format_release_commit_message(package_names_with_versions, issue=None):
    m = 'release %s' % ', '.join([
        '%s %s' % (package_name, version)
        for package_name, version in sorted(package_names_with_versions)
    ])
    if issue:
        m = issue.upper() + ' ' + m
    return m


def format_branch_name(package_names_with_versions):
    raw_name = format_release_commit_message(package_names_with_versions)
    name = raw_name.lower().replace(' ', '_').replace('.', '-').replace(',', '')
    # adding some randomness, to assure that new branch is used
    random_part_len = 8
    random_part = uuid4().hex[:random_part_len]
    name = '%s_%s' % (name[:MAX_BRANCH_NAME_LENGTH - random_part_len - 1], random_part)
    return name


def build_changelog(package_name_to_changelog):
    if len(package_name_to_changelog) == 1:
        return list(package_name_to_changelog.values())[0]
    else:
        return '\n\n'.join(
            '[%s]\n%s' % (package_name, changelog_part.rstrip())
            for package_name, changelog_part in sorted(package_name_to_changelog.items())
        )


def parse_release_commit_message_to_package_names_with_versions(message):
    match = RELEASE_COMMIT_MESSAGE_RE.match(message)
    if not match:
        raise ValueError('Unable to parse packages and versions from %r' % message)
    packages_and_versions_raw = match.group(2)
    return [
        item.rsplit(' ', 1)
        for item in packages_and_versions_raw.split(', ')
    ]


def version_to_tuple(version):
    version = semver.parse(version)
    return (
        # Три номера версии переводит в инты метод semver.parse
        version['major'],
        version['minor'],
        version['patch'],
        version['prerelease'],
        version['build'],
    )


def merge_package_names_with_versions(first_list, second_list):
    first_dict, second_dict = dict(first_list), dict(second_list)
    return [
        [
            package_name,
            '.'.join(
                map(
                    str,
                    filter(
                        lambda x: x is not None,
                        max(
                            version_to_tuple(first_dict.get(package_name, '0.0.0')),
                            version_to_tuple(second_dict.get(package_name, '0.0.0')),
                        ),
                    ),
                ),
            ),
        ]
        for package_name in sorted(
            set(first_dict.keys()) | set(second_dict.keys()),
        )
    ]


def parse_changelog(changelog, package_names):
    """
    :param changelog: строка в формате:

    ```
    [package_name_1]

    * feature 1
    * feature 2

    [package_name_2]

    * feature 3
    * feature 4
    ```

    или:

    ```
    * feature 1
    * feature 2
    ```

    :return: dict (package_name -> changelog)
    """
    result = {}
    package_name, features = None, []

    if len(package_names) == 1:
        package_name = package_names[0]

    for line in changelog.split('\n'):
        line = line.rstrip()
        if not line:
            continue

        section_title_match = SECTION_TITLE_RE.match(line)
        if section_title_match:
            if package_name and features:
                result[package_name] = '\n'.join(features)
            package_name = section_title_match.group(1)
            features = []
        else:
            features.append(line)

    if package_name and features:
        result[package_name] = '\n'.join(features)

    return result


def merge_changelogs(first_changelog_str, first_package_names, second_changelog_str, second_package_names):
    parsed_first = parse_changelog(first_changelog_str, first_package_names)
    parsed_second = parse_changelog(second_changelog_str, second_package_names)
    for package_name, changes in parsed_second.items():
        if parsed_first.get(package_name):
            parsed_first[package_name] += '\n%s' % changes
        else:
            parsed_first[package_name] = changes
    return build_changelog(parsed_first)


def merge_pull_request(review_id):
    print('Merging PR %s...' % review_id)
    max_tries = 5
    for i in range(max_tries):
        try:
            run_command(
                'ya pr merge --id={review_id} --no-require=ship --no-require=build --no-require=test --auto'.format(
                    review_id=review_id,
                ),
            )
            run_command(
                'ya pr merge --id={review_id} --now'.format(
                    review_id=review_id,
                ),
            )
            print('Will be merged soon.')
            return
        except RunCommandException as e:
            if i < max_tries - 1:
                # Review may be unpublished yet. Let's wait a bit.
                sleep(2.0)
            else:
                # Failed to merge. Let's move on.
                print('FAILED to merge. %s\n\nYou can merge it manually.' % e)


def configure_release_command(commander):
    subcommander = commander.add_command(
        'release',
        run_with_prompt(run_release_command),
        expand_kwargs=False,
    )
    configure_changelog_args(subcommander)
    configure_conductor_args(commander)
    configure_issue_args(subcommander)
    configure_sandbox_args(subcommander)
    subcommander.add_argument(
        'paths',
        nargs='*',
        help='Path of package to be released',
        default=[],
        metavar='PATH',
    ).add_argument(
        '-oc', '--only-commit',
        action='store_true',
        help='Make release commit without creating changelog (also discards ST, conductor and sandbox subtasks)',
        default=False,
        dest='only_commit',
    ).add_argument(
        '-nsb', '--no-sandbox',
        action='store_false',
        help='Do not create sandbox task for building',
        dest='sandbox_required',
        default=True,
    ).add_argument(
        '-i', '--issue',
        help='Update existing ST release issue instead of creating new one',
        dest='issue',
    ).add_argument(
        '-ni', '--no-issue',
        action='store_false',
        help='Do not create issue in ST for the release',
        dest='issue_required',
        default=True,
    ).add_argument(
        '-nc', '--no-conductor',
        action='store_false',
        help='Do not create conductor ticket for the release',
        dest='conductor_required',
        default=True,
    ).add_argument(
        '-ef', '--extra-files',
        nargs='+',
        help='Extra files to add to commit',
        dest='extra_files',
        default=[],
    )


def run_release_command(parsed_args):
    repository_manager = get_repository_manager()
    package_names_with_versions = []
    package_name_to_path = {}
    package_name_to_config = {}
    package_name_to_changelog = {}
    files_to_commit = parsed_args.extra_files

    config = Configurator(
        'passport-ci',
        configs=['~/.ci.yaml?', './.ci.yaml?'],
    )
    paths = parsed_args.paths or config.get('release', {}).get('default_targets', ['.'])

    for path in paths:
        if os.path.isabs(path):
            raise ValueError('Invalid path `%s`: must be relative' % path)
        elif path != '.':
            path = os.path.join('.', path)  # Configurator won't be able to find config otherwise

        child_config = Configurator(
            'passport-ci',
            configs=['~/.ci.yaml?', os.path.join(path, '.ci.yaml?')],
        )
        packaging = Packaging(base_path=path, deb_path=child_config.get('deb_path'))

        if parsed_args.only_commit:
            _, package_name, version = packaging.ensure()
        else:
            _, package_name, version = run_changelog_command(parsed_args, path=path, config=child_config)
            if not version:
                print('Skipping %s: nothing changed' % path)
                continue

            recent_changelog = packaging.get_last_changelog()
            package_name_to_changelog[package_name] = recent_changelog

        package_name_to_path[package_name] = path
        package_name_to_config[package_name] = child_config
        package_names_with_versions.append((package_name, version))
        files_to_commit += (
            [packaging.changelog_path] +
            child_config.get('release', {}).get('extra_files', [])
        )

    if not files_to_commit:
        # no release targets or nothing changed
        return False

    full_changelog = build_changelog(package_name_to_changelog)

    if not parsed_args.only_commit and parsed_args.issue_required:
        if parsed_args.issue:
            # Parsing and updating existing release ticket
            issue_info = get_issue_info(parsed_args.issue, config=config)
            if issue_info['status'] == 'closed':
                print('Unable to edit %s: it is already closed.' % parsed_args.issue)
                return False

            old_ticket_summary = issue_info['summary']
            old_full_changelog = issue_info['description']
            old_package_names_with_versions = parse_release_commit_message_to_package_names_with_versions(old_ticket_summary)

            full_changelog = merge_changelogs(
                first_changelog_str=old_full_changelog,
                first_package_names=[
                    first_package_name
                    for first_package_name, first_version in old_package_names_with_versions
                ],
                second_changelog_str=full_changelog,
                second_package_names=[
                    second_package_name
                    for second_package_name, second_version in package_names_with_versions
                ],
            )
            package_names_with_versions = merge_package_names_with_versions(
                old_package_names_with_versions,
                package_names_with_versions,
            )
            ticket_summary = format_release_commit_message(
                package_names_with_versions,
                issue=None,
            )
            update_issue(issue=parsed_args.issue, summary=ticket_summary, description=full_changelog, config=config)
        else:
            # Creating new release ticket
            ticket_summary = format_release_commit_message(
                package_names_with_versions,
                issue=None,
            )
            parsed_args.issue = run_issue_command(
                parsed_args=parsed_args,
                path='.',
                summary=ticket_summary,
                description=full_changelog,
                config=config,
            )

    pull_request_url = repository_manager.pull_request(
        files=files_to_commit,
        message=format_release_commit_message(
            package_names_with_versions,
            issue=parsed_args.issue,
        ),
        branch_name=format_branch_name(package_names_with_versions),
    )
    review_id = None
    if pull_request_url:
        print('\n-----===== Pull Request =====-----')
        print(pull_request_url)
        review_id = REVIEW_URL_RE.match(pull_request_url).group(1)
        merge_pull_request(review_id)

    if parsed_args.only_commit:
        # changelog already committed, nothing left to do
        return True

    if parsed_args.conductor_required:
        run_conductor_command(
            path='.',
            package_names_with_versions=package_names_with_versions,
            comment=full_changelog,
            release_issue=parsed_args.issue,
            config=config,
        )

    if parsed_args.sandbox_required:
        sandbox_task_urls = []
        for package_name, version in package_names_with_versions:
            if package_name not in package_name_to_path:
                # package not changed, no need to rebuild it
                continue
            task = run_sandbox_command(
                parsed_args=parsed_args,
                path=package_name_to_path[package_name],
                package_name=package_name,
                version=version,
                review_id=review_id,
                config=package_name_to_config[package_name] if package_name in package_name_to_config else None,
            )
            sandbox_task_urls.append(task.url)

        if sandbox_task_urls and parsed_args.issue:
            post_comment(parsed_args.issue, '\n'.join(sandbox_task_urls), config=config)

    return True
