from __future__ import absolute_import

import os

from passport.backend.ci.common import run_with_prompt
from passport.backend.library.configurator import Configurator
from passport.backend.library.packaging import Packaging
from passport.backend.library.repo import get_relative_path
from passport.backend.utils.common import smart_merge_dicts
from sandbox.common import rest
from sandbox.common.proxy import OAuth


def configure_sandbox_args(commander):
    commander.add_argument(
        '-ns', '--do-not-start',
        action='store_true',
        help='Do not start Sandbox task',
        dest='do_not_start',
        default=False,
    )
    commander.add_argument(
        '-i', '--issue',
        help='Issue ticket',
        dest='issue',
        default=None,
    )
    commander.add_argument(
        '-r', '--review_id',
        help='Review ID to wait for merge',
        dest='review_id',
        default=None,
    )


def configure_sandbox_command(commander):
    subcommander = commander.add_command(
        'sandbox',
        run_with_prompt(run_sandbox_command),
        expand_kwargs=False,
    )
    configure_sandbox_args(subcommander)


class SandboxTask(object):
    def __init__(self, package_file_path, package_name, version, config, issue=None, review_id=None):
        if not config.has('sandbox', 'token'):
            raise ValueError('No Sandbox token')
        self.sandbox_client = rest.Client(auth=OAuth(config['sandbox']['token']))

        derived_tags = []
        if package_name.startswith('yandex-passport-'):
            derived_tags = package_name[len('yandex-passport-'):].split('-')

        json_file = get_relative_path(package_file_path)
        task_params = {
            'build_type': 'release',
            'checkout_mode': 'auto',
            'compress_package_archive': True,
            'packages': json_file,
            'package_type': 'debian',
            'publish_package': True,
            'run_tests': True,
            'save_build_output': True,
            'use_aapi_fuse': True,
            'use_arc_instead_of_aapi': True,
            'use_new_format': True,
            'ya_timeout': 10800,
            'wait_for_review': review_id,
            'ya_yt_store': False,
        }
        task_params.setdefault('publish_to_mapping', {json_file: 'passport'})
        task_params = smart_merge_dicts(
            task_params,
            config.get('sandbox', {}).get('task', {}).get('context', {}),
            list_policy='override',
        )

        self.args = {
            'description': self.get_description(
                package_name=package_name,
                version=version,
                issue=issue,
            ),
            'owner': 'PASSPORT',
            'tags': ['passport'] + derived_tags,
            'type': 'PASSPORT_YA_PACKAGE_2',
            'custom_fields': [
                {'name': key, 'value': val}
                for key, val in task_params.items()
            ],
        }

        self.task = self.sandbox_client.task(self.args)

    def get_description(self, package_name, version, issue=None):
        if not issue:
            return package_name + ' ' + version
        else:
            return issue.upper() + ' ' + package_name + ' ' + version

    def run(self):
        self.sandbox_client.batch.tasks.start.update([self.task['id']])

    @property
    def id(self):
        return self.task['id']

    @property
    def url(self):
        return 'https://sandbox.yandex-team.ru/task/{id}'.format(id=self.id)

    def __str__(self):
        return str(self.args)

    def __repr__(self):
        return str(self)


def run_sandbox_command(parsed_args, path='.', package_name=None, version=None, review_id=None, config=None):
    if config is None:
        config = Configurator('passport-ci', configs=['~/.ci.yaml?', os.path.join(path, '.ci.yaml?')])
    packaging = Packaging(base_path=path, deb_path=config.get('deb_path'))
    if not package_name or not version:
        _, package_name, version = packaging.ensure()

    try:
        sandbox_task = SandboxTask(
            package_file_path=os.path.join(
                packaging.base_path,
                packaging.deb_path,
                config.get('package_file_path', package_name + '.json'),
            ),
            package_name=package_name,
            version=version,
            config=config,
            issue=parsed_args.issue,
            review_id=review_id or parsed_args.review_id,
        )
    except (IOError, ValueError):
        raise RuntimeError('Run `ci config` to acquire Sandbox token')

    if not parsed_args.do_not_start:
        sandbox_task.run()

    print('\n-----===== Sandbox Task =====-----')
    print(sandbox_task.url)
    return sandbox_task
