import json

import requests
import requests.adapters


class BaseClient(object):
    base_url = None
    headers = None
    timeout = 1

    def __init__(self, base_url=None, timeout=None, max_pool_size=10, headers=None, token=None):
        self.base_url = base_url or self.base_url or None
        if self.base_url:
            self.base_url = self.base_url.rstrip('/')

        self.timeout = timeout or self.timeout
        self.max_pool_size = max_pool_size
        self.headers = headers or self.headers or {}

        if token:
            self.headers['Authorization'] = 'OAuth %s' % token

        self.session = requests.Session()

        self.requests_adapter = requests.adapters.HTTPAdapter(self.max_pool_size)
        self.session.mount('http://', self.requests_adapter)
        self.session.mount('https://', self.requests_adapter)

    def request(
        self,
        method,
        url,
        params=None,
        data=None,
        files=None,
        headers=None,
        cookies=None,
        verify='/etc/ssl/certs/ca-certificates.crt',
        response_processor=None
    ):
        real_headers = dict(self.headers)
        real_headers.update(headers or {})

        response_processor = response_processor or self.response_processor

        if real_headers.get('Content-Type') == 'application/json':
            data = json.dumps(data)

        response = self.session.request(
            method,
            self.base_url + '/' + url.lstrip('/') if self.base_url else url,
            params=params,
            data=data,
            files=files,
            headers=real_headers,
            cookies=cookies,
            timeout=self.timeout,
            verify=verify,
        )
        return response_processor(response)

    def get(self, *args, **kwargs):
        return self.request('get', *args, **kwargs)

    def post(self, *args, **kwargs):
        return self.request('post', *args, **kwargs)

    def delete(self, *args, **kwargs):
        return self.request('delete', *args, **kwargs)

    def put(self, *args, **kwargs):
        return self.request('put', *args, **kwargs)

    def response_processor(self, response):
        if response.headers['Content-Type'] == 'application/json':
            return json.loads(response.text)
        return response.text
