# -*- coding: utf-8 -*-

from __future__ import absolute_import

import logging

from cassandra import (
    ConsistencyLevel,
    OperationTimedOut,
    Timeout,
    Unavailable,
)
from cassandra.cluster import Cluster
from cassandra.concurrent import execute_concurrent_with_args
from cassandra.policies import WhiteListRoundRobinPolicy
from passport.backend.clients.cassandra.exceptions import CassandraException
from retry import retry


log = logging.getLogger(__name__)
error_log = logging.getLogger('error.' + __name__)


class CassandraClient(object):

    INSERT_PHONES_STATS_TEMPLATE = 'INSERT INTO %(table_name)s (phone, data) VALUES (?, ?) USING TTL %(ttl)s'

    def __init__(self, hosts, write_consistency_level, keyspace):
        self.hosts = hosts
        self.write_consistency_level = write_consistency_level
        self.keyspace = keyspace

        self.initialized = False

    def prepare_queries(self, table_name, ttl):
        if self.initialized:
            return

        self.cluster = Cluster(
            contact_points=self.hosts,
            load_balancing_policy=WhiteListRoundRobinPolicy(self.hosts),
            protocol_version=3,
        )
        self.session = self.cluster.connect(self.keyspace)

        phones_template = self.INSERT_PHONES_STATS_TEMPLATE % dict(
            table_name=table_name,
            ttl=ttl,
        )
        self.prepared_insert_phones_query = self.session.prepare(phones_template)
        self.prepared_insert_phones_query.consistency_level = ConsistencyLevel.name_to_value[self.write_consistency_level]

        self.initialized = True

    def insert_phone(self, table_name, ttl, phone, data):
        try:
            self.prepare_queries(table_name=table_name, ttl=ttl)
            self.session.execute(self.prepared_insert_phones_query, (phone, data.encode('utf-8')))
        except (Unavailable, Timeout, OperationTimedOut) as e:
            log.error('Cannot write phone to cassandra for phone %s', phone, exc_info=e)
            raise CassandraException()

    @retry(CassandraException, tries=5, delay=5, logger=error_log)
    def insert_phones(self, rewindable_iterator, table_name, ttl, concurrency):
        try:
            self.prepare_queries(table_name=table_name, ttl=ttl)
            execute_concurrent_with_args(
                self.session,
                self.prepared_insert_phones_query,
                rewindable_iterator,
                concurrency=concurrency,
            )
        except (Unavailable, Timeout, OperationTimedOut) as e:
            log.error('Cannot write phones to cassandra', exc_info=e)
            rewindable_iterator.rewind()
            raise CassandraException()
