# -*- coding: utf-8 -*-

import logging
import re

from passport.backend.clients.base import BaseClient
from passport.backend.clients.conductor.exceptions import CreationImpossibleError


log = logging.getLogger(__name__)
error_log = logging.getLogger('error.' + __name__)


class ConductorClient(BaseClient):
    base_url = 'https://c.yandex-team.ru/'
    headers = {
        'Accept': 'application/json',
        'Content-Type': 'application/json',
    }
    timeout = 10

    def get_ticket(self, id):
        return self.get('api/v1/tickets/%s' % id)

    def create_ticket(self, packages_with_versions=None, comment=None, branch='testing', do_not_autoinstall=False):
        packages = {}
        versions = {}
        for i, package_with_version in enumerate(packages_with_versions):
            packages[str(i)] = package_with_version['package']
            versions[str(i)] = package_with_version['version']
        ticket_key = self.post(
            'auth_update/ticket_add',
            data={
                'ticket': {
                    'branch': branch,
                    'comment': comment,
                    'do_not_autoinstall': 1 if do_not_autoinstall else 0,
                },
                'package': packages,
                'version': versions,
            },
            response_processor=self._extract_ticket_key,
        )
        return self.make_ticket_url(ticket_key)

    def make_ticket_url(self, ticket_key):
        return '{base_url}/tickets/{ticket_number}'.format(
            base_url=self.base_url.rstrip('/'),
            ticket_number=ticket_key,
        )

    def _extract_ticket_key(self, response):
        r = re.search('^Ticket ([0-9]+)', response.text)
        if r:
            return r.group(1)
        else:
            raise CreationImpossibleError(response.text)
