#!/usr/bin/env python
# coding: utf-8

"""
Passport Golovan configuration util
"""

from __future__ import (
    print_function,
    unicode_literals,
)

import argparse
import json
import logging
import os
import re
import socket
import sys
import urllib2


try:
    import contrib.python.yenv
except ImportError:
    import yenv


__version__ = '1.1'

CONDUCTOR_URL_TEMPLATE = 'https://c.yandex-team.ru/api-cached/hosts/{}?format=json'
CONDUCTOR_HTTP_TIMEOUT = 10

HOST_GEO_RE = re.compile(r'^[^\.]+?-(?P<geo>\w)\d+\.')
HOST_GEO_MAPPING = {
    'f': 'man',
    'i': 'iva',
    'm': 'myt',
    's': 'sas',
    'v': 'vla',
}

DEFAULT_GEO = 'none'
DEFAULT_CTYPE = yenv.type
DEFAULT_PRJ = 'passport'


class ArgsParser(argparse.ArgumentParser):
    def __init__(self, version=__version__, geo=DEFAULT_GEO, ctype=DEFAULT_CTYPE, prj=DEFAULT_PRJ, *args, **kwargs):
        super(ArgsParser, self).__init__(
            formatter_class=argparse.RawDescriptionHelpFormatter,
            *args,
            **kwargs
        )
        self.add_argument('--version', action='version', version='%(prog)s {}'.format(version))
        self.add_argument('--debug', dest='debug', action='store_true', help='Print debug info')
        self.add_argument('-i', '--itype', dest='itype', default='common', help='default: %(default)s')
        self.add_argument('-c', '--ctype', dest='ctype', default=ctype, help='default: %(default)s')
        self.add_argument('-p', '--prj', dest='prj', default=prj, help='default: %(default)s')
        self.add_argument('-g', '--geo', dest='geo', default=geo, help='default: %(default)s')
        self.add_argument('-t', '--tier', dest='tier', default='none', help='default: %(default)s')

    def error(self, message, stderr=None):
        stderr = stderr or sys.stderr
        print('error: %s' % message, file=stderr)
        print('-' * 20, file=stderr)
        self.print_help(file=stderr)
        sys.exit(2)


def _load(suffix, default):
    var_name = 'YENV_%s' % suffix.upper()

    value = os.environ.get(var_name)

    if not value:
        value = default

        try:
            with open('/etc/yandex/environment.%s' % suffix) as f:
                value = f.read().strip()
        except IOError:
            pass

        os.environ[var_name] = value

    return value


def fetch_url(url, timeout=60):
    f = urllib2.urlopen(url)
    return f.read()


def fetch_geo_from_conductor():
    result = ''
    try:
        fqdn = socket.getfqdn()
        data = fetch_url(
            CONDUCTOR_URL_TEMPLATE.format(fqdn),
            timeout=CONDUCTOR_HTTP_TIMEOUT,
        )
        return json.loads(data)[0]['root_datacenter'].strip().lower()
    except Exception as e:
        logging.debug(e, exc_info=True)

    return result


def fetch_geo_from_hostname():
    result = ''
    fqdn = socket.getfqdn()
    match = HOST_GEO_RE.match(fqdn)
    if match:
        result = HOST_GEO_MAPPING.get(match.group('geo').lower(), '')
    return result


def fetch_geo():
    result = fetch_geo_from_hostname()

    if not result:
        result = fetch_geo_from_conductor()

    return result or DEFAULT_GEO


def run():
    args = ArgsParser(
        description=__doc__,
        prj=_load('project', DEFAULT_PRJ),
        geo=fetch_geo(),
    ).parse_args()

    logging.basicConfig(
        level=logging.DEBUG if args.debug else logging.ERROR,
        format='%(levelname)s [%(name)s] %(message)s',
    )

    print('localhost@{itype} a_itype_{itype} a_ctype_{ctype} a_geo_{geo} a_prj_{prj} a_tier_{tier}'.format(
        itype=args.itype,
        ctype=args.ctype + '.intranet' if yenv.name == 'intranet' else args.ctype,
        prj=args.prj,
        geo=args.geo,
        tier=args.tier,
    ))


if __name__ == '__main__':
    run()
