# -*- coding: utf-8 -*-
from collections import defaultdict
from enum import Enum

from passport.backend.core.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)


class Platforms(Enum):
    android = 1
    ios = 2


PLATFORM_TO_PUSH_SERVICES_MAPPING = {
    Platforms.android: ['fcm', 'gcm'],
    Platforms.ios: ['apns'],
}

PUSH_SERVICE_TO_PLATFORM_MAPPING = {
    push_service: platform
    for platform, push_services in PLATFORM_TO_PUSH_SERVICES_MAPPING.items()
    for push_service in push_services
}


class UnknownPushService(ValueError):
    """ Неизвестный push сервис подписки """


@lazy_loadable('AMCapabilitiesManager')
class AMCapabilitiesManager(object):
    def __init__(self):
        # {platform: [{'version': (x,y), 'caps': {cap1, cap2, ...}), ...]} по возрастанию версии
        self.capabilities_by_platform = defaultdict(list)
        # {capability: {platform: min_version, ...}}
        self.min_version_by_caps = defaultdict(dict)

        self.existing_capabilities = set()

        for platform, capabilities in [
            (Platforms.android, settings.AM_CAPABILITIES_BY_VERSION_ANDROID),
            (Platforms.ios, settings.AM_CAPABILITIES_BY_VERSION_IOS),
        ]:
            collected_caps = set()
            for version in sorted(capabilities.keys()):
                collected_caps |= capabilities[version]
                self.capabilities_by_platform[platform].append(
                    dict(version=version, caps=collected_caps),
                )

                for capability in capabilities[version]:
                    if platform not in self.min_version_by_caps[capability]:
                        self.min_version_by_caps[capability][platform] = version
                    self.existing_capabilities.add(capability)

    def get_min_version_by_cap(self, cap, platform):
        try:
            return self.min_version_by_caps[cap][platform]
        except KeyError as err:
            raise ValueError('Capability {} for platform {} not found: {}'.format(cap, platform, err))

    def get_min_am_version_by_caps(self, platform, required_capabilities):
        """
        Получить минимальную версию AM по набору возможностей

        :param platform: платформа (enum Platforms)
        :param required_capabilities: список требуемых возможностей
        :type required_capabilities: Iterable[str]
        :return: version tuple or None if caps are empty
        """
        if not required_capabilities:
            return None
        return max(
            self.get_min_version_by_cap(cap, platform)
            for cap in required_capabilities
        )


def get_am_capabilities_manager():
    # type:() -> AMCapabilitiesManager
    return LazyLoader.get_instance('AMCapabilitiesManager')


def get_platform_by_push_service(push_service):
    return PUSH_SERVICE_TO_PLATFORM_MAPPING.get(push_service)
