# -*- coding: utf-8 -*-
import json
import uuid

from passport.backend.core.am_pushes.common import (
    get_am_capabilities_manager,
    Platforms,
)
from passport.backend.core.protobuf.am_push_protocol.am_push_protocol_pb2 import PushMessageRequest
from passport.backend.utils.string import (
    smart_bytes,
    smart_text,
)


class AppTargetingTypes(object):
    GLOBAL = 0              # Отправить во все приложения
    GLOBAL_WITH_CAPS = 1    # Отправить во все приложения, способные обработать пуш
    ONE_APP_PER_DEVICE = 2  # Отправить в одно приложение на каждом устройстве
    CLIENT_DECIDES = 3      # Автовыбор на стороне клиента


APP_TARGETING_TYPE_TO_NAME = {
    getattr(AppTargetingTypes, attr): attr
    for attr in dir(AppTargetingTypes) if attr.isupper()
}


class SubscriptionSources(object):
    AM = 0
    YAKEY = 1


SUBSCRIPTION_SOURCE_TO_NAME = {
    getattr(SubscriptionSources, attr): attr
    for attr in dir(SubscriptionSources) if attr.isupper()
}


class PushRequestRecipient(object):
    def __init__(
        self,
        uid,
        app_targeting_type=AppTargetingTypes.GLOBAL,
        custom_app_priority=None,
        required_am_capabilities=None,
        required_platforms=None,
        require_trusted_device=False,
        device_ids=None,
        subscription_source=SubscriptionSources.AM,
        context=None,
    ):
        """
        Получатель push-сообщения

        :rtype: object
        :param uid: uid пользователя
        :type uid: int
        :param app_targeting_type: способ определения, какое приложение
            покажет пуш
        :type app_targeting_type: int (константы AppTargetingTypes.*)
        :param custom_app_priority: специальный приоритет приложений
        :type custom_app_priority: Optional[List[str]]
        :param required_am_capabilities: требуемые возможности AM
        :type required_am_capabilities: Optional[List[str]]
        :param required_platforms: требуемые платформы
        :type required_platforms: Optional[List[int]] (константы Platforms.*)
        :param require_trusted_device: отправлять только на доверенные устройства
        :type require_trusted_device: bool
        :param device_ids: отправлять только на указанные устройства
        :type device_ids: Optional[List[str]]
        :param subscription_source: в какие приложения отправлять
        :type subscription_source: int (константы SubscriptionSources.*)
        :param context: контекст пуша, произвольные данные
        :type context: str
        """
        self.uid = uid
        self.app_targeting_type = app_targeting_type

        if self.app_targeting_type != AppTargetingTypes.ONE_APP_PER_DEVICE and custom_app_priority:
            raise ValueError(
                'Cannot set custom_app_priority for app_targeting_type '
                'other than ONE_APP_PER_DEVICE',
            )
        self.custom_app_priority = custom_app_priority

        self.subscription_source = subscription_source

        if self.app_targeting_type == AppTargetingTypes.GLOBAL and required_am_capabilities:
            raise ValueError('Cannot set required_am_capabilities for app_targeting_type "GLOBAL"')
        self.required_am_capabilities = required_am_capabilities
        if self.required_am_capabilities:
            self._validate_am_capabilities()

        if self.app_targeting_type == AppTargetingTypes.GLOBAL and required_platforms:
            raise ValueError('Cannot set required_platforms for app_targeting_type "GLOBAL"')
        self.required_platforms = required_platforms

        if self.app_targeting_type == AppTargetingTypes.GLOBAL and require_trusted_device:
            raise ValueError('Cannot set require_trusted_device for app_targeting_type "GLOBAL"')
        if self.subscription_source != SubscriptionSources.AM and require_trusted_device:
            raise ValueError(
                'Cannot set require_trusted_device for subscription_source "%s"' % (
                    SUBSCRIPTION_SOURCE_TO_NAME[self.subscription_source],
                ),
            )
        self.require_trusted_device = require_trusted_device

        if self.app_targeting_type == AppTargetingTypes.GLOBAL and device_ids:
            raise ValueError('Cannot set device_ids for app_targeting_type "GLOBAL"')
        self.device_ids = device_ids

        self.context = context

    def _validate_am_capabilities(self):
        manager = get_am_capabilities_manager()
        non_existent = [
            cap for cap in self.required_am_capabilities
            if cap not in manager.existing_capabilities
        ]
        if non_existent:
            raise ValueError('Capabilities not found: {}'.format(non_existent))

    def to_proto(self):
        return PushMessageRequest.PushMessageRecipient(
            uid=self.uid,
            app_targeting_type=self.app_targeting_type,
            custom_app_priority=self.custom_app_priority,
            required_am_capabilities=self.required_am_capabilities,
            required_platforms=[p.value for p in self.required_platforms or []],
            require_trusted_device=self.require_trusted_device,
            device_ids=self.device_ids,
            subscription_source=self.subscription_source,
            context=self.context,
        )

    @classmethod
    def from_proto(cls, proto):
        obj = cls.__new__(cls)
        obj.uid = proto.uid
        obj.app_targeting_type = proto.app_targeting_type
        obj.custom_app_priority = proto.custom_app_priority or None
        obj.required_am_capabilities = proto.required_am_capabilities or None
        obj.required_platforms = [Platforms(p) for p in proto.required_platforms] or None
        obj.require_trusted_device = proto.require_trusted_device
        obj.device_ids = proto.device_ids or None
        obj.subscription_source = proto.subscription_source
        obj.context = proto.context or None
        return obj


class PushRequest(object):
    AM_PROTO_VERSION = '1.0'

    def __init__(
        self,
        push_service,
        event_name,
        recipients,
        title=None,
        body=None,
        subtitle=None,
        webview_url=None,
        require_web_auth=True,
        push_id=None,
        expire_time=None,
        extra_data=None,
    ):
        """
        Запрос на push сообщение
        Если пуш не silent, то нужно задать либо text, либо webview_url.

        :param push_service: имя сервиса: может определять, требуется ли
            специальная обработка этого типа пуша
        :type push_service: str
        :param event_name: имя события: может определять, требуется ли
            специальная обработка этого типа пуша
        :type push_service: str
        :param recipients: получатели пуша
        :type recipients: PushRequestRecipient|List[PushRequestRecipient]
        :param title: заголовок сообщения
        :type title: Optional[str]
        :param body: текст сообщения
        :type body: Optional[str]
        :param subtitle: подзаголовок сообщения, для android пришивается
            к text как subtitle + '\n' + text
        :type subtitle: Optional[str]
        :param webview_url: url для webview
        :type webview_url: Optional[str]
        :param require_web_auth: требовать куки для webview, default=True
        :type require_web_auth: bool
        :param push_id: id рассылки, если не указан - генерируется автоматически
        :type push_id: str
        :param expire_time: время актуальности пуша
        :type expire_time: Optional[int]
        :param extra_data: произвольные поля, которые будут переданы в пуше
        :type extra_data: Optional[dict]
        """
        self.push_service = push_service
        self.event_name = event_name

        if not isinstance(recipients, list):
            recipients = [recipients]
        self.recipients = recipients

        self.title = title
        self.body = body
        self.subtitle = subtitle

        if webview_url:
            self.webview_url = webview_url
            self.require_web_auth = require_web_auth
        else:
            self.webview_url = None
            self.require_web_auth = None

        self.push_id = push_id or self._generate_push_id()

        self.expire_time = expire_time

        self.extra_data = extra_data

    @staticmethod
    def _generate_push_id():
        return str(uuid.uuid4())

    def to_proto(self):
        assert isinstance(self.extra_data, dict) or self.extra_data is None
        proto = PushMessageRequest(
            push_service=self.push_service,
            event_name=self.event_name,
            recipients=(r.to_proto() for r in self.recipients),
            text_body=PushMessageRequest.PushMessageTextBody(
                title=smart_bytes(self.title),
                body=smart_bytes(self.body) if self.body is not None else '',
                subtitle=smart_bytes(self.subtitle) if self.subtitle is not None else '',
            ),
            extra_json=json.dumps(self.extra_data) if self.extra_data else '',
            push_id=self.push_id,
            expire_time=self.expire_time,
        )
        if self.webview_url is not None:
            proto.webview_body.webview_url = smart_bytes(self.webview_url)
            proto.webview_body.require_web_auth = self.require_web_auth
        return proto

    @classmethod
    def from_proto(cls, proto):
        obj = cls.__new__(cls)
        obj.push_service = proto.push_service
        obj.event_name = proto.event_name
        obj.recipients = [
            PushRequestRecipient.from_proto(r) for r in proto.recipients
        ]
        obj.extra_data = json.loads(proto.extra_json) if proto.extra_json else None
        assert isinstance(obj.extra_data, dict) or obj.extra_data is None

        obj.title = smart_text(proto.text_body.title) or None
        obj.body = smart_text(proto.text_body.body) or None
        obj.subtitle = smart_text(proto.text_body.subtitle) or None

        obj.push_id = smart_text(proto.push_id)
        obj.expire_time = proto.expire_time or None

        if proto.HasField('webview_body'):
            obj.webview_url = smart_text(proto.webview_body.webview_url)
            obj.require_web_auth = proto.webview_body.require_web_auth
        else:
            obj.webview_url = None
            obj.require_web_auth = None

        return obj
