# -*- coding: utf-8 -*-
from library.python.bloom import BloomFilter
from passport.backend.core.dynamic_config import (
    BaseDynamicConfig,
    LoadConfigsError,
)


# Как правило, конфиги для bloom-фильтра очень большие, их стоит перечитывать только на старте приложения,
# чтобы не замедлять обработку пользовательских запросов
DEFAULT_CACHE_TIME = 2 * 24 * 3600


class BaseBloomSet(BaseDynamicConfig):
    """
    https://en.wikipedia.org/wiki/Bloom_filter

    Потребление памяти (в битах) пропорционально -ln(max_error) * element_count / (ln 2)^2
    """
    def __init__(self, filename, max_error, cache_time=None):
        super(BaseBloomSet, self).__init__(
            config_filenames=[filename],
            cache_time=cache_time or DEFAULT_CACHE_TIME,
        )
        self._max_error = max_error

    def read_config_file(self, filename):
        # В отличие от остальных DynamicConfig, не будем сохранять все сырые данные в памяти, а сразу будем строить
        # блум-фильтр. Файл придётся прочесть дважды: первым прочтением определить требуемый размер блум-фильтра,
        # вторым - построить блум-фильтр.
        try:
            with open(filename, 'r') as f:
                for i, _ in enumerate(f):
                    pass
            element_count = i + 1

            bloom_filter = BloomFilter(capacity=element_count, error=self._max_error)
            with open(filename, 'r') as f:
                for line in f:
                    item = line.strip()
                    if item:
                        bloom_filter.add(item.encode('utf8'))

            return bloom_filter
        except IOError as e:
            raise LoadConfigsError(e)

    def has_item(self, item_str):
        if self.config is None:
            raise RuntimeError('%s not loaded yet' % self.__class__.__name__)
        return self.config.has(item_str.encode('utf8'))

    def __contains__(self, item_str):
        return self.has_item(item_str)
