# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.abc.exceptions import (
    ABCAuthorizationInvalidError,
    ABCPermanentError,
    ABCTemporaryError,
    BaseABCError,
)
from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger


# ABC API v4 требует явно указать все поля, которые нужны в ответе. По-умолчанию возвращают совсем мало
GET_SERVICE_MEMBERS_FIELDS = ','.join((
    'id',
    'state',
    'person.id',
    'person.login',
    'person.first_name',
    'person.last_name',
    'person.uid',
    'service.id',
    'service.slug',
    'service.name',
    'service.parent',
    'role.id',
    'role.code',
    'role.name',
    'role.service',
    'role.scope.slug',
    'role.scope.name',
    'role.scope.id',
    'created_at',
    'modified_at',
))

log = logging.getLogger('passport.abc')


def abc_http_error_handler(raw_response):
    if raw_response.status_code == 504:
        # Backend timeout (фикс для VAULT-608)
        log.warning(
            u'Request failed with response=%s code=%s',
            trim_message(raw_response.content.decode('utf-8')),
            raw_response.status_code,
        )
        raise ABCTemporaryError('ABC backend timeout')
    elif raw_response.status_code >= 500:
        log.warning(
            u'Request failed with response=%s code=%s',
            trim_message(raw_response.content.decode('utf-8')),
            raw_response.status_code,
        )
        raise ABCPermanentError('Server is down')
    elif raw_response.status_code == 403:
        raise ABCAuthorizationInvalidError('Invalid authorization')
    elif raw_response.status_code != 200:
        raise ABCPermanentError('Bad response status code: %s' % raw_response.status_code)


class ABC(BaseBuilder, JsonParserMixin):
    base_error_class = BaseABCError
    temporary_error_class = ABCTemporaryError
    parser_error_class = ABCPermanentError

    page_size = 50
    max_pages = 2**16  # Для защиты от зацикливания курсорной пейджинации

    """
    Дока к апи: https://wiki.yandex-team.ru/intranet/abc/api/
    """

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None,
                 tvm_dst_alias='abc', use_tvm=False, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='abc')
        super(ABC, self).__init__(
            url=url or settings.ABC_URL,
            timeout=timeout or settings.ABC_TIMEOUT,
            retries=retries or settings.ABC_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias=tvm_dst_alias if use_tvm else None,
            **kwargs
        )

    def _make_request(self, method='GET', url_suffix='', params=None, data=None, oauth_token=None, session_id=None,
                      parser=None, **kwargs):
        headers = {}
        if self.tvm_dst_alias is None:
            if oauth_token is not None:
                headers['Authorization'] = 'OAuth %s' % oauth_token
            elif session_id is not None:
                headers['Cookie'] = 'Session_id=%s' % session_id
            else:
                raise ValueError('Either oauth_token or sessionid is required when TVM is not used')

        return self._request_with_retries_simple(
            url_suffix=url_suffix,
            method=method,
            params=params,
            data=data,
            http_error_handler=abc_http_error_handler,
            parser=parser or self.parse_json,
            error_detector=None,
            headers=headers,
            **kwargs
        )

    def _make_request_with_cursor_pagination(self, method='GET', url_suffix='', params=None, data=None,
                                             oauth_token=None, session_id=None, page_size=None, max_pages=None,
                                             **kwargs):
        """
        Ходим по ссылке из поля next в ответе ABC API v4
        """
        results = []
        page_size = page_size or self.page_size
        max_pages = max_pages or self.max_pages

        url = self.url
        params = dict(params) if params is not None else {}
        params['page_size'] = page_size

        page = 0
        while page < max_pages:
            rv = self._make_request(
                url=url,
                url_suffix=url_suffix,
                method=method,
                params=params,
                data=data,
                oauth_token=oauth_token,
                session_id=session_id,
                **kwargs
            )
            results += rv['results']
            page += 1

            url = rv.get('next')
            url_suffix = ''
            params = None

            if not url:
                return results

        log.warning('Max pages loaded, stopping')
        return results

    def get_service_members(self, filter_args=None, **kwargs):
        params = dict(filter_args) if filter_args is not None else {}
        if 'fields' not in params:
            params['fields'] = GET_SERVICE_MEMBERS_FIELDS

        return self._make_request_with_cursor_pagination(
            url_suffix='v4/services/members/',
            method='GET',
            params=params,
            **kwargs
        )


def get_abc():
    return ABC()  # pragma: no cover
