# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.afisha.exceptions import (
    AfishaApiInvalidResponseError,
    AfishaApiNotFoundError,
    AfishaApiPermanentError,
    AfishaApiTemporaryError,
    BaseAfishaApiError,
)
from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.afisha_api')


def afisha_api_error_detector(response, raw_response):
    if raw_response.status_code == 404 or response.get('status') == 404:
        raise AfishaApiNotFoundError()

    if raw_response.status_code != 200 or response.get('status') != 200 or 'data' not in response:
        message = u'AfishaApi bad response (status=%s code=%s response=%s)' % (
            response.get('status'),
            raw_response.status_code,
            trim_message(raw_response.content.decode('utf-8')),
        )
        raise AfishaApiInvalidResponseError(message)


def afisha_api_schedule_error_detector(response, raw_response):
    if (
        response.get('status') == 400 and
        'need to specify city' in response.get('message', '').lower()
    ):
        raise AfishaApiNotFoundError()

    afisha_api_error_detector(response, raw_response)


def afisha_api_http_error_handler(raw_response):
    if raw_response.status_code >= 500:
        message = u'Request failed with response=%s code=%s' % (
            trim_message(raw_response.content.decode('utf-8')),
            raw_response.status_code,
        )
        raise AfishaApiPermanentError(message)


def afisha_api_response_processor(response):
    return response['data']


class AfishaApi(BaseBuilder, JsonParserMixin):
    """
    Документация к API https://github.yandex-team.ru/pages/Afisha/backend/api/resources.html
    """
    base_error_class = BaseAfishaApiError
    temporary_error_class = AfishaApiTemporaryError
    parser_error_class = AfishaApiInvalidResponseError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='afisha_api')
        super(AfishaApi, self).__init__(
            url=url or settings.AFISHA_API_URL,
            timeout=timeout or settings.AFISHA_API_TIMEOUT,
            retries=retries or settings.AFISHA_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            **kwargs
        )

    def events(self, uid, tags=None):
        """
        Для истории: можно ходить в ручку 3.x/users/%s/favourites/
        и получать только список event_ids.
        Сейчас получаем объекты с минимальными данными по событиям
        и флагом посещенности данным пользователем.
        """
        params = {}
        if tags:
            params['tags'] = tags
        try:
            return self._request_with_retries_simple(
                url_suffix='3.x/users/%s/favourites/events/' % uid,
                method='GET',
                params=params,
                error_detector=afisha_api_error_detector,
                parser=self.parse_json,
                http_error_handler=afisha_api_http_error_handler,
                response_processor=afisha_api_response_processor,
            )
        except AfishaApiNotFoundError:
            return {'events': []}

    def events_schedules(self, city_id, event_ids, limit=None, offset=None, date=None,
                         tags=None, place_id=None, period=None, is_sale_available=None):
        params = {
            'limit': limit,
            'offset': offset,
            'event_id': event_ids,
            'period': period,
            'date': date,
            'tag': tags,
            'place_id': place_id,
            'sale_available': is_sale_available,
        }
        try:
            return self._request_with_retries_simple(
                url_suffix='3.x/cities/%s/schedule_info' % city_id,
                method='GET',
                params=params,
                error_detector=afisha_api_schedule_error_detector,
                parser=self.parse_json,
                http_error_handler=afisha_api_http_error_handler,
                response_processor=afisha_api_response_processor,
            )
        except AfishaApiNotFoundError:
            return []


def get_afisha_api():
    return AfishaApi()  # pragma: no cover
