# -*- coding: utf-8 -*-
from collections import namedtuple
import datetime
import json
import logging

from passport.backend.core.builders.base.base import (
    BaseBuilder,
    parser_trivial,
)
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.utils.time import datetime_to_unixtime

from .exceptions import (
    AntifraudApiInvalidResponse,
    AntifraudApiPermanentError,
    AntifraudApiTemporaryError,
    BaseAntifraudApiError,
)


log = logging.getLogger('passport.antifraud_api')


ScoreAction = namedtuple('ScoreAction', ['ALLOW', 'DENY'])._make(['ALLOW', 'DENY'])
ScoreResponse = namedtuple('ScoreResponse', ['action', 'reason', 'tags'])
SCORE_TAGS_ALLOWED = (
    'sms', 'flash_call', 'call', 'phone_hint', 'email_hint', 'question', 'bank_sms', 'push_2fa',
    'email_code',
)
UIDCardsResponse = namedtuple('UIDCardsResponse', ['action', 'reason', 'tags', 'uids_with_cards'])


def score_error_detector(response, _raw_response):
    required_keys = ['status', 'action', 'reason', 'tags']
    for key in required_keys:
        if key not in response:
            raise AntifraudApiInvalidResponse('Key "{}" is absent from response'.format(key))
    if response['status'] != 'success':
        raise AntifraudApiTemporaryError('Got status: "{}", expected "success"'.format(response['status']))
    else:
        action = response['action']
        if action not in ScoreAction:
            raise AntifraudApiInvalidResponse('Unexpected action {}'.format(action))


def antifraud_http_error_handler(raw_response):
    if raw_response.status_code != 200:
        message = u'Request failed with response=%s code=%s' % (
            trim_message(raw_response.content.decode('utf-8')),
            raw_response.status_code,
        )
        raise AntifraudApiPermanentError(message)


def score_response_processor(response):
    response_tags = [tag for tag in (response['tags'] or []) if tag in SCORE_TAGS_ALLOWED]
    return ScoreResponse(
        action=response['action'],
        reason=response['reason'],
        tags=response_tags,
    )


def uid_cards(response):
    response_tags = [tag for tag in (response['tags'] or [])]
    uids_to_cards = response.get('verified_cards_per_uid', dict())
    uids_with_cards = dict()
    for uid, cards in uids_to_cards.items():
        uids_with_cards[int(uid)] = bool(cards)
    return UIDCardsResponse(
        action=response['action'],
        reason=response['reason'],
        tags=response_tags,
        uids_with_cards=uids_with_cards,
    )


class AntifraudApi(BaseBuilder, JsonParserMixin):
    """
    https://wiki.yandex-team.ru/antispam/antifrod/protokol/
    """
    base_error_class = BaseAntifraudApiError
    temporary_error_class = AntifraudApiTemporaryError
    parser_error_class = AntifraudApiInvalidResponse

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='antifraud_api')
        super(AntifraudApi, self).__init__(
            url=url or settings.ANTIFRAUD_API_URL,
            timeout=timeout or settings.ANTIFRAUD_API_TIMEOUT,
            retries=retries or settings.ANTIFRAUD_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='antifraud_api',
            **kwargs
        )

    @property
    def request_deadline(self):
        deadline = datetime.datetime.now() + datetime.timedelta(seconds=settings.ANTIFRAUD_API_TIMEOUT)
        return int(datetime_to_unixtime(deadline) * 1e3)

    def score(self, params):
        return self._request_with_retries_simple(
            url_suffix='/score',
            method='POST',
            params={'consumer': 'passport'},
            data=json.dumps(params),
            error_detector=score_error_detector,
            parser=self.parse_json,
            headers={'X-Request-Deadline': self.request_deadline},
            http_error_handler=antifraud_http_error_handler,
            response_processor=score_response_processor,
        )

    def save(self, params):
        return self._request_with_retries_simple(
            url_suffix='/save',
            method='POST',
            params={'consumer': 'passport'},
            data=json.dumps(params),
            error_detector=None,
            parser=parser_trivial,
            http_error_handler=antifraud_http_error_handler,
        )

    def uid_cards(self, uids):
        # uids - массив интов
        # https://wiki.yandex-team.ru/antispam/antifrod/protokol/#zaproskartskotorymiuidodnazhdyuspeshnosovershilverifirovannujuoperaciju
        return self._request_with_retries_simple(
            url_suffix='/execute',
            method='POST',
            params={'app': 'uid-cards'},
            data=json.dumps({'uids': [str(uid) for uid in uids]}),
            error_detector=None,
            parser=self.parse_json,
            headers={'X-Request-Deadline': self.request_deadline},
            http_error_handler=antifraud_http_error_handler,
            response_processor=uid_cards,
        )


def get_antifraud_api():
    return AntifraudApi()
