# -*- coding: utf-8 -*-
import json
import uuid

from passport.backend.core.builders.antifraud import (
    AntifraudApi,
    ScoreAction,
)
from passport.backend.core.builders.base.faker.fake_builder import (
    BaseFakeBuilder,
    FakeBuilderError,
)
from passport.backend.core.builders.blackbox.faker.blackbox import blackbox_build_badauth_counts
from passport.backend.core.conf import settings
from passport.backend.core.cookies.utils import parse_cookie
from passport.backend.core.env import Environment
from passport.backend.core.test.time_utils.time_utils import TimeNow
from passport.backend.utils.common import noneless_dict
from six.moves.urllib.parse import urlparse


class FakeAntifraudAPI(BaseFakeBuilder):
    def __init__(self):
        super(FakeAntifraudAPI, self).__init__(AntifraudApi)

    @staticmethod
    def parse_method_from_request(http_method, url, data, headers=None):
        url_path = urlparse(url).path

        if http_method == 'POST' and url_path == '/score':
            return 'score'
        elif http_method == 'POST' and url_path == '/save':
            return 'save'
        elif http_method == 'POST' and url_path == '/execute':
            return 'uid_cards'
        raise FakeBuilderError('Unknown url: "%s"' % url_path)  # pragma: no cover


class AntifraudScoreParams(object):
    _freq_features = [
        'as_list_freq',
        'browser_freq',
        'browser_name_freq',
        'browser_os_freq',
        'city_freq',
        'country_freq',
        'ip_freq',
        'it_am_version_freq',
        'it_as_list_freq',
        'it_city_freq',
        'it_country_freq',
        'it_ip_freq',
        'os_family_freq',
        'os_name_freq',
        'su_as_list_freq',
        'su_city_freq',
        'su_country_freq',
        'su_ip_freq',
        'yandexuid_freq',
    ]
    _all_features = _freq_features + [
        'AS',
        'am_version',
        'available_challenges',
        'badauth_counts',
        'browser_id',
        'channel',
        'city_id',
        'country_id',
        'device_id',
        'device_ifv',
        'device_name',
        'external_id',
        'fresh_profiles',
        'has_cards',
        'has_cards_by_linking',
        'has_cloud_token',
        'has_own_cards',
        'input_login',
        'ip',
        'is_mobile',
        'known_device',
        'lah_uids',
        'login_id',
        'os_id',
        'profile_loaded',
        'reg_date',
        'request',
        'retpath',
        'sms_2fa',
        'social_provider_code',
        'social_userid',
        'sub_channel',
        'surface',
        't',
        'traffic_fp',
        'trusted_device_ids',
        'uid',
        'user_agent',
        'yandexuid_timestamp',
    ]

    def __init__(self):
        for feature in self._all_features:
            setattr(self, feature, None)

    @classmethod
    def default(cls):
        self = AntifraudScoreParams()

        self.badauth_counts = blackbox_build_badauth_counts()
        self.channel = 'auth'
        self.fresh_profiles = []
        self.has_cloud_token = False
        self.lah_uids = []
        self.profile_loaded = True
        self.request = 'auth'
        self.sms_2fa = False
        self.sub_channel = settings.ANTIFRAUD_AUTH_SUB_CHANNEL
        self.t = TimeNow(as_milliseconds=True)
        self.trusted_device_ids = []

        for feature in self._freq_features:
            setattr(self, feature, dict())
        return self

    @classmethod
    def build_social_auth(cls):
        self = AntifraudScoreParams()

        self.channel = 'auth'
        self.request = 'auth'
        self.fresh_profiles = []
        self.has_cloud_token = False
        self.profile_loaded = False
        self.sms_2fa = False
        self.sub_channel = settings.ANTIFRAUD_AUTH_SUB_CHANNEL
        self.t = TimeNow(as_milliseconds=True)
        self.trusted_device_ids = []

        for feature in self._freq_features:
            setattr(self, feature, dict())

        return self

    def as_dict(self):
        return noneless_dict({feature: getattr(self, feature) for feature in self._all_features})

    @staticmethod
    def build_env_from_headers(headers):
        return Environment(
            accept_language=headers.get('Ya-Client-Accept-Language'),
            cookies=parse_cookie(headers.get('Ya-Client-Cookie')),
            user_agent=headers.get('Ya-Client-User-Agent'),
            user_ip=headers.get('Ya-Consumer-Client-Ip'),
        )

    def populate_from_headers(self, headers):
        env = self.build_env_from_headers(headers)
        self.ip = str(env.user_ip) if env.user_ip else None
        self.user_agent = env.user_agent if env.user_agent else None

        profile = env.make_profile()
        self.AS = profile.AS_list
        self.browser_id = profile.browser_id
        self.city_id = profile.city_id
        self.country_id = profile.country_id
        self.os_id = profile.os_id
        self.yandexuid_timestamp = profile.yandexuid_timestamp


def antifraud_score_response(action=ScoreAction.ALLOW, tags=None):
    tags = tags or []
    return json.dumps({
        'status': 'success',
        'action': action,
        'reason': 'some-reason',
        'tags': tags,
    })


def antifraud_verified_cards_per_uid_response(uids, action=ScoreAction.ALLOW, tags=None):
    cards = {}
    for uid in uids:
        cards[str(uid)] = {
            uuid.uuid4().hex: 5,
        }

    tags = tags or []
    retval = {
        'action': action,
        'tags': tags,
        'reason': '',
        'risk_score': 0,
        'rule_score': 0,
        'status': 'success',
        'tx_id': 'app/uid-cards:gen-adc04269-383f-4536-b7ad-4f4c8bfa9285',
        'verified_cards_per_uid': cards,
    }
    return json.dumps(retval)
