# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.avatars_mds_api.exceptions import (
    AvatarsMdsApiBadImageFormatError,
    AvatarsMdsApiImageNotFoundError,
    AvatarsMdsApiInvalidFileSizeError,
    AvatarsMdsApiInvalidImageSizeError,
    AvatarsMdsApiPermanentError,
    AvatarsMdsApiTemporaryError,
    BaseAvatarsMdsApiError,
)
from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.avatars_mds_api')


def parse_raw_file(raw_response):
    file_object = raw_response.raw
    file_object.decode_content = True
    return file_object


def download_error_detector(response, raw_response):
    status = raw_response.status_code
    if status == 404:
        raise AvatarsMdsApiImageNotFoundError()
    elif status != 200:
        raise AvatarsMdsApiTemporaryError('status=%s' % status)


def upload_error_detector(response, raw_response):
    status = raw_response.status_code
    description = response.get('description', '')

    if status == 200:
        return
    elif status == 403 and 'group-id' in response.get('attrs', {}):
        return  # гонка, аватарка уже залита
    elif status == 400:
        if 'cannot process image' in description or 'cannot get image info' in description:
            raise AvatarsMdsApiBadImageFormatError(description)
    elif status == 415:
        if 'Image is too small' in description:
            raise AvatarsMdsApiInvalidImageSizeError(description)
        elif 'bytes more than maximum allowed' in description or 'bytes less than minimum allowed' in description:
            raise AvatarsMdsApiInvalidFileSizeError(description)
        else:
            raise AvatarsMdsApiBadImageFormatError(description)
    elif status == 434:
        raise AvatarsMdsApiImageNotFoundError(description)
    elif status >= 500:
        raise AvatarsMdsApiTemporaryError('status=%s, description=%s' % (status, description))

    raise AvatarsMdsApiPermanentError('bad request: %s' % (description or 'reason unknown'))


def delete_error_detector(response, raw_response):
    status = raw_response.status_code
    if status not in [200, 202, 404]:
        raise AvatarsMdsApiTemporaryError('status=%s' % status)


class AvatarsMdsApi(BaseBuilder, JsonParserMixin):
    base_error_class = BaseAvatarsMdsApiError
    temporary_error_class = AvatarsMdsApiTemporaryError
    parser_error_class = AvatarsMdsApiPermanentError

    def __init__(self, namespace, read_url=None, write_url=None, timeout=None, retries=None):
        super(AvatarsMdsApi, self).__init__(
            url=write_url or settings.AVATARS_WRITE_URL,
            timeout=timeout or settings.AVATARS_TIMEOUT,
            retries=retries or settings.AVATARS_RETRIES,
            logger=log,
            graphite_logger=GraphiteLogger(service='avatars_mds_api'),
        )
        self.namespace = namespace
        self.read_url = read_url or settings.AVATARS_READ_URL

    def get_read_url(self, group_id, key, size):
        return '{read_url}/get-{namespace}/{group_id}/{key}/{size}'.format(
            read_url=self.read_url,
            namespace=self.namespace,
            group_id=group_id,
            key=key,
            size=size,
        )

    def upload_from_file(self, key, file_):
        """
        Загружает картинку в сторадж
        :return: group_id - идентификатор группы стораджа
        """
        rv = self._request_with_retries_simple(
            url_suffix='/put-{namespace}/{key}'.format(
                namespace=self.namespace,
                key=key,
            ),
            method='POST',
            files={'file': file_},
            error_detector=upload_error_detector,
            http_error_handler=None,
            parser=self.parse_json,
        )
        return rv.get('group-id') or rv['attrs']['group-id']

    def upload_from_url(self, key, url):
        """
        Скачивает картинку по урлу и загружает её в сторадж
        :return: group_id - идентификатор группы стораджа
        """
        rv = self._request_with_retries_simple(
            url_suffix='/put-{namespace}/{key}'.format(
                namespace=self.namespace,
                key=key,
            ),
            method='GET',
            params={'url': url},
            error_detector=upload_error_detector,
            http_error_handler=None,
            parser=self.parse_json,
        )
        return rv.get('group-id') or rv['attrs']['group-id']

    def download(self, url=None, group_id=None, key=None, size=None):
        if url is not None:
            # NOTE: Ответственность за валидацию url (в т.ч. что он принадлежит MDS и не может быть задан
            # пользователем полностью произвольно) лежит на вызывающем.
            # Если нужно скачать картинку извне интранета - этот метод использовать нельзя, а нужно использовать Zora.
            pass
        elif group_id is not None and key is not None and size is not None:
            url = self.get_read_url(group_id, key, size)
        else:
            raise ValueError('Not enough params')

        file_object = self._request_with_retries_simple(
            url=url,
            method='GET',
            error_detector=download_error_detector,
            http_error_handler=None,
            parser=parse_raw_file,
            stream=True,
        )
        return file_object

    def delete(self, group_id, key):
        self._request_with_retries_simple(
            url_suffix='/delete-{namespace}/{group_id}/{key}'.format(
                namespace=self.namespace,
                group_id=group_id,
                key=key,
            ),
            method='POST',
            error_detector=delete_error_detector,
            http_error_handler=None,
            parser=lambda r: None,
        )


def get_avatars_mds_api(namespace):
    return AvatarsMdsApi(namespace=namespace)
