# -*- coding: utf-8 -*-

from __future__ import absolute_import

from contextlib import contextmanager
from functools import wraps
import json
import logging

from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import (
    mask_sensitive_fields,
    trim_message,
)
from passport.backend.core.tvm import get_tvm_credentials_manager
from passport.backend.core.useragent.sync import get_useragent
from passport.backend.utils.common import (
    encode_query_mapping,
    flatten_values,
    remove_none_values,
)
from passport.backend.utils.string import (
    escape_unprintable_bytes,
    smart_bytes,
    smart_text,
)
import six
from six.moves import range
from six.moves.urllib.parse import (
    urlencode,
    urljoin,
)


log = logging.getLogger(u'passport.builders')


@six.python_2_unicode_compatible
class RequestInfo(object):
    def __init__(self, url, get_args, post_args, headers=None):
        self._url = url
        self.get_args = get_args
        self.post_args = post_args
        self.headers = headers

    def _make_url(self, args):
        url = smart_text(self._url)
        if args:
            args = remove_none_values(args)
            args = flatten_values(args)
            encoded_args = encode_query_mapping(args)
            url = '%s?%s' % (url, smart_text(urlencode(encoded_args)))
        return url

    @property
    def url(self):
        return self._make_url(self.get_args)

    def __str__(self):
        url = self._make_url(mask_sensitive_fields(self.get_args))
        url = trim_message(smart_text(url))
        result = 'url: ' + url
        if self.post_args:
            post_record = self.post_args
            if isinstance(post_record, dict):
                encoded_dict = encode_query_mapping(mask_sensitive_fields(self.post_args))
                post_record = urlencode(encoded_dict)
            result = result + ' POST args: ' + smart_text(
                trim_message(escape_unprintable_bytes(smart_bytes(post_record))),
            )
        return result


class BaseBuilder(object):
    base_error_class = None
    temporary_error_class = None

    def __init__(self, url, timeout, retries, logger, useragent=None,
                 graphite_logger=None, statbox_logger=None, ca_cert=None,
                 client_side_cert=None, client_side_key=None,
                 tvm_dst_alias=None, tvm_credentials_manager=None,
                 reconnect_on_retries=False):
        self.url = url
        self.useragent = useragent or get_useragent()
        self.timeout = timeout
        self.retries = retries
        self.reconnect_on_retries = reconnect_on_retries
        self.logger = logger
        self.graphite_logger = graphite_logger
        self.statbox_logger = statbox_logger
        self.ca_cert = ca_cert if ca_cert is not None else settings.SSL_CA_CERT
        self.client_side_cert = client_side_cert
        self.client_side_key = client_side_key

        self.tvm_dst_alias = tvm_dst_alias
        if self.tvm_dst_alias:
            self.tvm_credentials_manager = tvm_credentials_manager or get_tvm_credentials_manager()

        self.service_name = self.__class__.__name__

        if bool(client_side_cert) != bool(client_side_key):
            raise ValueError(u'Both client_side_cert and client_side_key must be set.')

    def _add_tvm_service_ticket(self, headers, dst_alias):
        headers.update({
            'X-Ya-Service-Ticket': self.tvm_credentials_manager.get_ticket_by_alias(dst_alias),
        })

    def _request_with_retries(self, method, request_info, parser,
                              error_detector=None, http_error_handler=None,
                              response_processor=None, files=None, headers=None,
                              cookies=None, **kwargs):
        """
        Выполняем запрос с повторными попытками при сетевых проблемах или временных ошибках сервиса.
        """
        if self.tvm_dst_alias:
            headers = headers or {}
            self._add_tvm_service_ticket(headers, dst_alias=self.tvm_dst_alias)

        error_message = ''
        self.logger.debug(u'%s request %s', self.service_name, request_info)
        for i in range(self.retries):
            try:
                with self._in_graphite_logger_context(
                    retries_left=self.retries - i - 1,
                ):
                    raw_response = self._request(
                        method,
                        request_info.url,
                        request_info.post_args,
                        files=files,
                        headers=headers,
                        cookies=cookies,
                        reconnect=self.reconnect_on_retries and i > 0,
                        **kwargs
                    )

                if http_error_handler is not None:
                    http_error_handler(raw_response)

                # Парсим полученную строку (JSON, XML, etc.)
                response = parser(raw_response)

                # Ищем в ответе сообщение об ошибке, кидаем исключение если нашли.
                if error_detector is not None:
                    error_detector(response, raw_response)

                # Проводим пост-обработку полученных данных, подготавливаем ответ.
                output = response_processor(response) if response_processor else response

                return output
            except (self.useragent.request_error_class, self.temporary_error_class) as e:  # we can try again
                self.logger.debug(u'%s request %s attempt failed with error: "%r"', self.service_name, request_info, e)
                error_message = smart_text(e)
            except self.base_error_class as e:  # there's no hope
                self.logger.warning(u'%s request failed for %s with error: "%r"', self.service_name, request_info, e)
                raise

        self.logger.error(u'%s request failed for %s', self.service_name, request_info)
        raise self.temporary_error_class(error_message)

    def _request_with_retries_simple(self, error_detector, parser,
                                     http_error_handler=None,
                                     method='GET', url=None, url_suffix='', params=None,
                                     data=None, json_data=None, response_processor=None, files=None, headers=None,
                                     cookies=None, **kwargs):

        url = url or self.url

        if data is not None and json_data is not None:
            raise ValueError('data and json_data can not be specified both')
        elif json_data is not None:
            data = json.dumps(json_data)
            headers = headers or {}
            headers['Content-Type'] = 'application/json'

        return self._request_with_retries(
            method=method,
            request_info=RequestInfo(urljoin(smart_text(url), url_suffix), params, data),
            parser=parser,
            error_detector=error_detector,
            http_error_handler=http_error_handler,
            headers=headers,
            response_processor=response_processor,
            files=files,
            cookies=cookies,
            **kwargs
        )

    def _request(self, method, url, data=None, files=None, headers=None,
                 cookies=None, **kwargs):
        if data is not None:
            kwargs['data'] = data

        if files is not None:
            kwargs['files'] = files

        if headers is not None:
            kwargs['headers'] = headers

        if cookies is not None:
            kwargs['cookies'] = cookies

        if self.ca_cert is not None:
            kwargs['verify'] = self.ca_cert
        if self.client_side_cert and self.client_side_key:
            kwargs['cert'] = (self.client_side_cert, self.client_side_key)

        response = self.useragent.request(
            method,
            url,
            timeout=self.timeout,
            graphite_logger=self.graphite_logger,
            statbox_logger=self.statbox_logger,
            **kwargs
        )
        return response

    @contextmanager
    def _in_graphite_logger_context(self, **params):
        if self.graphite_logger is None:
            yield
            return
        with self.graphite_logger.make_context(**params):
            yield

    def format_response(self, response):
        content = ''
        # проверка на None потому, что у requests хитрый __nonzero__
        if response is not None and response.content:
            content = trim_message(response.content.decode('utf-8'))
        status = ''
        if response is not None:
            status = response.status_code
        return u'status=%s, content=%s' % (status, content)


def parser_trivial(raw_response):
    return raw_response.content.decode('utf8')


def forward_content(func):
    @wraps(func)
    def wrapper(response):
        return func(response.content)
    return wrapper
