# -*- coding: utf-8 -*-

from functools import partial
import inspect
import itertools
import json

from mock import (
    Mock,
    patch,
)
from nose.tools import (
    eq_,
    ok_,
)
from passport.backend.core.test.test_utils import (
    assert_call_has_kwargs,
    check_all_url_params_match,
    check_data_contains_params,
    check_url_contains_params,
    check_url_equals,
    get_unicode_query_dict_from_url,
    iterdiff,
    nth_call,
    single_entrant_patch,
)
from passport.backend.core.undefined import Undefined
import six
from six import iteritems
from six.moves import (
    collections_abc,
    xmlrpc_client as xmlrpclib,
)


class FakeBuilderError(Exception):
    """
    Ошибка, говорящая нам о том, что в FakeBuilder что-то пошло не так.
    Например что-то незамоканно.
    """


def _is_exception_instance_or_type(object):
    """
    Проверить, что объект является инстансом или типом исключения.
    """
    return (isinstance(object, Exception) or inspect.isclass(object) and issubclass(object, Exception))


class FakedRequest(object):
    _undefined = object()

    def __init__(self, method, url, post_args, files, headers, cookies):
        self._method = method
        self._url = url
        self.post_args = post_args
        self._files = files
        self._headers = headers
        self._cookies = cookies

    def assert_properties_equal(self, method=_undefined, url=_undefined,
                                post_args=_undefined, files=_undefined,
                                headers=_undefined, cookies=_undefined, json_data=_undefined):
        if method is not self._undefined:
            self._compare_arg(self._method, method, u'method')
        if url is not self._undefined:
            check_url_equals(self._url, url)
        if post_args is not self._undefined:
            self._compare_arg(self.post_args, post_args, u'POST arguments')
        elif json_data is not self._undefined:
            self._compare_arg(json.loads(self.post_args), json_data, u'POST (JSON) arguments')
            self.assert_headers_contain({u'Content-Type': u'application/json'})
        if files is not self._undefined:
            self._compare_files(self._files, files)
        if headers is not self._undefined:
            self._compare_arg(self._headers, headers, u'headers')
        if cookies is not self._undefined:
            self._compare_arg(self._cookies, cookies, u'cookies')

    def url_starts_with(self, expected_url_start):
        return self._url.startswith(expected_url_start)

    def assert_url_starts_with(self, expected_url_start):
        ok_(
            self.url_starts_with(expected_url_start),
            u'"{actual}" does not start with expected "{expected}"'.format(
                actual=self._url,
                expected=expected_url_start,
            ),
        )

    def assert_query_equals(self, expected_query):
        check_all_url_params_match(self._url, expected_query)

    def assert_query_contains(self, expected_query):
        check_url_contains_params(self._url, expected_query)

    @property
    def query_params(self):
        return self.get_query_params()

    def get_query_params(self):
        return get_unicode_query_dict_from_url(self._url)

    def assert_post_data_contains(self, expected_data):
        check_data_contains_params(self.post_args, expected_data)

    def assert_post_data_equals(self, expected_data):
        eq_(self.post_args, expected_data)

    def assert_headers_contain(self, expected_headers):
        try:
            actual = set(self._headers.items())
        except AttributeError:
            actual = set(self._headers)
        try:
            expected = set(expected_headers.items())
        except AttributeError:
            expected = set(expected_headers)

        if not expected.issubset(actual):
            delta = u', '.join(map(repr, expected - actual))
            raise AssertionError(u'Expected headers not found: %s' % (delta,))

    def method_equals(self, expected_method, faker):
        actual_method = faker.parse_method_from_request(
            self._method,
            self._url,
            self.post_args,
            self._headers,
        )
        return actual_method == expected_method

    def _compare_arg(self, actual, expected, arg_desc):
        iter_eq = iterdiff(eq_)
        iter_eq(
            actual,
            expected,
            u'Expected {arg_name} {expect}, got {actual}'.format(
                expect=expected,
                actual=actual,
                arg_name=arg_desc,
            ),
        )

    def _compare_files(self, actual, expected):
        iterdiff(eq_(actual.keys(), expected.keys(), u'File args are different'))
        for arg_name in expected:
            (actual_file_name,
             actual_content,
             actual_mime_type) = self._get_file_object_properties(actual[arg_name])
            (expected_file_name,
             expected_content,
             expected_mime_type) = self._get_file_object_properties(expected[arg_name])

            if expected_file_name is not Undefined:
                eq_(actual_file_name, expected_file_name)
            if expected_mime_type is not Undefined:
                eq_(actual_mime_type, expected_mime_type)
            eq_(actual_content, expected_content)

    def _read_with_revert(self, file):
        pos = file.tell()
        content = file.read()
        file.seek(pos)
        return content

    def _get_file_object_properties(self, arg):
        if not isinstance(arg, (list, tuple)):
            arg = (arg,)

        if len(arg) == 3:
            file_name, content_or_file, mime_type = arg
        elif len(arg) == 2:
            file_name, content_or_file = arg
            mime_type = Undefined
        elif len(arg) == 1:
            content_or_file = arg[0]
            mime_type = Undefined
            file_name = Undefined
        else:
            raise ValueError(u'Unexpected value: %r' % (arg,))

        try:
            content = self._read_with_revert(content_or_file)
        except AttributeError:
            content = content_or_file
        return file_name, content, mime_type

    def __repr__(self):
        return (
            u'<{class_name} {s._method} {s._url} post_arguments={s.post_args} '
            u'headers={s._headers} cookies={s._cookies} '
            u'files={s._files}>'.format(
                class_name=self.__class__.__name__,
                s=self,
            )
        )


@single_entrant_patch
class BaseFakeBuilder(object):
    service_name_template = 'service_%s_response'
    _faked_request_class = FakedRequest

    def __init__(self, target_builder):
        self._mock = Mock()
        self._mock.request.side_effect = self._request
        self._patch = patch.object(
            target_builder,
            '_request',
            self._mock.request,
        )
        self.set_response_value_without_method = partial(self.set_response_value, None)
        self.set_response_side_effect_without_method = partial(self.set_response_side_effect, None)
        self._methods = set()

    def start(self):
        self._patch.start()

    def stop(self):
        self._patch.stop()

    def __enter__(self):
        self.start()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.stop()

    def _get_mock_name(self, method):
        return self.service_name_template % (method or '')

    def get_mock_object(self, method):
        obj = getattr(self._mock, self._get_mock_name(method))
        return obj

    def set_response_value(self, method, value, status=200, headers=None):
        headers = headers or {}
        self._methods.add(method)
        obj = self.get_mock_object(method)
        if not hasattr(value, u'content'):
            # Библиотека requests гарантирует, что content -- это байтовая строка.
            if isinstance(value, six.text_type):
                value = value.encode(u'utf-8')
            value = Mock(content=value, status_code=status, encoding=u'utf-8', headers=headers)
        obj.return_value = value
        # Если оставить установленный side_effect, то он бы имел приоритет над
        # return_value
        obj.side_effect = None

    def set_response_side_effect(self, method, side_effect):
        self._methods.add(method)
        if isinstance(side_effect, collections_abc.Iterable):
            side_effect = self._make_each_side_effect_response_or_exception(side_effect)

        obj = getattr(self._mock, self._get_mock_name(method))
        obj.side_effect = side_effect
        obj.return_value = None

    def _make_each_side_effect_response_or_exception(self, side_effect):
        # Если передано исключение или похожий на Response объект - оставляем их как есть,
        # иначе преобразуем все элементы в Mock(content=item, status_code=200)
        mocked_side_effect = []
        for item in side_effect:
            if (
                hasattr(item, 'content') or
                _is_exception_instance_or_type(item)
            ):
                mocked_side_effect.append(item)
            else:
                if isinstance(item, six.text_type):
                    item = item.encode(u'utf-8')
                mocked_side_effect.append(
                    Mock(content=item, status_code=200, encoding=u'utf-8'),
                )
        return mocked_side_effect

    def extend_response_side_effect(self, method, side_effect):
        if not isinstance(side_effect, collections_abc.Iterable):
            raise NotImplementedError()

        obj = getattr(self._mock, self._get_mock_name(method))
        old_side_effect = obj.side_effect

        if not (old_side_effect is None or isinstance(old_side_effect, collections_abc.Iterable)):
            raise NotImplementedError()

        side_effect = self._make_each_side_effect_response_or_exception(side_effect)
        self._methods.add(method)
        obj.side_effect = side_effect if old_side_effect is None else itertools.chain(old_side_effect, side_effect)
        obj.return_value = None

    @staticmethod
    def parse_method_from_request(http_method, url, data, headers=None):
        """
        Нужно вернуть имя метода или None (тогда будем брать return_value и side_effect из
        общего места, не зависящего от метода).
        """
        return

    @property
    def requests(self):
        call_list = self.request.call_args_list
        requests = []
        for call in call_list:
            requests.append(self._faked_request_class(
                call[0][0],
                call[0][1],
                call[0][2],
                call[1]['files'],
                call[1]['headers'],
                call[1]['cookies'],
            ))
        return requests

    def get_requests_by_url_prefix(self, url_prefix):
        return [request for request in self.requests if request.url_starts_with(url_prefix)]

    def get_requests_by_method(self, method):
        return [r for r in self.requests if r.method_equals(method, self)]

    def _request(self, http_method, url, data, files=None, headers=None, cookies=None, **kwargs):
        # Тут хитрая магия
        # Дано:
        # 1. у билдера метод request вызывает _request, происходит это всё в контексте вьюхи
        # 2. flask после работы закрывает файловый дескриптор
        # 3. в тестах ассерты проверяем после работы вьюхи, т.е. файл уже закрыт
        # Поэтому, пока находимся в контексте вьюхи, вычитываем файл в переменную и перематываем позицию в начало.
        # В тестах проверяем call_args_list для метода request, поэтому для него модифицируем call_args_list,
        # заменяя оригинальный файл копией, потому что оригинал уже успел закрыть flask.
        if files:
            for f, stream in iteritems(files):
                content = stream.read()
                stream.seek(0)
                self.request.call_args_list[-1][1]['files'][f] = six.BytesIO(content)

        method_name = self.parse_method_from_request(http_method, url, data, headers=headers)
        # если не находим замоканный метод, то пробрасываем до method=None
        if method_name not in self._methods:
            method_name = None
        method_mock = getattr(self._mock, self._get_mock_name(method_name))
        return method_mock(http_method, url, data, headers=headers, cookies=cookies)

    def __getattr__(self, name):
        return getattr(self._mock, name)


def assert_builder_requested(faker, times=None):
    """
    Утверждаю, что builder за которым следит faker делал запрос times раз,
    или, если times не задано, хотя бы раз.
    """
    if times is not None:
        eq_(faker.request.call_count, times)
    else:
        ok_(faker.request.called)


def assert_builder_url_contains_params(faker, params, callnum=-1):
    """
    Утверждаю, что builder за которым следит faker делал запрос с параметрами
    params.

    callnum -- порядковый номер запроса.
    """
    url = nth_call(faker.request, callnum).arg(1)
    check_url_contains_params(url, params)


def assert_builder_url_params_equal(faker, params, callnum=-1):
    """
    Утверждаю, что builder за которым следит faker делал запрос только
    с параметрами params.

    callnum -- порядковый номер запроса.
    """
    url = nth_call(faker.request, callnum).arg(1)
    check_all_url_params_match(url, params)


def assert_builder_headers_equal(faker, headers, callnum=-1):
    """
    Утверждаю, что builder за которым следит faker делал запрос с заголовками
    headers.

    callnum -- порядковый номер запроса.
    """
    assert_call_has_kwargs(nth_call(faker.request, callnum), headers=headers)


def assert_builder_cookies_equal(faker, cookies, callnum=-1):
    """
    Утверждаю, что builder за которым следит faker делал запрос с куками
    cookies.

    callnum -- порядковый номер запроса.
    """
    eq_(nth_call(faker.request, callnum).kwarg('cookies'), cookies)


def assert_builder_data_equals(faker, data, callnum=-1, exclude_fields=[]):
    """
    Утверждаю, что builder за которым следит faker делал запрос с параметрами
    передаваемыми в POST равными data.

    callnum -- порядковый номер запроса.
    """
    called_args = nth_call(faker.request, callnum).arg(2)
    for field in exclude_fields:
        called_args.pop(field)

    eq_(nth_call(faker.request, callnum).arg(2), data)


class FakedXMLRPCRequest(FakedRequest):
    def assert_xmlrpc_method_called(self, method, params):
        self.assert_properties_equal(
            method='POST',
            headers={
                'User-Agent': 'passport-xmlrpclib-wrapper',
                'Content-Type': 'text/xml',
            },
        )

        actual_params, actual_method = xmlrpclib.loads(self.post_args)

        eq_(actual_method, method)
        eq_(actual_params, params)
