# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.bbro_api.exceptions import (
    BaseBBroApiError,
    BBroApiInvalidRequestError,
    BBroApiInvalidResponseError,
    BBroApiTemporaryError,
)
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.utils.string import escape_unprintable_bytes


# Расшифровка есть в https://st.yandex-team.ru/PASSP-14923
ALLOWED_IDS = {
    '7', '8', '9',
    '10',
    '31', '32', '33', '34', '35', '36', '37', '38', '39',
    '40', '41', '42',
    '55', '56', '58', '59',
    '60', '61', '64', '65', '66', '67',
    '91', '92', '93', '94',
}
log = logging.getLogger('passport.bbro_api')


def safe_uint64_cast(x):
    try:
        r = int(x)
        if r.bit_length() <= 64:
            return r
    except:
        pass


class BBroApi(BaseBuilder, JsonParserMixin):
    base_error_class = BaseBBroApiError
    temporary_error_class = BBroApiTemporaryError
    parser_error_class = BBroApiInvalidResponseError
    accept_empty_response = True

    def __init__(self,
                 url=None, useragent=None, timeout=None, retries=None,
                 graphite_logger=None):
        timeout = timeout or settings.BBRO_API_TIMEOUT
        retries = retries or settings.BBRO_API_RETRIES
        graphite_logger = graphite_logger or GraphiteLogger(service='bbro')
        super(BBroApi, self).__init__(
            url=url or settings.BBRO_API_URL,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='bbro',
        )

    def http_error_handler(self, response):
        if response.status_code != 200:
            log.warning(
                u'Request failed with response=%s code=%s',
                trim_message(escape_unprintable_bytes(response.content)),
                response.status_code,
            )
            raise BBroApiTemporaryError()

    def parse_bbro_segments(self, segments):
        bt_counters = {}
        for segment in segments:
            if segment.get('name') == 'bt-counter' and segment.get('counter_id') in ALLOWED_IDS:
                bt_counters[segment['counter_id']] = segment['value'].split(':', 1)[1]
        return bt_counters

    def process_response(self, response):
        if not response or not response.get('data'):
            return {}
        return self.parse_bbro_segments(response['data'][0]['segment'])

    def get_bt_counters_by_yandexuid(self, yandexuid):
        if safe_uint64_cast(yandexuid) is None:
            log.warning(
                u'Invalid yandexuid has been passed: %s',
                yandexuid,
            )
            raise BBroApiInvalidRequestError()
        cookies = dict(
            yandexuid=yandexuid,
        )
        params = {
            'operation': 6,
            'client': 'auth',
        }
        return self._request_with_retries_simple(
            method='GET',
            params=params,
            http_error_handler=self.http_error_handler,
            error_detector=None,
            cookies=cookies,
            parser=self.parse_json,
            response_processor=self.process_response,
        )


def get_bbro_api(timeout=None):
    return BBroApi()  # pragma: no cover
