# -*- coding: utf-8 -*-
# Документация к API https://wiki.yandex-team.ru/Bilet/api/
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.bilet_api.exceptions import (
    BaseBiletApiError,
    BiletApiInvalidResponseError,
    BiletApiNoAuthError,
    BiletApiPermanentError,
    BiletApiTemporaryError,
)
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.core.utils.domains import get_keyspace_by_host
from passport.backend.utils.common import remove_none_values


log = logging.getLogger('passport.bilet_api')


def bilet_api_error_detector(response, raw_response):
    if raw_response.status_code != 200 or response.get('status') != 'success' or 'result' not in response:
        log.warning(
            u'BiletApi bad response (status=%s code=%s response=%s)',
            response.get('status'),
            raw_response.status_code,
            trim_message(raw_response.content.decode('utf-8')),
        )
        if response.get('status') == 'no-auth':
            raise BiletApiNoAuthError()
        raise BiletApiInvalidResponseError()


def bilet_api_http_error_handler(raw_response):
    if raw_response.status_code >= 500:
        log.warning(
            u'Request failed with response=%s code=%s',
            trim_message(raw_response.content.decode('utf-8')),
            raw_response.status_code,
        )
        raise BiletApiPermanentError()


def bilet_api_response_processor(response):
    return response['result']


class BiletApi(BaseBuilder, JsonParserMixin):
    base_error_class = BaseBiletApiError
    temporary_error_class = BiletApiTemporaryError
    parser_error_class = BiletApiInvalidResponseError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='bilet_api')
        self.api_key = settings.BILET_API_KEY
        self.client_key = settings.BILET_API_CLIENT_KEY
        super(BiletApi, self).__init__(
            url=url or settings.BILET_API_URL,
            timeout=timeout or settings.BILET_API_TIMEOUT,
            retries=retries or settings.BILET_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            **kwargs
        )

    def _build_headers(self, user_ip=None, user_agent=None, request_id=None, yandexuid=None,
                       sessionid=None, host=None, oauth_token=None):
        headers = {
            'Accept-Language': 'RU',
            'User-Agent': user_agent,
            'WWW-Authenticate': 'ApiKey %s' % self.api_key,
            'X-Tickets-Client': 'ClientKey %s' % self.client_key,
            'X-Forwarded-For': str(user_ip) if user_ip else None,
            'X-Tickets-YUID': yandexuid,
            'X-Request-Id': request_id,
        }
        headers = remove_none_values(headers)
        if sessionid:
            if not host:
                raise ValueError('Host is required when sessionid is specified')
            # TODO: перед включением sessguard для этого сервиса переделать передачу кук и хоста (PASSP-30037)
            host = get_keyspace_by_host(host)
            headers.update({
                'Authorization': 'SessionId %s' % sessionid,
                'X-Frontend-Host': host,
            })
        elif oauth_token:
            headers['Authorization'] = 'OAuth %s' % oauth_token
        return headers

    def orders(self, limit=None, offset=None, sort=None, approved_only=None,
               user_ip=None, user_agent=None, request_id=None, yandexuid=None,
               sessionid=None, host=None, oauth_token=None):
        params = dict(limit=limit, offset=offset, sort=sort)
        params = remove_none_values(params)
        if approved_only:
            params['approved_only'] = 'true'
        return self._request_with_retries_simple(
            url_suffix='v2/orders/',
            method='GET',
            params=params,
            headers=self._build_headers(
                user_ip=user_ip,
                yandexuid=yandexuid,
                sessionid=sessionid,
                host=host,
                oauth_token=oauth_token,
                user_agent=user_agent,
                request_id=request_id,
            ),
            error_detector=bilet_api_error_detector,
            parser=self.parse_json,
            http_error_handler=bilet_api_http_error_handler,
            response_processor=bilet_api_response_processor,
        )

    def order(self, order_id, user_ip=None, user_agent=None, request_id=None, yandexuid=None,
              sessionid=None, host=None, oauth_token=None):
        return self._request_with_retries_simple(
            url_suffix='v2/orders/%s/' % order_id,
            method='GET',
            headers=self._build_headers(
                user_ip=user_ip,
                yandexuid=yandexuid,
                sessionid=sessionid,
                host=host,
                oauth_token=oauth_token,
                user_agent=user_agent,
                request_id=request_id,
            ),
            error_detector=bilet_api_error_detector,
            parser=self.parse_json,
            http_error_handler=bilet_api_http_error_handler,
            response_processor=bilet_api_response_processor,
        )

    def user_points(self, user_ip=None, user_agent=None, request_id=None, yandexuid=None,
                    sessionid=None, host=None, oauth_token=None):
        return self._request_with_retries_simple(
            url_suffix='v2/user/points/',
            method='GET',
            headers=self._build_headers(
                user_ip=user_ip,
                yandexuid=yandexuid,
                sessionid=sessionid,
                host=host,
                oauth_token=oauth_token,
                user_agent=user_agent,
                request_id=request_id,
            ),
            error_detector=bilet_api_error_detector,
            parser=self.parse_json,
            http_error_handler=bilet_api_http_error_handler,
            response_processor=bilet_api_response_processor,
        )


def get_bilet_api():
    return BiletApi()  # pragma: no cover
