# -*- coding: utf-8 -*-
# Документация (не совсем актуальная) https://wiki.yandex-team.ru/Balance/Simple/xmlrpc/

import logging

from passport.backend.core.builders.billing.exceptions import (
    BaseBillingError,
    BillingInvalidResponseError,
    BillingTemporaryError,
    BillingXMLRPCFaultError,
)
from passport.backend.core.builders.xmlrpc import XMLRPCBaseBuilder
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import mask_sensitive_fields
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.core.utils.domains import get_keyspace_by_host
from passport.backend.utils.common import encode_query_mapping
from passport.backend.utils.string import smart_text


log = logging.getLogger('passport.billing')


class BillingXMLRPC(XMLRPCBaseBuilder):
    base_error_class = BaseBillingError
    temporary_error_class = BillingTemporaryError
    parser_error_class = BillingInvalidResponseError
    fault_error_class = BillingXMLRPCFaultError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None,
                 graphite_logger=None, token=None):
        timeout = timeout or settings.BILLING_XMLRPC_TIMEOUT
        retries = retries or settings.BILLING_XMLRPC_RETRIES
        graphite_logger = graphite_logger or GraphiteLogger(service='billing_xmlrpc')
        super(BillingXMLRPC, self).__init__(
            url=url or settings.BILLING_XMLRPC_URL,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
        )

        self.token = token or settings.BILLING_TOKEN
        self.handle = self.get_xmlrpc_handle()

    def get_params_description_for_method_call(self, method_name, method_params):
        _, params_dict = method_params
        encoded_dict = encode_query_mapping(mask_sensitive_fields(params_dict))
        return ', '.join('%s=%s' % (smart_text(key), smart_text(encoded_dict[key])) for key in sorted(encoded_dict.keys()))

    def list_payment_methods(self, uid, user_ip, ym_schema=None):
        params = dict(
            uid=str(uid),
            user_ip=str(user_ip),
        )
        if ym_schema:
            params.update(ym_schema=smart_text(ym_schema))
        return self.handle.BalanceSimple.ListPaymentMethods(
            self.token,
            params,
        )

    def create_binding(self, uid, return_path=None, back_url=None, timeout=None, currency=None,
                       region_id=None, lang=None, template_tag=None, domain_sfx=None):
        params = dict(
            uid=str(uid),
        )
        for name, value in zip(
            ['return_path', 'back_url', 'currency', 'lang', 'template_tag', 'domain_sfx'],
            [return_path, back_url, currency, lang, template_tag, domain_sfx],
        ):
            if value is not None:
                params[name] = smart_text(value)
        if timeout is not None:
            params.update(timeout=int(timeout))
        if region_id is not None:
            params.update(region_id=int(region_id))
        return self.handle.BalanceSimple.CreateBinding(
            self.token,
            params,
        )

    def do_binding(self, purchase_token):
        return self.handle.BalanceSimple.DoBinding(
            self.token,
            dict(purchase_token=smart_text(purchase_token)),
        )

    def check_binding(self, purchase_token):
        return self.handle.BalanceSimple.CheckBinding(
            self.token,
            dict(purchase_token=smart_text(purchase_token)),
        )

    def unbind_card(self, session_id, user_ip, host, card):
        # TODO: перед включением sessguard для этого сервиса переделать передачу кук и хоста (PASSP-30037)
        host = get_keyspace_by_host(host)
        return self.handle.BalanceSimple.UnbindCard(
            self.token,
            dict(
                session_id=smart_text(session_id),
                user_ip=str(user_ip),
                card=smart_text(card),
                host=smart_text(host),
            ),
        )

    def migrate_binding(self, source_uid, destination_uid, migration_id):
        response = self.handle.BalanceSimple.MigrateBinding(
            self.token,
            dict(
                source_uid=str(source_uid),
                destination_uid=str(destination_uid),
                migration_id=smart_text(migration_id),
            ),
        )
        if response['status'] == 'error':
            raise BaseBillingError()
