# -*- coding: utf-8 -*-
from passport.backend.core.builders.base.faker.fake_builder import (
    BaseFakeBuilder,
    FakedXMLRPCRequest,
)
from passport.backend.core.builders.billing import BillingXMLRPC
from six import iteritems
from six.moves import xmlrpc_client as xmlrpclib


XMLRPC_NAME_TO_METHOD_NAME = {
    'BalanceSimple.ListPaymentMethods': 'list_payment_methods',
    'BalanceSimple.CreateBinding': 'create_binding',
    'BalanceSimple.DoBinding': 'do_binding',
    'BalanceSimple.CheckBinding': 'check_binding',
    'BalanceSimple.UnbindCard': 'unbind_card',
    'BalanceSimple.MigrateBinding': 'migrate_binding',
}

METHOD_NAME_TO_XMLRPC_NAME = {v: k for k, v in iteritems(XMLRPC_NAME_TO_METHOD_NAME)}

METHOD_NAMES = XMLRPC_NAME_TO_METHOD_NAME.values()


class FakeBillingXMLRPC(BaseFakeBuilder):
    _faked_request_class = FakedXMLRPCRequest

    def __init__(self):
        super(FakeBillingXMLRPC, self).__init__(BillingXMLRPC)

    @staticmethod
    def parse_method_from_request(http_method, url, data, headers=None):
        _, xmlrpc_name = xmlrpclib.loads(data)
        return XMLRPC_NAME_TO_METHOD_NAME[xmlrpc_name]

    def set_response_value(self, method, response):
        if method not in METHOD_NAMES:
            raise ValueError(
                'Set response value for unknown method: {method}'.format(
                    method=method,
                ),
            )
        super(FakeBillingXMLRPC, self).set_response_value(method, response)

    def set_response_side_effect(self, method, side_effect):
        if method not in METHOD_NAMES:
            raise ValueError(
                'Set response side effect for unknown method: {method}'.format(
                    method=method,
                ),
            )
        super(FakeBillingXMLRPC, self).set_response_side_effect(
            method,
            side_effect,
        )


def serialize_billing_response(method, data):
    return xmlrpclib.dumps(
        params=(data,),
        methodname=METHOD_NAME_TO_XMLRPC_NAME[method],
        methodresponse=True,
    ).encode('utf-8')


def billing_list_payment_methods_response(serialize=True):
    data = {
        'payment_methods': {
            'card-3600': {
                'currency': 'RUB',
                'id': '3600',
                'name': 'my card4',
                'number': '5276****1319',
                'region_id': '225',
                'system': 'MasterCard',
                'type': 'card',
            },
            'phone-10': {
                'currency': 'RUB',
                'id': '10',
                'mru': 1,
                'number': '9065700968',
                'operator': 'beeline',
                'payment_confirm': 'by_sms',
                'payment_timeout': 60,
                'region_id': '225',
                'type': 'phone'
            },
            'yandex_money-41003321876593': {
                'currency': 'RUB',
                'id': '41003321876593',
                'number': '41003321876593',
                'region_id': '225',
                'type': 'yandex_money',
            },
        },
        'status': 'success',
    }
    if serialize:
        data = serialize_billing_response('list_payment_methods', data)
    return data


TEST_PURCHASE_TOKEN = '46cd61d5b33ad31732c8180e83be2dda'


def billing_create_binding_response(serialize=True):
    data = {
        'purchase_token': TEST_PURCHASE_TOKEN,
        'status': 'success',
    }
    if serialize:
        data = serialize_billing_response('create_binding', data)
    return data


def billing_do_binding_response(serialize=True):
    data = {
        'binding_form': {
            '_TARGET': 'https://tmongo1f.yandex.ru/web/binding',
            'purchase_token': TEST_PURCHASE_TOKEN,
        },
        'status': 'success',
    }
    if serialize:
        data = serialize_billing_response('do_binding', data)
    return data


def billing_check_binding_response(serialize=True):
    data = {
        'purchase_token': TEST_PURCHASE_TOKEN,
        'start_ts': '2016-08-18T15:09:42+03:00',
        'status': 'in_progress',
        'status_desc': 'in progress',
        'timeout': '1200',
        'trust_payment_id': '57b5a57b795be20bde650388',
    }
    if serialize:
        data = serialize_billing_response('check_binding', data)
    return data


def billing_unbind_card_response(serialize=True):
    data = {
        'status': 'success',
    }
    if serialize:
        data = serialize_billing_response('unbind_card', data)
    return data


def billing_migrate_binding_response(serialize=True):
    data = {
        'status': 'success',
    }
    if serialize:
        data = serialize_billing_response('migrate_binding', data)
    return data


def billing_error_response(method, status_code='error_code', serialize=True):
    data = {
        'status': 'error',
        'status_code': status_code,
        'status_desc': 'who cares?',
    }
    if serialize:
        data = serialize_billing_response(method, data)
    return data
