# -*- coding: utf-8 -*-
from functools import partial
import logging
from operator import itemgetter

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.blackbox.exceptions import (
    BaseBlackboxError,
    BlackboxInvalidResponseError,
    BlackboxTemporaryError,
)
from passport.backend.core.builders.blackbox.parsers import (
    errors_in_blackbox_response,
    parse_blackbox_check_device_signature_response,
    parse_blackbox_check_rfc_totp_response,
    parse_blackbox_create_oauth_token_response,
    parse_blackbox_create_pwd_hash_response,
    parse_blackbox_family_info_response,
    parse_blackbox_generate_public_id_response,
    parse_blackbox_get_all_tracks_response,
    parse_blackbox_get_oauth_tokens_response,
    parse_blackbox_get_recovery_keys_response,
    parse_blackbox_login_v1_response,
    parse_blackbox_login_v2_response,
    parse_blackbox_loginoccupation_response,
    parse_blackbox_lrandoms_response,
    parse_blackbox_oauth_response,
    parse_blackbox_phone_bindings_response,
    parse_blackbox_phone_operations_response,
    parse_blackbox_pwdhistory_response,
    parse_blackbox_sessionid_response,
    parse_blackbox_test_pwd_hashes_response,
    parse_blackbox_userinfo_response,
    parse_blackbox_webauthn_credentials_response,
    parse_deletion_operations_response,
    parse_find_pdd_accounts_response,
    parse_get_single_email_response,
    parse_yakey_backup_response,
)
from passport.backend.core.builders.blackbox.url_builders import BlackboxRequestBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.blackbox')


def get_response_content(response):
    return response.content


class Blackbox(BaseBuilder, BlackboxRequestBuilder, JsonParserMixin):
    LRANDOMS_PATH = 'lrandoms.txt'
    base_error_class = BaseBlackboxError
    temporary_error_class = BlackboxTemporaryError
    parser_error_class = BlackboxInvalidResponseError

    def __init__(self, blackbox=None, useragent=None, timeout=None, retries=None, graphite_logger=None,
                 use_tvm=True, tvm_dst_alias='blackbox', **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='blackbox')
        super(Blackbox, self).__init__(
            url=blackbox or settings.BLACKBOX_URL,
            timeout=timeout or settings.BLACKBOX_TIMEOUT,
            retries=retries or settings.BLACKBOX_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias=tvm_dst_alias if use_tvm else None,
            **kwargs
        )

    def _request_with_retries_bb(self, method, request_info, response_processor=None,
                                 parser=None, error_detector=errors_in_blackbox_response):
        if not parser:
            parser = self.parse_json

        return self._request_with_retries(
            method=method,
            parser=parser,
            request_info=request_info,
            error_detector=error_detector,
            response_processor=response_processor,
            headers=request_info.headers,
        )

    def userinfo(self, *args, **kwargs):
        req = self.build_userinfo_request(*args, **kwargs)
        processor = partial(parse_blackbox_userinfo_response, request=req)
        return self._request_with_retries_bb(
            'POST',
            req,
            response_processor=processor,
        )

    def pwdhistory(self, *args, **kwargs):
        request_info = self.build_pwdhistory_request(*args, **kwargs)
        return self._request_with_retries_bb(
            'POST',
            request_info,
            response_processor=parse_blackbox_pwdhistory_response,
        )

    def get_single_email(self, *args, **kwargs):
        req = self.build_get_single_email_request(*args, **kwargs)
        return self._request_with_retries_bb(
            'POST',
            req,
            response_processor=partial(parse_get_single_email_response, request=req),
        )

    def test_pwd_hashes(self, *args, **kwargs):
        request_info = self.build_test_pwd_hashes_request(*args, **kwargs)
        return self._request_with_retries_bb(
            'POST',
            request_info,
            response_processor=parse_blackbox_test_pwd_hashes_response,
        )

    def create_pwd_hash(self, *args, **kwargs):
        request_info = self.build_create_pwd_hash_request(*args, **kwargs)
        return self._request_with_retries_bb(
            'POST',
            request_info,
            response_processor=parse_blackbox_create_pwd_hash_response,
        )

    def hosted_domains(self, *args, **kwargs):
        url = self.build_hosted_domains_request(*args, **kwargs)
        return self._request_with_retries_bb(
            'GET',
            url,
        )

    def loginoccupation(self, *args, **kwargs):
        return self._request_with_retries_bb(
            'GET',
            self.build_loginoccupation_request(*args, **kwargs),
            response_processor=parse_blackbox_loginoccupation_response,
        )

    def createsession(self, *args, **kwargs):
        return self._request_with_retries_bb(
            'GET',
            self.build_createsession_request(*args, **kwargs),
        )

    def create_oauth_token(self, *args, **kwargs):
        return self._request_with_retries_bb(
            'GET',
            self.build_create_oauth_token_request(*args, **kwargs),
            response_processor=parse_blackbox_create_oauth_token_response,
        )

    def oauth(self, *args, **kwargs):
        req = self.build_oauth_request(*args, **kwargs)
        processor = partial(parse_blackbox_oauth_response, request=req)
        return self._request_with_retries_bb(
            'GET',
            req,
            response_processor=processor,
        )

    def lrandoms(self):
        url = self.build_lrandoms_request()
        return self._request_with_retries_bb(
            'GET',
            url,
            response_processor=parse_blackbox_lrandoms_response,
            error_detector=None,
            parser=get_response_content,
        )

    def login(self, *args, **kwargs):
        # Если не указано обратного, используем вторую версию метода
        version = kwargs.pop('version', settings.BLACKBOX_LOGIN_DEFAULT_VERSION)
        parser = parse_blackbox_login_v1_response if version == 1 else parse_blackbox_login_v2_response
        req = self.build_login_request(version, *args, **kwargs)
        processor = partial(parser, request=req)

        return self._request_with_retries_bb(
            'POST',
            req,
            response_processor=processor,
        )

    def edit_totp(self, *args, **kwargs):
        request_info = self.build_edit_totp_request(*args, **kwargs)
        return self._request_with_retries_bb(
            'POST',
            request_info,
        )

    def sessionid(self, *args, **kwargs):
        req = self.build_sessionid_request(*args, **kwargs)
        processor = partial(parse_blackbox_sessionid_response, request=req)
        return self._request_with_retries_bb(
            'GET',
            req,
            response_processor=processor,
        )

    def editsession(self, *args, **kwargs):
        url = self.build_editsession_request(*args, **kwargs)
        return self._request_with_retries_bb(
            'GET',
            url,
        )

    def find_pdd_accounts(self, *args, **kwargs):
        url = self.build_find_pdd_accounts_request(*args, **kwargs)
        return self._request_with_retries_bb(
            'GET',
            url,
            response_processor=parse_find_pdd_accounts_response,
        )

    def phone_bindings(self, need_current=True, need_history=True,
                       phone_numbers=None, phone_ids=None, bound_after_time=None,
                       should_ignore_binding_limit=None, need_unbound=True):
        """
        Входные параметры
            need_current
                нужны действительные привязки

            need_unbound
                нужны отвязки (по лимиту)

            need_history
                нужны когда-либо происходившие привязки

            phone_numbers
                список из номеров телефонов

            phone_ids
                список из идентификаторов телефона

            bound_after_time
                Нужны только привязки созданные после данного времени
                (unixtime).

            should_ignore_binding_limit
                Нужны привязки у которых флаг should_ignore_binding_limit равен
                данному (описание should_ignore_binding_limit смотри в ответе).

        Возвращается список привязок, где привязка это словарь с полями
            type
                тип записи, строка из множества {"history", "current", "unbound"},

            phone_number
                номер телефона -- объект PhoneNumber,

            phone_id
                идентификатор телефона или None,

            uid
                идентификатор пользователя,

            binding_time
                когда телефон был привязан (unixtime)

            should_ignore_binding_limit
                Привязка игнорируется, когда количество привязок сравнивают
                лимитом.
        """
        if not any([phone_ids, phone_numbers]):
            return []

        url = self.build_phone_bindings_request(
            need_current,
            need_history,
            need_unbound,
            phone_numbers,
            phone_ids,
            bound_after_time,
            should_ignore_binding_limit,
        )
        return self._request_with_retries_bb(
            'GET',
            url,
            response_processor=partial(
                parse_blackbox_phone_bindings_response,
                need_current=need_current,
                need_history=need_history,
                need_unbound=need_unbound,
            ),
        )

    def get_hosts(self, *args, **kwargs):
        """
        Вызывает метод get_hosts, который отдает содержимое таблицы hosts или domain_hosts
        """
        url = self.build_get_hosts_request(*args, **kwargs)
        return self._request_with_retries_bb(
            'GET',
            url,
            response_processor=itemgetter('hosts'),
        )

    def get_track(self, *args, **kwargs):
        """
        Вызывает метод get_track, который отдает содержимое записи в таблице tracks
        """
        url = self.build_get_track_request(*args, **kwargs)
        return self._request_with_retries_bb(
            'GET',
            url,
        )

    def get_all_tracks(self, uid):
        """
        Вызывает метод get_all_track, который отдает содержимое всех записей для заданного uid в таблице tracks
        """
        url = self.build_get_all_tracks_request(uid)
        return self._request_with_retries_bb(
            'GET',
            url,
            response_processor=parse_blackbox_get_all_tracks_response,
        )

    def phone_operations(self, *args, **kwargs):
        """
        Возвращает список операций время жизни которых истекло до заданного
        времени.
        """
        url = self.build_phone_operations_request(*args, **kwargs)
        return self._request_with_retries_bb(
            'GET',
            url,
            response_processor=parse_blackbox_phone_operations_response,
        )

    def yakey_backup(self, *args, **kwargs):
        """
        Возвращает бэкапы секретов Ключа для данного номера
        """
        url = self.build_yakey_backup_request(*args, **kwargs)
        return self._request_with_retries_bb(
            'GET',
            url,
            response_processor=parse_yakey_backup_response,
        )

    def deletion_operations(self, deleted_after, deleted_before, chunk_count, chunk_no):
        """
        Поиск операций удаления пользователя.
        """
        url = self.build_deletion_operations_request(
            deleted_after=deleted_after,
            deleted_before=deleted_before,
            chunk_count=chunk_count,
            chunk_no=chunk_no,
        )
        return self._request_with_retries_bb('GET', url, response_processor=parse_deletion_operations_response)

    def get_recovery_keys(self, *args, **kwargs):
        """
        Возвращает ключ для менеджера паролей Браузера
        """
        url = self.build_get_recovery_keys_request(*args, **kwargs)
        return self._request_with_retries_bb(
            'GET',
            url,
            response_processor=parse_blackbox_get_recovery_keys_response,
        )

    def check_rfc_totp(self, uid, totp):
        """
        Проверяет одноразовый пароль. В случае успеха возвращает также и время,
        для которого otp оказался валидным.
        """
        url = self.build_check_rfc_totp_request(uid=uid, totp=totp)
        return self._request_with_retries_bb(
            'GET',
            url,
            response_processor=parse_blackbox_check_rfc_totp_response,
        )

    def sign(self, value, ttl, sign_space):
        """
        :param value: urlsafe string
        :param ttl: время жизни подписи
        :param sign_space: идентификатор подписываемого пространства
        """
        url = self.build_sign_request(value, ttl, sign_space)
        return self._request_with_retries_bb(
            'GET',
            url,
        )

    def check_sign(self, signed_value, sign_space):
        url = self.build_check_sign_request(signed_value, sign_space)
        return self._request_with_retries_bb(
            'GET',
            url,
        )

    def check_device_signature(
        self,
        nonce,
        nonce_sign_space,
        device_id,
        signature,
        timestamp=None,
        public_key=None,
        version=None,
    ):
        request = self.build_check_device_signature_request(
            nonce,
            nonce_sign_space,
            device_id,
            signature,
            timestamp,
            public_key,
            version,
        )
        return self._request_with_retries_bb(
            'POST',
            request,
            response_processor=parse_blackbox_check_device_signature_response,
        )

    def get_device_public_key(self, device_id):
        request = self.build_get_device_public_key_request(device_id)
        return self._request_with_retries_bb('GET', request)

    def family_info(self, family_id, get_place=False, get_members_info=None):
        """
        Получить информацию о семье
        :param family_id: строка ("f%i" % family_numeric_id)
        :returns: None или dict с информацией о семье
        """
        url = self.build_family_info_request(
            family_id=family_id,
            get_members_info=get_members_info,
            get_place=get_place,
        )
        return self._request_with_retries_bb(
            'GET',
            url,
            response_processor=partial(
                parse_blackbox_family_info_response,
                family_id,
            ),
        )

    def generate_public_id(self, uid):
        request = self.build_generate_public_id_request(uid=uid)
        return self._request_with_retries_bb(
            'GET',
            request,
            response_processor=parse_blackbox_generate_public_id_response,
        )

    def webauthn_credentials(self, credential_external_id):
        request = self.build_webauthn_credentials_request(credential_external_id=credential_external_id)
        return self._request_with_retries_bb(
            'GET',
            request,
            response_processor=parse_blackbox_webauthn_credentials_response,
        )

    def get_oauth_tokens(
        self,
        uid,
        full_info=False,
        xtoken_only=False,
        get_is_xtoken_trusted=False,
        get_login_id=False,
        device_id=None,
        client_id_internal=None,
    ):
        request = self.build_get_oauth_tokens_request(
            uid=uid,
            full_info=full_info,
            xtoken_only=xtoken_only,
            get_is_xtoken_trusted=get_is_xtoken_trusted,
            get_login_id=get_login_id,
            device_id=device_id,
            client_id=client_id_internal,
        )
        return self._request_with_retries_bb(
            'GET',
            request,
            response_processor=parse_blackbox_get_oauth_tokens_response,
        )

    def check_ip(self, ip):
        request = self.build_check_ip_request(ip=ip)
        return self._request_with_retries_bb('GET', request)


def get_blackbox():
    return Blackbox()  # pragma: no cover
