# -*- coding: utf-8 -*-

from datetime import (
    datetime,
    timedelta,
)
from functools import partial
import json

import mock
from nose.tools import (
    assert_in,
    ok_,
)
from passport.backend.core.builders import blackbox
from passport.backend.core.builders.base.faker.fake_builder import (
    BaseFakeBuilder,
    FakedRequest,
)
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_CHECK_RFC_TOTP_VALID_STATUS,
    BLACKBOX_CHECK_SIGN_STATUS_OK,
    BLACKBOX_EDIT_TOTP_STATUS_ERROR,
    BLACKBOX_EDIT_TOTP_STATUS_OK,
    BLACKBOX_GET_DEVICE_PUBLIC_KEY_STATUS,
)
from passport.backend.core.builders.blackbox.parsers import (
    PHONE_OP_FIELDS,
    PHONE_OP_TYPE_IDX,
)
from passport.backend.core.eav_type_mapping import (
    ALIAS_NAME_TO_TYPE,
    ATTRIBUTE_NAME_TO_TYPE as AT,
    EXTENDED_ATTRIBUTES_PHONE_NAME_TO_TYPE_MAPPING,
    EXTENDED_ATTRIBUTES_WEBAUTHN_NAME_TO_TYPE_MAPPING,
    get_attr_name,
    get_attr_type,
    SID_TO_SUBSCRIPTION_ATTR,
)
from passport.backend.core.services import SERVICES
from passport.backend.core.test.serializers import serializers
from passport.backend.core.test.test_utils import get_unicode_query_dict_from_url
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_KIDDISH,
    ACCOUNT_TYPE_KINOPOISK,
    ACCOUNT_TYPE_KOLONKISH,
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_MAILISH,
    ACCOUNT_TYPE_NEOPHONISH,
    ACCOUNT_TYPE_PHONISH,
    ACCOUNT_TYPE_SOCIAL,
    ACCOUNT_TYPE_YAMBOT,
    determine_account_type_by_aliases,
)
from passport.backend.core.types.bit_vector.bit_vector import PhoneOperationFlags
from passport.backend.core.types.email.email import domain_from_email
from passport.backend.utils.common import (
    map_dict,
    merge_dicts,
)
from passport.backend.utils.string import smart_unicode
from passport.backend.utils.time import (
    datetime_to_integer_unixtime,
    datetime_to_string,
    datetime_to_unixtime,
)
from six import (
    iteritems,
    string_types,
)
from six.moves.urllib.parse import (
    parse_qs,
    urlparse,
)


_DEFAULT = object()


def phone_number_to_str(phone_number):
    if not phone_number.startswith(u'+'):
        raise ValueError(u"phone_number should be started with '+'")
    return phone_number[1:]


def datetime_to_str(dt):
    unix_time = int(datetime_to_unixtime(dt))
    return str(unix_time)


def phone_datetime_to_str(time):
    if isinstance(time, string_types):
        return time

    if time is None:
        return u'0'

    return datetime_to_str(time)


def number_to_str(val):
    if val is None:
        return u'0'
    return str(val)


def flag_to_str(val):
    return '1' if val else '0'


PHONE_ATTR_SERIALIZE = {
    u'number': phone_number_to_str,
    u'created': phone_datetime_to_str,
    u'bound': phone_datetime_to_str,
    u'confirmed': phone_datetime_to_str,
    u'admitted': phone_datetime_to_str,
    u'secured': phone_datetime_to_str,
    u'is_default': flag_to_str,
    u'is_bank': flag_to_str,
}

OP_DEFAULTS = {
    u'uid': 1,
    u'type': u'bind',
    u'started': datetime(2000, 1, 20, 0, 0, 1),
    u'finished': None,
    u'code_value': u'3232',
    u'code_checks_count': 0,
    u'code_send_count': 1,
    u'code_last_sent': datetime(2000, 1, 20, 0, 0, 1),
    u'code_confirmed': None,
    u'password_verified': None,
    u'flags': PhoneOperationFlags(0),
    u'phone_id': None,
    u'phone_id2': None,
}

OP_SERIALIZE_MAPPING = {
    u'id': number_to_str,
    u'uid': number_to_str,
    u'phone_id': number_to_str,
    u'security_identity': number_to_str,
    u'type': lambda val: str(PHONE_OP_TYPE_IDX[val] if isinstance(val, string_types) else val),
    u'started': phone_datetime_to_str,
    u'finished': phone_datetime_to_str,
    u'code_value': lambda val: val if val is not None else u'',
    u'code_checks_count': number_to_str,
    u'code_send_count': number_to_str,
    u'code_last_sent': phone_datetime_to_str,
    u'code_confirmed': phone_datetime_to_str,
    u'password_verified': phone_datetime_to_str,
    u'flags': lambda val: str(int(val)) if val is not None else u'0',
    u'phone_id2': number_to_str,
}

DEFAULT_BINDING = {
    u'type': u'history',
    u'number': u'79011111111',
    u'phone_id': u'1',
    u'uid': u'1',
    u'bound': u'12345',
    u'flags': u'0',
}


SERIALIZE_BINDING_MAPPING = {
    u'phone_id': serializers.number_to_str,
    u'uid': serializers.number_to_str,
    u'bound': phone_datetime_to_str,
    u'flags': lambda f: serializers.number_to_str(int(f)),
    u'number': phone_number_to_str,
}


SERIALIZE_WEBAUTHN_DEVICE_MAPPING = {
    'created': datetime_to_str,
}

ERROR_VALUE_TO_ID = {
    u'INVALID_PARAMS': 2,
    u'DB_EXCEPTION': 10,
}

# Подписки, которые по каким-то причинам не запрашиваются в ЧЯ
ABONDONED_SERVICES = [SERVICES[s] for s in [
    'contest',
    'dev',
    'dz',
    'galatasaray',
    'mobileproxy',
    'mpop',
    'phonenumber-alias',
    'pvo',
    'sfera',
    'tune',
    'yandsearch',
    'yastaff',
]]


class _BlackboxRequest(FakedRequest):
    def assert_contains_attributes(self, expected):
        """
        Утверждает, что множество атрибутов expected было в запросе.

        Входные параметры
            expected
                Множество имён атрибутов
        """
        types = self._get_comma_separated_args('attributes')
        actual = set(map(get_attr_name, types))
        expected = set(expected)

        ok_(expected <= actual, u'Attributes not found %r' % (expected - actual,))

    def assert_contains_dbfields(self, expected):
        """
        Утверждает, что множество fields expected было в запросе.

        Входные параметры
            expected
                Множество имён dbfields
        """
        actual = set(self._get_comma_separated_args('dbfields'))
        expected = set(expected)
        ok_(expected <= actual, u'Dbfields not found %r' % (expected - actual,))

    def assert_has_all_subscriptions(self):
        all_sids = set(s.sid for s in SERVICES.values())
        all_sids -= set(s.sid for s in ABONDONED_SERVICES)

        all_attr_sids = set()
        all_dbfield_sids = set()
        for sid in all_sids:
            if sid in SID_TO_SUBSCRIPTION_ATTR:
                all_attr_sids.add(SID_TO_SUBSCRIPTION_ATTR[sid])
            else:
                all_dbfield_sids.add(sid)

        self.assert_contains_dbfields({'subscription.suid.' + str(s) for s in all_dbfield_sids})
        self.assert_contains_attributes([get_attr_name(str(t)) for t in all_attr_sids])

    def _get_comma_separated_args(self, arg_name):
        if self._method == 'POST':
            args = {k: [v] for k, v in iteritems(self.post_args)}
        else:
            args = get_unicode_query_dict_from_url(self._url)
        assert_in(arg_name, args)
        return args[arg_name][0].split(',')


class FakeBlackbox(BaseFakeBuilder):
    service_name_template = 'blackbox_%s_response'
    _faked_request_class = _BlackboxRequest

    def __init__(self):
        super(FakeBlackbox, self).__init__(blackbox.Blackbox)
        self.set_blackbox_response_value = self.set_response_value
        self.set_blackbox_response_side_effect = self.set_response_side_effect

        lrandoms_method = 'lrandoms_%s' % blackbox.Blackbox.LRANDOMS_PATH
        self.set_blackbox_lrandoms_response_value = partial(self.set_response_value, lrandoms_method)
        self.set_blackbox_lrandoms_response_side_effect = partial(self.set_response_side_effect, lrandoms_method)

    @staticmethod
    def parse_method_from_request(http_method, url, data, headers=None):
        url = smart_unicode(url)
        parsed = urlparse(url)
        path = parsed.path.strip('/')
        if path == 'blackbox':
            query = parsed.query
            if http_method != 'POST':
                method = parse_qs(query)['method'][0]
            else:
                method = data['method']
        else:
            method = 'lrandoms_' + path
        return method


def get_parsed_blackbox_response(method_name, json_data, request=None):
    parser = getattr(blackbox, 'parse_blackbox_%s_response' % method_name, None)
    data = json.loads(json_data)
    return parser(data, request=request) if parser else data


def blackbox_userinfo(uid=1, karma=0, login='test', display_login='', domid='', domain='',
                      subscribed_to=None, unsubscribed_from=None, dbfields=None,
                      emails=None, enabled=True, domain_enabled=True,
                      display_name=None, default_avatar_key=None,
                      crypt_password=None, attributes=None, aliases=None,
                      pin_status=None, phones=None, phone_operations=None,
                      birthdate='1963-05-15', firstname='\\u0414', lastname='\\u0424',
                      gender='1', language='ru', timezone='Europe/Moscow', country='ru',
                      city=u'Москва', id=1, phone_bindings=None, email_attributes=None,
                      is_avatar_empty=False, is_display_name_empty=False, public_name=None,
                      billing_features=None, login_id=None, family_info=None, public_id=None,
                      webauthn_credentials=None):
    """
    Тестовый ответ метода userinfo ЧЯ.

    @param login - в случае ПДД, unicode-строка с доменом.
    См. https://ml.yandex-team.ru/thread/2370000000904833093/#message2370000001021652476
    @param display_login - логин пользователя, как он его ввёл при регистрации (без нормализации). Применимо
    только для портальных пользователей.

    В ответе ЧЯ, ненормализованная форма логина - в полях login, subscription.login.8. В остальных местах
    логин нормализован. Для соц. и phonish-пользователей, поле login пустое, но поле subscription.login.8 содержит
    синтетический логин.
    TODO: удалить параметр domain
    """
    subscribed_to = subscribed_to or []
    unsubscribed_from = unsubscribed_from or []

    # если алиасы явно не передали, то считаем аккаунт нормальным
    if aliases is None and uid is not None:
        aliases = {'portal': login}

    type_ = determine_account_type_by_aliases(aliases or {})
    if type_ in (
        ACCOUNT_TYPE_SOCIAL,
        ACCOUNT_TYPE_PHONISH,
        ACCOUNT_TYPE_YAMBOT,
        ACCOUNT_TYPE_KIDDISH,
        ACCOUNT_TYPE_KINOPOISK,
        ACCOUNT_TYPE_KOLONKISH,
        ACCOUNT_TYPE_NEOPHONISH,
    ):
        display_login = ''
    elif 'pdd' in (aliases or {}):
        display_login = login
        domain = domain_from_email(login).encode('idna').decode('utf-8')
    elif type_ in (ACCOUNT_TYPE_LITE, ACCOUNT_TYPE_MAILISH):
        display_login = login
    else:
        display_login = display_login or login

    retval = {
        'id': str(id),
        'uid': {},
        'login': display_login,
        'karma': {'value': karma},
        'karma_status': {'value': karma},
        'dbfields': {
            'accounts.ena.uid': '1' if enabled else '0',
            'userinfo.sex.uid': gender,
        },
        'attributes': {
            str(AT['person.firstname']): firstname,
            str(AT['person.lastname']): lastname,
            str(AT['person.birthday']): birthdate,
            str(AT['person.country']): country,
            str(AT['person.city']): city,
            str(AT['person.timezone']): timezone,
            str(AT['person.language']): language,
            str(AT['account.normalized_login']): login,
            str(AT['account.is_disabled']): '0' if enabled else '1',
            str(AT['account.is_available']): '1' if enabled else '0',
        },
    }

    # uid = None для тестов это uid не найден
    if uid is not None:
        retval['uid'] = {
            'value': uid,
            'domid': '',
            'domain': '',
        }
        if 'pdd' in (aliases or {}):
            retval['uid']['domid'] = domid or '1234'
            retval['uid']['domain'] = domain
            retval['uid']['domain_ena'] = '1' if domain_enabled else '0'

        if type_ in (ACCOUNT_TYPE_SOCIAL, ACCOUNT_TYPE_PHONISH, ACCOUNT_TYPE_NEOPHONISH):
            subscription_login_8 = login
        else:
            subscription_login_8 = display_login
        retval['dbfields'].update({
            'subscription.login_rule.8': '1',
            'subscription.suid.8': '1',
            'subscription.login.8': subscription_login_8,
        })

    for sid in subscribed_to:
        retval['dbfields']['subscription.suid.%s' % sid] = '1'
    for sid in unsubscribed_from:
        retval['dbfields']['subscription.suid.%s' % sid] = '0'

    if dbfields:
        retval['dbfields'].update(dbfields)

    # emails это список dict-ов с необходимыми полями,
    # если какие-то поля отсутствуют, то они будут выставлены по дефолту в False,
    # а дефолтным адресом будет ''.
    if emails:
        retval['address-list'] = []
        for email in emails:
            retval['address-list'].append({
                'default': email.get('default', False),
                'prohibit-restore': email.get('prohibit-restore', False),
                'validated': email.get('validated', False),
                'native': email.get('native', False),
                'rpop': email.get('rpop', False),
                'unsafe': email.get('unsafe', False),
                'silent': email.get('silent', False),
                'address': email.get('address', ''),
                'born-date': email.get('born-date', '2013-09-12 16:33:59'),
            })

    if email_attributes:
        retval['emails'] = email_attributes

    # display_name это dict с необходимыми полями, если часть из них отсутствует,
    # то они будут дефолтными
    if display_name is None:
        display_name = {}
    retval['display_name'] = {
        'name': display_name.get('name', ''),
        'display_name_empty': is_display_name_empty,
    }
    if public_name is not None:
        retval['display_name']['public_name'] = public_name
    if default_avatar_key is not None:
        retval['display_name']['avatar'] = {
            'default': default_avatar_key,
            'empty': is_avatar_empty,
        }
    if 'social' in display_name:
        retval['display_name']['social'] = {
            'provider': display_name['social'].get('provider', 'fb'),
            'profile_id': display_name['social'].get('profile_id', '1'),
        }

    attributes = attributes or {}
    if crypt_password:
        if ':' not in crypt_password:
            raise ValueError('No version in crypt_password: %s' % crypt_password)
        attributes['password.encrypted'] = crypt_password
    for attr_name, attr_value in iteritems(attributes):
        attr_type = str(get_attr_type(attr_name))
        retval['attributes'][attr_type] = attr_value

    if aliases:
        retval['aliases'] = dict([
            (
                str(ALIAS_NAME_TO_TYPE[alias_name]),
                alias,
            ) for alias_name, alias in iteritems(aliases)
        ])

    if pin_status is not None:
        retval['pin_status'] = pin_status

    if login_id is not None:
        retval['login_id'] = login_id

    if phones is not None:
        retval['phones'] = [build_blackbox_phone(phone) for phone in phones]

    if phone_operations is not None:
        _phone_operations = list()
        for phone_op in phone_operations:
            if 'uid' not in phone_op:
                phone_op = dict(phone_op, uid=uid)
            _phone_operations.append(phone_op)

        retval['phone_operations'] = build_blackbox_phone_operations(
            _phone_operations,
        )

    if phone_bindings is not None:
        retval['phone_bindings'] = build_phone_bindings(phone_bindings)

    if billing_features is not None:
        retval['billing_features'] = billing_features

    if family_info is not None:
        retval['family_info'] = family_info

    if public_id is not None:
        retval['public_id'] = public_id

    if webauthn_credentials is not None:
        retval['webauthn_credentials'] = [
            build_blackbox_webauthn_credential(cred)
            for cred in webauthn_credentials
        ]

    return retval


def build_blackbox_phone(phone):
    return {u'id': str(phone[u'id']),
            u'attributes': build_blackbox_phone_attrs(phone)}


def build_blackbox_phone_attrs(phone):
    attrs = {}
    for attr_name in phone:
        if attr_name == u'id':
            # Идентификатора телефона не должно быть в атрибутах.
            continue
        if phone[attr_name] is None:
            # ЧЯ не возвращает атрибуты без значения.
            continue
        idx = EXTENDED_ATTRIBUTES_PHONE_NAME_TO_TYPE_MAPPING[attr_name]
        serialize = PHONE_ATTR_SERIALIZE[attr_name]
        attrs[str(idx)] = serialize(phone[attr_name])
    return attrs


def build_blackbox_phone_operations(operations):
    if not operations:
        return {}

    op_ids = [op.get('id') for op in operations if op.get('id') is not None]
    if op_ids:
        max_op_id = max(op_id for op_id in op_ids)
    else:
        max_op_id = 0
    next_op_id = max_op_id + 1

    ret = {}
    for op in operations:
        op_id = op.get('id')
        if op_id is None:
            op_id = op['id'] = next_op_id
            next_op_id += 1
        ret[str(op_id)] = build_blackbox_phone_operation(op)
    return ret


def build_blackbox_phone_operation(operation):
    if u'security_identity' not in operation:
        if u'phone_number' not in operation:
            raise ValueError(u'phone_number should be defined in operation')
        operation = dict(
            operation,
            security_identity=phone_number_to_str(operation[u'phone_number']),
        )
    operation = merge_dicts(OP_DEFAULTS, operation)
    operation = map_dict(operation, OP_SERIALIZE_MAPPING)
    operation_tuple = (operation[field_name] for field_name, _ in PHONE_OP_FIELDS)
    return u','.join(operation_tuple)


def build_blackbox_webauthn_credential(credential):
    attrs = {}
    id_ = credential.pop('id', 1)
    for attr_name, attr_value in credential.items():
        if attr_value in ('', None):
            # ЧЯ не возвращает атрибуты без значения
            continue
        attr_id = EXTENDED_ATTRIBUTES_WEBAUTHN_NAME_TO_TYPE_MAPPING[attr_name]
        serialize = SERIALIZE_WEBAUTHN_DEVICE_MAPPING.get(attr_name, str)
        attrs[str(attr_id)] = serialize(attr_value)

    return {
        'id': str(id_),
        'attributes': attrs,
    }


def blackbox_userinfo_response(uid=1, uids=None, karma=0, login='test', display_login='', domid='', domain='',
                               subscribed_to=None, unsubscribed_from=None, dbfields=None, emails=None, enabled=True,
                               domain_enabled=True, display_name=None, default_avatar_key=None,
                               crypt_password=None, attributes=None,
                               aliases=None, pin_status=None, phones=None, phone_operations=None,
                               birthdate='1963-05-15', firstname='\\u0414', lastname='\\u0424', gender='1',
                               language='ru', timezone='Europe/Moscow',
                               country='ru', city=u'Москва', id=1, phone_bindings=None, email_attributes=None,
                               is_avatar_empty=False, is_display_name_empty=False, public_name=None,
                               billing_features=None, login_id=None, family_info=None, public_id=None,
                               webauthn_credentials=None):
    kwargs = dict(
        karma=karma,
        login=login,
        display_login=display_login,
        domid=domid,
        domain=domain,
        subscribed_to=subscribed_to,
        unsubscribed_from=unsubscribed_from,
        dbfields=dbfields,
        emails=emails,
        enabled=enabled,
        domain_enabled=domain_enabled,
        display_name=display_name,
        default_avatar_key=default_avatar_key,
        crypt_password=crypt_password,
        attributes=attributes,
        aliases=aliases,
        pin_status=pin_status,
        phones=phones,
        phone_operations=phone_operations,
        birthdate=birthdate,
        firstname=firstname,
        lastname=lastname,
        gender=gender,
        language=language,
        timezone=timezone,
        country=country,
        city=city,
        id=id,
        phone_bindings=phone_bindings,
        email_attributes=email_attributes,
        is_avatar_empty=is_avatar_empty,
        is_display_name_empty=is_display_name_empty,
        public_name=public_name,
        billing_features=billing_features,
        login_id=login_id,
        family_info=family_info,
        public_id=public_id,
        webauthn_credentials=webauthn_credentials,
    )
    if uids:
        retval = dict(users=[blackbox_userinfo(uid=u, **kwargs) for u in uids])
    else:
        retval = dict(users=[blackbox_userinfo(uid=uid, **kwargs)])
    return json.dumps(retval).encode('utf-8')


def blackbox_find_pdd_accounts_response(uids, total_count=None, count=None):
    if not total_count:
        total_count = len(uids)

    response = {
        'uids': [str(uid) for uid in uids],
        'total_count': str(total_count),
        'count': str(count or len(uids)),
    }
    return json.dumps(response).encode('utf-8')


def blackbox_pwdhistory_response(found=True):
    response = dict()
    if found:
        response['password_history_result'] = blackbox.BLACKBOX_PWDHISTORY_FOUND_STATUS
    else:
        response['password_history_result'] = blackbox.BLACKBOX_PWDHISTORY_NOT_FOUND_STATUS
    return json.dumps(response).encode('utf-8')


def blackbox_test_pwd_hashes_response(hashes=None):
    hashes = hashes or {'1:abcd': True, '2:cdef': False}
    response = dict(hashes=hashes)
    return json.dumps(response).encode('utf-8')


def blackbox_create_pwd_hash_response(password_hash='1:$1$ZpIv/mTU$cpgitheV.58a1A.N2KpcG.'):
    response = dict(hash=password_hash)
    return json.dumps(response).encode('utf-8')


def blackbox_hosted_domains_response(domain_admin=42, count=3,
                                     domain='domain.ru', domid=1, mx=0,
                                     master_domain=None,
                                     is_enabled=False, options_json=None,
                                     can_users_change_password=None,
                                     default_uid=0):

    if count <= 0:
        return json.dumps({
            'hosted_domains': [],
        })

    if options_json is not None:
        options = options_json
    else:
        options = {}
        if can_users_change_password is not None:
            if can_users_change_password in [True, '1']:
                options['can_users_change_password'] = '1'
            else:
                options['can_users_change_password'] = '0'

        options = json.dumps(options) if options else ''

    generated_domains = [
        {
            'born_date': '2010-10-12 15:03:24',
            'default_uid': str(default_uid),
            'admin': str(domain_admin),
            'domid': domid,
            'options': options,
            'slaves': '',
            'master_domain': master_domain or '',
            'mx': str(mx),
            'domain': domain,
            'ena': '1' if is_enabled else '0',
        },
    ]

    domain_parts = domain.split('.')
    domain = '.'.join(domain_parts[:-1])
    tld = domain_parts[-1]

    for i in range(1, count):
        generated_domains.append(
            {
                'born_date': '2010-10-12 15:03:24',
                'default_uid': str(default_uid),
                'admin': domain_admin,
                'domid': domid + i,
                'options': options,
                'slaves': '',
                'master_domain': master_domain or '',
                'mx': '0',
                'domain': '%s-%d.%s' % (domain, i, tld),
                'ena': '1' if is_enabled else '0',
            },
        )

    response = {
        'hosted_domains': generated_domains,
    }
    return json.dumps(response).encode('utf-8')


def blackbox_userinfo_response_multiple(parameter_sets):
    userinfos = []
    for parameter_set in parameter_sets:
        userinfos.append(blackbox_userinfo(**parameter_set))
    retval = dict(users=userinfos)
    return json.dumps(retval).encode('utf-8')


def blackbox_loginoccupation_response(statuses, uids=None):
    if uids is None:
        uids = {}
    response = {
        'logins': {
            login: {k: v for k, v in {'status': status, 'uid': uids.get(login)}.items() if v is not None}
            for login, status in statuses.items()
        },
    }
    return json.dumps(response).encode('utf-8')


def blackbox_createsession_response(is_pdd=False, domain='.yandex.ru',
                                    authid='123:1422501443:126', time='123', ip='84.201.166.67', host='126',
                                    default_uid=None, session_value=None, ssl_session_value=None,
                                    sessguard_value=None, sessguard_hosts=None, login_id=None):
    pdd_session_value = '2:1381494432.0.0.1130000000001038.8:1381494432362:1476061956:127.0.1.1.0.pochta:39205.4469.8977724a7611fb9c4875b992abf8b9d1'
    pdd_domain_value = 'pochta'
    if session_value is None:
        session_value = '2:session'
    if ssl_session_value is None:
        ssl_session_value = '2:sslsession'
    if sessguard_value is None:
        sessguard_value = '1.sessguard'
    response = {
        'login_id': login_id or authid,
        'authid': {
            'id': authid,
            'time': time,
            'ip': ip,
            'host': host,
        },
        'new-session': {
            'domain': pdd_domain_value if is_pdd else domain,
            'expires': 0,
            'value': pdd_session_value if is_pdd else session_value,
        },
    }

    if default_uid is not None:
        response['default_uid'] = default_uid

    response['new-sslsession'] = {
        'domain': pdd_domain_value if is_pdd else domain,
        'expires': 1370874827,
        'value': pdd_session_value if is_pdd else ssl_session_value,
        'secure': True,
    }

    if sessguard_hosts:
        response['new-sessguards'] = {
            host: _make_resigned_sessguard(host, sessguard_value)
            for host in sessguard_hosts
        }

    return json.dumps(response).encode('utf-8')


def blackbox_editsession_response(*args, **kwargs):
    return blackbox_createsession_response(*args, **kwargs)


def blackbox_editsession_delete_empty_response(authid='123:1422501443:126', time='123',
                                               ip='84.201.166.67', host='126'):
    response = {
        'default_uid': '',
        'authid': {
            'id': authid,
            'time': time,
            'ip': ip,
            'host': host,
        },
    }

    return json.dumps(response).encode('utf-8')


def blackbox_create_oauth_token_response(access_token, token_id=1):
    return json.dumps({
        'oauth_token': access_token,
        'token_id': str(token_id),
    }).encode('utf-8')


def blackbox_oauth_response(status=blackbox.BLACKBOX_OAUTH_VALID_STATUS,
                            error='OK', has_user_in_token=True, uid=1, karma=0, login='test', display_login='',
                            domid='', domain='', subscribed_to=None, unsubscribed_from=None, dbfields=None,
                            scope='', emails=None, enabled=True, default_avatar_key=None, display_name=None,
                            attributes=None, aliases=None, phones=None, phone_operations=None,
                            birthdate='1963-05-15', firstname='\\u0414', lastname='\\u0424', gender='1',
                            language='ru', timezone='Europe/Moscow', country='ru', city=u'Москва',
                            oauth_token_info=None, phone_bindings=None, crypt_password=None, is_avatar_empty=False,
                            is_display_name_empty=False, user_ticket=None, email_attributes=None, client_id='fake_clid',
                            public_name=None, login_id=None, billing_features=None, family_info=None, public_id=None,
                            webauthn_credentials=None):
    response = dict(
        error=error,
        status={
            'id': blackbox.BLACKBOX_OAUTH_STATUS_TO_CODE[status],
            'value': status,
        },
    )

    if status == blackbox.BLACKBOX_OAUTH_VALID_STATUS:
        if has_user_in_token:
            response.update(blackbox_userinfo(
                uid=uid,
                karma=karma,
                login=login,
                display_login=display_login,
                domid=domid,
                domain=domain,
                subscribed_to=subscribed_to,
                unsubscribed_from=unsubscribed_from,
                dbfields=dbfields,
                emails=emails,
                enabled=enabled,
                display_name=display_name,
                default_avatar_key=default_avatar_key,
                attributes=attributes,
                aliases=aliases,
                phones=phones,
                phone_bindings=phone_bindings,
                phone_operations=phone_operations,
                birthdate=birthdate,
                firstname=firstname,
                lastname=lastname,
                gender=gender,
                language=language,
                timezone=timezone,
                country=country,
                city=city,
                crypt_password=crypt_password,
                is_avatar_empty=is_avatar_empty,
                is_display_name_empty=is_display_name_empty,
                public_name=public_name,
                login_id=login_id,
                email_attributes=email_attributes,
                billing_features=billing_features,
                family_info=family_info,
                public_id=public_id,
                webauthn_credentials=webauthn_credentials,
            ))
        else:
            uid = None
        response['oauth'] = dict(
            oauth_token_info or {},
            uid=uid,
            scope=scope,
            client_id=client_id,
        )
        if user_ticket is not None:
            response['user_ticket'] = user_ticket
    return json.dumps(response).encode('utf-8')


def blackbox_lrandoms_response():
    return '''
        1002322;2dL9OKKqcKHbljKQI70PMaaB7R08VnEn3jo5iAI62gPeCQ5zgI5fjjczFOMRvvaQ;1376769601
        1002323;2dL9OKKqcKHbljKQI70PMaaB7R08VnEn3jo5iAI62gPeCQ5zgI5fjjczFOMRvvaQ;1376769602
        1002324;2dL9OKKqcKHbljKQI70PMaaB7R08VnEn3jo5iAI62gPeCQ5zgI5fjjczFOMRvvaQ;1376769603
        1002325;2dL9OKKqcKHbljKQI70PMaaB7R08VnEn3jo5iAI62gPeCQ5zgI5fjjczFOMRvvaQ;1376769604
        1002326;2dL9OKKqcKHbljKQI70PMaaB7R08VnEn3jo5iAI62gPeCQ5zgI5fjjczFOMRvvaQ;1376769605
        1002327;2dL9OKKqcKHbljKQI70PMaaB7R08VnEn3jo5iAI62gPeCQ5zgI5fjjczFOMRvvaQ;1376769606
    '''.strip()


def _make_resigned_cookies(host):
    # По хорошему, тут нужно использовать get_keyspace_by_host. Но он зависит от сеттингов.
    # Так что воспользуемся эвристикой - откусим от хоста домен второго уровня.
    host_parts = host.split('.')
    domain = '.'.join([''] + host_parts[-2:])
    cookies = {
        'new-session': {
            'domain': domain,
            'expires': 0,
            'secure': True,
            'value': '3:session',
        },
        'new-sslsession': {
            'domain': domain,
            'expires': 0,
            'secure': True,
            'value': '3:sslsession',
        },
    }
    return cookies


def _make_resigned_sessguard(host, value='1.sessguard'):
    return {
        'sessguard': {
            'domain': '.' + host,
            'expires': 0,
            'value': value,
        },
    }


def make_blackbox_sessionid_response(status, ttl, age, have_password, authid,
                                     time, ip, host, error, prolong_cookies, allow_more_users,
                                     resign_for_domains, sessguard_hosts, login_id=None, connection_id=None):
    response = {
        'error': error,
        'status': {
            'id': blackbox.BLACKBOX_SESSIONID_STATUS_TO_CODE[status],
            'value': status,
        },
        'authid': {
            'id': authid,
            'time': time,
            'ip': ip,
            'host': host,
        },
        'allow_more_users': allow_more_users,
    }
    if status in (
        blackbox.BLACKBOX_SESSIONID_VALID_STATUS,
        blackbox.BLACKBOX_SESSIONID_NEED_RESET_STATUS,
        blackbox.BLACKBOX_SESSIONID_DISABLED_STATUS,
    ):
        response.update({
            'ttl': ttl,
            'age': age,
        })
        if login_id is not None:
            response.update(login_id=login_id)
        if connection_id is not None:
            response.update(connection_id=connection_id)
    if status in (
        blackbox.BLACKBOX_SESSIONID_VALID_STATUS,
        blackbox.BLACKBOX_SESSIONID_NEED_RESET_STATUS,
    ):
        if prolong_cookies:
            response.update(_make_resigned_cookies('yandex.ru'))
        if resign_for_domains:
            response.update(resigned_cookies={
                domain: _make_resigned_cookies(domain)
                for domain in resign_for_domains
            })
        if sessguard_hosts is not None:  # в некоторых случаях ЧЯ возвращает пустой список
            response.update({
                'new-sessguards': {
                    host: _make_resigned_sessguard(host)
                    for host in sessguard_hosts
                }
            })

    return response


def make_blackbox_sessionid_user(
    status=blackbox.BLACKBOX_SESSIONID_VALID_STATUS,
    age=0,
    have_password=True,
    allow_plain_text=True,
    social_profile_id=None,
    uid=1,
    karma=0,
    login='test',
    display_login='',
    domid='',
    domain='',
    subscribed_to=None,
    unsubscribed_from=None,
    dbfields=None,
    emails=None,
    enabled=True,
    default_avatar_key=None,
    crypt_password=None,
    display_name=None,
    attributes=None,
    aliases=None,
    is_lite_session=False,
    phones=None,
    phone_operations=None,
    birthdate='1963-05-15',
    firstname='\\u0414',
    lastname='\\u0424',
    gender='1',
    language='ru',
    timezone='Europe/Moscow',
    country='ru',
    city=u'Москва',
    phone_bindings=None,
    email_attributes=None,
    is_avatar_empty=False,
    is_display_name_empty=False,
    public_name=None,
    billing_features=None,
    family_info=None,
    public_id=None,
    webauthn_credentials=None,
    is_scholar_session=None,
):
    user = blackbox_userinfo(
        uid,
        karma,
        login,
        display_login,
        domid,
        domain,
        subscribed_to,
        unsubscribed_from,
        dbfields,
        emails,
        enabled,
        default_avatar_key=default_avatar_key,
        email_attributes=email_attributes,
        crypt_password=crypt_password,
        display_name=display_name,
        attributes=attributes,
        aliases=aliases,
        phones=phones,
        phone_operations=phone_operations,
        phone_bindings=phone_bindings,
        birthdate=birthdate,
        firstname=firstname,
        lastname=lastname,
        gender=gender,
        language=language,
        timezone=timezone,
        country=country,
        city=city,
        is_avatar_empty=is_avatar_empty,
        is_display_name_empty=is_display_name_empty,
        public_name=public_name,
        billing_features=billing_features,
        family_info=family_info,
        public_id=public_id,
        webauthn_credentials=webauthn_credentials,
    )
    user['status'] = {
        'id': blackbox.BLACKBOX_SESSIONID_STATUS_TO_CODE[status],
        'value': status,
    }
    user['auth'] = {
        'password_verification_age': age,
        'have_password': have_password,
        'secure': True,
        'allow_plain_text': allow_plain_text,
    }
    if social_profile_id:
        user['auth']['social'] = {'profile_id': social_profile_id}
    if is_scholar_session:
        user['auth']['is_scholar_session'] = True

    if uid is None:
        del user['uid']
        del user['karma_status']
        del user['display_name']
        del user['dbfields']
        del user['karma']
    else:
        user['uid']['lite'] = is_lite_session

    return user


def blackbox_sessionid_multi_append_user(
    response,
    status=blackbox.BLACKBOX_SESSIONID_VALID_STATUS,
    age=0,
    have_password=True,
    allow_plain_text=True,
    social_profile_id=None,
    uid=1,
    karma=0,
    login='test',
    display_login='',
    domid='',
    domain='',
    subscribed_to=None,
    unsubscribed_from=None,
    item_id=1,
    dbfields=None,
    emails=None,
    enabled=True,
    default_avatar_key=None,
    crypt_password=None,
    display_name=None,
    attributes=None,
    aliases=None,
    is_lite_session=False,
    phones=None,
    phone_operations=None,
    birthdate='1963-05-15',
    firstname='\\u0414',
    lastname='\\u0424',
    gender='1',
    language='ru',
    timezone='Europe/Moscow',
    country='ru',
    city=u'Москва',
    phone_bindings=None,
    is_display_name_empty=False,
    public_name=None,
    billing_features=None,
    family_info=None,
    public_id=None,
    email_attributes=None,
    webauthn_credentials=None,
    is_scholar_session=None,
):
    response = json.loads(response)
    user = make_blackbox_sessionid_user(
        status,
        age,
        have_password,
        allow_plain_text,
        social_profile_id,
        uid,
        karma,
        login,
        display_login,
        domid,
        domain,
        subscribed_to,
        unsubscribed_from,
        dbfields,
        emails,
        enabled,
        default_avatar_key=default_avatar_key,
        crypt_password=crypt_password,
        display_name=display_name,
        attributes=attributes,
        aliases=aliases,
        is_lite_session=is_lite_session,
        phones=phones,
        phone_operations=phone_operations,
        phone_bindings=phone_bindings,
        birthdate=birthdate,
        firstname=firstname,
        lastname=lastname,
        gender=gender,
        language=language,
        timezone=timezone,
        country=country,
        city=city,
        is_display_name_empty=is_display_name_empty,
        public_name=public_name,
        billing_features=billing_features,
        family_info=family_info,
        public_id=public_id,
        email_attributes=email_attributes,
        webauthn_credentials=webauthn_credentials,
        is_scholar_session=is_scholar_session,
    )

    user['id'] = uid if uid is not None else item_id
    response['users'].append(user)
    return json.dumps(response).encode('utf-8')


def blackbox_sessionid_multi_append_invalid_session(response,
                                                    item_id=1, status=blackbox.BLACKBOX_SESSIONID_INVALID_STATUS):
    """
    Метод добавляет невалидную сессию без данных о пользователе в мультиответ. Такой ответ возвращает ЧЯ,
    если, например, аккаунт был удален.
    """
    response = json.loads(response)
    response['users'].append(
        {
            'id': item_id,
            'status': status,
        },
    )
    return json.dumps(response).encode('utf-8')


def blackbox_sessionid_response(
    status=blackbox.BLACKBOX_SESSIONID_VALID_STATUS,
    ttl=0,
    age=0,
    have_password=True,
    allow_plain_text=True,
    authid='123:1422501443:126',
    time='123',
    ip='84.201.166.67',
    host='126',
    social_profile_id=None,
    error='OK',
    uid=1,
    karma=0,
    login='test',
    display_login='',
    domid='',
    domain='',
    subscribed_to=None,
    unsubscribed_from=None,
    dbfields=None,
    emails=None,
    prolong_cookies=None,
    resign_for_domains=None,
    sessguard_hosts=None,
    enabled=True,
    default_avatar_key=None,
    userinfo_for_invalid=False,
    crypt_password=None,
    display_name=None,
    attributes=None,
    aliases=None,
    is_lite_session=False,
    phones=None,
    phone_operations=None,
    birthdate='1963-05-15',
    firstname='\\u0414',
    lastname='\\u0424',
    gender='1',
    language='ru',
    timezone='Europe/Moscow',
    country='ru',
    city=u'Москва',
    phone_bindings=None,
    is_display_name_empty=False,
    public_name=None,
    billing_features=None,
    login_id=None,
    family_info=None,
    public_id=None,
    email_attributes=None,
    webauthn_credentials=None,
    is_scholar_session=None,
):

    response = make_blackbox_sessionid_response(
        status,
        ttl,
        age,
        have_password,
        authid,
        time,
        ip,
        host,
        error=error,
        prolong_cookies=prolong_cookies,
        resign_for_domains=resign_for_domains,
        sessguard_hosts=sessguard_hosts,
        allow_more_users=True,
        login_id=login_id,
    )

    if status in [blackbox.BLACKBOX_SESSIONID_VALID_STATUS,
                  blackbox.BLACKBOX_SESSIONID_NEED_RESET_STATUS] or userinfo_for_invalid:
        response.update({
            'ttl': ttl,
            'age': age,
        })
        response.update(
            make_blackbox_sessionid_user(
                status,
                age,
                have_password,
                allow_plain_text,
                social_profile_id,
                uid,
                karma,
                login,
                display_login,
                domid,
                domain,
                subscribed_to,
                unsubscribed_from,
                dbfields,
                emails,
                enabled,
                default_avatar_key=default_avatar_key,
                crypt_password=crypt_password,
                display_name=display_name,
                attributes=attributes,
                aliases=aliases,
                is_lite_session=is_lite_session,
                phones=phones,
                phone_operations=phone_operations,
                phone_bindings=phone_bindings,
                birthdate=birthdate,
                firstname=firstname,
                lastname=lastname,
                gender=gender,
                language=language,
                timezone=timezone,
                country=country,
                city=city,
                is_display_name_empty=is_display_name_empty,
                public_name=public_name,
                billing_features=billing_features,
                family_info=family_info,
                public_id=public_id,
                email_attributes=email_attributes,
                webauthn_credentials=webauthn_credentials,
                is_scholar_session=is_scholar_session,
            ),
        )

    return json.dumps(response).encode('utf-8')


def blackbox_sessionid_multi_response(
    status=blackbox.BLACKBOX_SESSIONID_VALID_STATUS,
    ttl=0,
    age=0,
    have_password=True,
    allow_plain_text=True,
    authid='123:1422501443:126',
    time='123',
    ip='84.201.166.67',
    host='126',
    social_profile_id=None,
    error='OK',
    uid=1,
    karma=0,
    login='test',
    display_login='',
    domid='',
    domain='',
    subscribed_to=None,
    unsubscribed_from=None,
    dbfields=None,
    emails=None,
    prolong_cookies=False,
    resign_for_domains=None,
    sessguard_hosts=None,
    enabled=True,
    default_avatar_key=None,
    default_user_status=None,
    crypt_password=None,
    display_name=None,
    item_id=None,
    allow_more_users=True,
    attributes=None,
    aliases=None,
    is_lite_session=False,
    phones=None,
    phone_operations=None,
    birthdate='1963-05-15',
    firstname='\\u0414',
    lastname='\\u0424',
    gender='1',
    language='ru',
    timezone='Europe/Moscow',
    country='ru',
    city=u'Москва',
    phone_bindings=None,
    email_attributes=None,
    user_ticket=None,
    is_avatar_empty=False,
    is_display_name_empty=False,
    public_name=None,
    login_id=None,
    connection_id=None,
    billing_features=None,
    family_info=None,
    public_id=None,
    webauthn_credentials=None,
    is_scholar_session=None,
):
    item_id = item_id or uid
    response = make_blackbox_sessionid_response(
        status,
        ttl,
        age,
        have_password,
        authid,
        time,
        ip,
        host,
        error=error,
        prolong_cookies=prolong_cookies,
        resign_for_domains=resign_for_domains,
        sessguard_hosts=sessguard_hosts,
        allow_more_users=allow_more_users,
        login_id=login_id,
        connection_id=connection_id,
    )

    response['default_uid'] = item_id
    user = make_blackbox_sessionid_user(
        default_user_status if default_user_status else status,
        age,
        have_password,
        allow_plain_text,
        social_profile_id,
        uid,
        karma,
        login,
        display_login,
        domid,
        domain,
        subscribed_to,
        unsubscribed_from,
        dbfields,
        emails,
        enabled,
        attributes=attributes,
        aliases=aliases,
        crypt_password=crypt_password,
        default_avatar_key=default_avatar_key,
        display_name=display_name,
        email_attributes=email_attributes,
        is_lite_session=is_lite_session,
        phones=phones,
        phone_operations=phone_operations,
        phone_bindings=phone_bindings,
        birthdate=birthdate,
        firstname=firstname,
        lastname=lastname,
        gender=gender,
        language=language,
        timezone=timezone,
        country=country,
        city=city,
        is_avatar_empty=is_avatar_empty,
        is_display_name_empty=is_display_name_empty,
        public_name=public_name,
        billing_features=billing_features,
        family_info=family_info,
        public_id=public_id,
        webauthn_credentials=webauthn_credentials,
        is_scholar_session=is_scholar_session,
    )
    user['id'] = item_id
    response['users'] = [user]

    if user_ticket is not None:
        response['user_ticket'] = user_ticket

    return json.dumps(response).encode('utf-8')


def blackbox_login_response(
    login_status=blackbox.BLACKBOX_LOGIN_VALID_STATUS,
    password_status=blackbox.BLACKBOX_PASSWORD_VALID_STATUS,
    error='OK',
    bruteforce_policy=None,
    uid=1,
    karma=0,
    login='test',
    display_login='',
    domid='',
    domain='',
    subscribed_to=None,
    unsubscribed_from=None,
    dbfields=None,
    emails=None,
    enabled=True,
    restricted_session=False,
    domain_enabled=True,
    display_name=None,
    crypt_password=None,
    attributes=None,
    totp_check_time=None,
    aliases=None,
    phones=None,
    phone_operations=None,
    birthdate='1963-05-15',
    firstname='\\u0414',
    lastname='\\u0424',
    gender='1',
    language='ru',
    timezone='Europe/Moscow',
    country='ru',
    city=u'Москва',
    version=2,
    status=blackbox.BLACKBOX_LOGIN_V1_VALID_STATUS,
    phone_bindings=None,
    is_avatar_empty=False,
    is_display_name_empty=False,
    user_ticket=None,
    full_info=False,
    email_attributes=None,
    default_avatar_key=None,
    public_name=None,
    allowed_second_steps=None,
    family_info=None,
    public_id=None,
    webauthn_credentials=None,
    is_scholar_session=None,
):
    """
    Имитируем ответ ЧЯ по методу login
    :param status: Результат проверки логина и пароля (для version=1)
    :param login_status: Результат проверки логина (для version=2)
    :param password_status: Результат проверки пароля (для version=2)
    :param error: Код ошибки. "OK" значит "нет ошибок"
    :param bruteforce_policy: Требование показа капчи
    :param karma: Значение кармы пользователя
    :param full_info: Если установлен, в ответе будет вся указанная информация по пользователю,
    даже если `login_status` и/или `password_status` говорят о неудаче
    :return: JSON-сериализованный ответ, как если бы он пришел после запроса в ЧЯ
    """
    response = {
        'error': error,
    }
    if version == 1:
        response.update(
            status={
                'id': blackbox.BLACKBOX_LOGIN_V1_STATUS_TO_CODE[status],
                'value': status,
            },
        )
    else:
        response.update(
            login_status={
                'id': blackbox.BLACKBOX_LOGIN_STATUS_TO_CODE[login_status],
                'value': login_status,
            },
            password_status={
                'id': blackbox.BLACKBOX_PASSWORD_STATUS_TO_CODE[password_status],
                'value': password_status,
            },
        )
        # NOTE: Это поле может появляться тогда, когда пароль и верный, и неверный
        if bruteforce_policy:
            response['bruteforce_policy'] = {
                'value': bruteforce_policy,
                'level': 0,
            }

    if (
        version == 2 or
        status in [
            blackbox.BLACKBOX_LOGIN_V1_VALID_STATUS,
            blackbox.BLACKBOX_LOGIN_V1_EXPIRED_STATUS,
            blackbox.BLACKBOX_LOGIN_V1_SECOND_STEP_REQUIRED_STATUS,
        ] or
        full_info
    ):
        response.update(
            blackbox_userinfo(
                uid,
                karma,
                login,
                display_login,
                domid,
                domain,
                subscribed_to,
                unsubscribed_from,
                dbfields,
                emails,
                enabled,
                domain_enabled,
                display_name=display_name,
                crypt_password=crypt_password,
                default_avatar_key=default_avatar_key,
                attributes=attributes,
                aliases=aliases,
                phones=phones,
                phone_operations=phone_operations,
                phone_bindings=phone_bindings,
                birthdate=birthdate,
                firstname=firstname,
                lastname=lastname,
                gender=gender,
                language=language,
                timezone=timezone,
                country=country,
                city=city,
                is_avatar_empty=is_avatar_empty,
                is_display_name_empty=False,
                public_name=public_name,
                email_attributes=email_attributes,
                family_info=family_info,
                public_id=public_id,
                webauthn_credentials=webauthn_credentials,
            ),
        )

        if restricted_session:
            response['restricted_session'] = True

        # Это поле может появляться, если входили используя TOTP
        if totp_check_time:
            response['totp_check_time'] = totp_check_time

        if user_ticket is not None:
            response['user_ticket'] = user_ticket

        if allowed_second_steps is not None:
            response['allowed_second_steps'] = ','.join(allowed_second_steps)

        if is_scholar_session:
            response['is_scholar_session'] = True

    response.update(badauth_counts=blackbox_build_badauth_counts())

    return json.dumps(response).encode('utf-8')


def blackbox_build_badauth_counts():
    return {
        'login': dict(value=2, limit=10),
        'login,ip': dict(value=1, limit=5),
    }


def blackbox_phone_bindings_response(bindings):
    return json.dumps({u'phone_bindings': build_phone_bindings(bindings)}).encode('utf-8')


def build_phone_bindings(bindings):
    bindings = [map_dict(binding, SERIALIZE_BINDING_MAPPING)
                for binding in bindings]
    bindings = [merge_dicts(DEFAULT_BINDING, binding) for binding in bindings]
    return bindings


def blackbox_edit_totp_response(status=True, totp_check_time=None, encrypted_secret='encrypted_secret'):
    result = dict(error=BLACKBOX_EDIT_TOTP_STATUS_OK if status else BLACKBOX_EDIT_TOTP_STATUS_ERROR)
    if totp_check_time is not None:
        result['totp_check_time'] = totp_check_time
    if status:
        result['secret_value'] = encrypted_secret
    else:
        result['junk_secret_value'] = encrypted_secret
    return json.dumps(result).encode('utf-8')


def blackbox_get_hosts_response(**kwargs):
    if kwargs.get('db_id') is not None:
        host_data = dict(host_id='1', prio='-10', sid='2', mx='mx')
        host_data.update(kwargs)
        return json.dumps({'hosts': [host_data]}).encode('utf-8')

    # default values here
    return json.dumps({
        'hosts': [
            dict(
                host_id='1', db_id='333', sid='2', host_number='0',
                mx='mxc22.yandex.ru', host_name='', host_ip='', prio='-100',
            ),
            dict(
                host_id='17', db_id='mxt', sid='99', host_number='777',
                mx='mx10.yandex.test', host_name='mxt', host_ip='192.168.0.1', prio='99',
            ),
            dict(
                host_id='19', db_id='lol', sid='-10', host_number='0',
                mx='', host_name='teux-test', host_ip='', prio='0',
            ),
        ],
    }).encode('utf-8')


def blackbox_track_item(uid=1, track_id='bc968a106e53c3a22ea1ceef3aa5ab12', is_found=True,
                        created=1123213213, expired=1123513213, content=_DEFAULT):
    item = {
        'uid': uid,
        'track_id': track_id,
    }
    if is_found:
        item.update({
            'created': str(created),
            'expired': str(expired),
            'content': content if content is not _DEFAULT else {'type': 1},
        })
    return item


def blackbox_get_track_response(uid, track_id, is_found=True, created=1123213213, expired=1123513213, content=_DEFAULT):
    item = blackbox_track_item(uid, track_id, is_found=is_found, created=created, expired=expired, content=content)
    return json.dumps(item).encode('utf-8')


def blackbox_get_all_tracks_response(items=_DEFAULT):
    if items == _DEFAULT:
        items = [blackbox_track_item()]
    return json.dumps(dict(track=items)).encode('utf-8')


def blackbox_phone_operations_response(phone_operations):
    phone_operations = list(build_blackbox_phone_operations(phone_operations).values())
    response = {
        'phone_operations': phone_operations,
        'count': len(phone_operations),
        'total_count': len(phone_operations),
    }
    return json.dumps(response).encode('utf-8')


def blackbox_json_error_response(error_value, error_message=u'Test error message'):
    return json.dumps({
        u'exception': {
            u'id': ERROR_VALUE_TO_ID.get(error_value, -1),
            u'value': error_value,
        },
        u'error': error_message,
    }).encode('utf-8')


def blackbox_yakey_backup_response(is_found=True, phone_number=79261234567,
                                   backup='abc123456def', updated=None,
                                   device_name=None, info_only=False):
    response = {'yakey_backups': []}
    backup_data = dict(
        phone_number=phone_number,
        backup=backup,
        updated=updated or datetime_to_integer_unixtime(datetime.now()),
        device_name=device_name,
    )
    if info_only:
        backup_data.pop('backup', None)
    if is_found:
        response['yakey_backups'].append(backup_data)
    return json.dumps(response).encode('utf-8')


def blackbox_deletion_operations_response(operations=None):
    if operations is None:
        operations = [{'uid': 1}]
    response = {
        'deletion_operations': [
            {
                'uid': str(op['uid']),
            }
            for op in operations
        ],
    }
    return json.dumps(response).encode('utf-8')


def blackbox_get_recovery_keys_response(recovery_key):
    response = {
        'recovery_key': recovery_key or '',
    }
    return json.dumps(response).encode('utf-8')


def blackbox_check_rfc_totp_response(status=BLACKBOX_CHECK_RFC_TOTP_VALID_STATUS, time=100500):
    response = {
        'status': status,
    }
    if status == BLACKBOX_CHECK_RFC_TOTP_VALID_STATUS:
        response.update(time=time)
    return json.dumps(response).encode('utf-8')


def blackbox_sign_response(signed_value='123.abc'):
    response = {
        'signed_value': signed_value,
    }
    return json.dumps(response).encode('utf-8')


def blackbox_check_sign_response(value='{}', status=BLACKBOX_CHECK_SIGN_STATUS_OK):
    response = {
        'status': status,
    }
    if status == BLACKBOX_CHECK_SIGN_STATUS_OK:
        response.update(value=value)
    return json.dumps(response).encode('utf-8')


def blackbox_check_device_signature_response(status='OK', error=''):
    response = {'status': status, 'error': error}
    return json.dumps(response).encode('utf-8')


def blackbox_get_device_public_key_response(
    status='OK',
    value='key1',
    version=1,
    owner_id=1,
    error='',
):
    response = {
        'status': status,
        'error': error,
        'value': value,
        'version': str(version),
        'owner_id': str(owner_id),
    }
    if status == BLACKBOX_GET_DEVICE_PUBLIC_KEY_STATUS.PUBLIC_KEY_NOT_FOUND:
        response.update(
            {
                'value': '',
                'version': '',
                'owner_id': '',
            },
        )
    return json.dumps(response).encode('utf-8')


def blackbox_family_info_response(
    family_id='f1',
    admin_uid='70500',
    uids=None,
    places=None,
    exists=True,
    with_places=True,
    kid_uids=None,
    kid_places=None,
    kid_first_place_offset=100,
    with_members_info=False,
):
    uids = uids or []
    if places is None:
        places = range(len(uids))
    kid_uids = kid_uids or []
    if kid_places is None:
        kid_places = range(kid_first_place_offset, kid_first_place_offset + len(kid_uids))

    response = {'family': {family_id: {}}}
    if exists:
        family = response['family'][family_id]
        family['admin_uid'] = admin_uid

        adults = [{'uid': u} for u in uids]
        kids = [{'uid': u} for u in kid_uids]

        if with_places:
            for adult, place in zip(adults, places):
                adult['place'] = place
            for kid, place in zip(kids, kid_places):
                kid['place'] = place

        if with_members_info:
            for kid in kids:
                kid['is_kid'] = True

        family['users'] = adults + kids

    return json.dumps(response).encode('utf-8')


def blackbox_generate_public_id_response(public_id):
    response = {'public_id': public_id}
    return json.dumps(response).encode('utf-8')


def blackbox_webauthn_credentials_response(credential_external_id, uid):
    response = {
        credential_external_id: {
            'uid': str(uid or ''),
        },
    }
    return json.dumps(response).encode('utf-8')


def build_blackbox_oauth_token(
    status_value='VALID',
    status_id=0,
    error='OK',
    login_id=None,
    **oauth_fields
):
    for field in ['ctime', 'issue_time', 'expire_time']:
        if isinstance(oauth_fields.get(field), datetime):
            oauth_fields[field] = datetime_to_string(oauth_fields[field])
    token_data = {
        'status': {
            'value': status_value,
            'id': status_id,
        },
        'error': error,
        'oauth': oauth_fields,
    }
    if login_id:
        token_data['login_id'] = login_id

    return token_data


def blackbox_get_oauth_tokens_response(tokens):
    response = {'tokens': [build_blackbox_oauth_token(**token) for token in tokens]}
    return json.dumps(response).encode('utf-8')


def blackbox_check_ip_response(yandexip):
    return json.dumps(dict(yandexip=yandexip))


class BlackboxYasmsConfigurator(object):
    """
    Модуль для unit-тестирования Я.Смсных методов.
    """
    def __init__(self, blackbox_faker):
        self._bindings_history = []
        self._current_bindings = []
        self._phone_on_account = []
        self._next_id = 1

        blackbox_faker.set_response_side_effect(
            u'userinfo',
            self._get_user_info_mock,
        )
        blackbox_faker.set_response_side_effect(
            u'phone_bindings',
            self._get_phone_bindings_mock,
        )

    def register_phone(self, uid, phone_number, time):
        ok_(self._find_phone_on_account(uid, phone_number) is None)
        ok_(self._find_current_binding(uid, phone_number) is None)

        phone_id = self._build_id()
        phone = {
            u'id': phone_id,
            u'uid': uid,
            u'phone_number': phone_number,
            u'created': time,
            u'bound': None,
            u'confirmed': None,
            u'operation': None,
        }
        phone[u'operation'] = {
            u'type': u'bind',
            u'id': phone_id,
            u'security_identity': int(phone_number),
            u'started': time,
            u'finished': time + timedelta(days=28),
        }
        self._phone_on_account.append({
            u'uid': uid,
            u'phone': phone,
        })

    def confirm_phone(self, uid, phone_number, time):
        phone_on_account = self._find_phone_on_account(uid, phone_number)
        ok_(phone_on_account is not None)
        ok_(self._find_current_binding(uid, phone_number) is None)

        phone = phone_on_account[u'phone']
        ok_(phone[u'operation'][u'type'] == u'bind')

        phone.update({
            u'bound': time,
            u'confirmed': time,
            u'operation': None,
        })

        binding = {
            u'phone_number': phone_number,
            u'phone_id': phone[u'id'],
            u'uid': uid,
            u'bound': time,
        }
        self._bindings_history.append(binding)
        self._current_bindings.append(binding.copy())

    def delete_phone(self, uid, phone_number, time):
        phone_on_account = self._find_phone_on_account(uid, phone_number)
        ok_(phone_on_account is not None)
        self._phone_on_account.remove(phone_on_account)
        binding = self._find_current_binding(uid, phone_number)
        if binding is not None:
            self._current_bindings.remove(binding)

    def confirm_and_delete_phone(self, uid, phone_number, time_list):
        for time in time_list:
            self.register_phone(uid, phone_number, time)
            self.confirm_phone(uid, phone_number, time)
            self.delete_phone(uid, phone_number, time)

    def _build_id(self):
        id_ = self._next_id
        self._next_id += 1
        return id_

    def _find_phone_on_account(self, uid, phone_number):
        for account in self._phone_on_account:
            if (account[u'uid'] == uid and
                    account[u'phone'][u'phone_number'] == phone_number):
                retval = account
                break
        else:
            retval = None
        return retval

    def _find_current_binding(self, uid, phone_number):
        for binding in self._current_bindings:
            if (binding[u'uid'] == uid and
                    binding[u'phone_number'] == phone_number):
                retval = binding
                break
        else:
            retval = None
        return retval

    def _get_user_info_mock(self, http_method, url, data, files=None,
                            headers=None, cookies=None):
        uid = int(data[u'uid'])

        phones = []
        phone_operations = []
        for phone_on_account in self._phone_on_account:
            if phone_on_account[u'uid'] != uid:
                continue
            phone = phone_on_account[u'phone']
            phones.append({
                u'id': phone[u'id'],
                u'number': phone[u'phone_number'],
                u'created': phone[u'created'],
                u'bound': phone[u'bound'],
                u'confirmed': phone[u'confirmed'],
            })
            operation = phone[u'operation']
            if operation is not None:
                phone_operations.append({
                    u'id': operation[u'id'],
                    u'phone_id': phone[u'id'],
                    u'type': operation[u'type'],
                    u'security_identity': operation[u'security_identity'],
                    u'started': operation[u'started'],
                    u'finished': operation[u'finished'],
                })

        return mock.Mock(
            name=u'user_info_response',
            status_code=200,
            content=blackbox_userinfo_response(
                uid=uid,
                phones=phones,
                phone_operations=phone_operations,
            ),
        )

    def _get_phone_bindings_mock(self, http_method, url, data, files=None,
                                 headers=None, cookies=None):
        parsed_url = urlparse(url)
        query = parse_qs(parsed_url.query)

        phone_numbers = [pn for pn in query['numbers'][0].split(',')]
        bindings_type = query['type'][0]

        bindings = []
        if bindings_type in ['all', 'history']:
            for binding in self._bindings_history:
                if binding[u'phone_number'] not in phone_numbers:
                    continue
                bindings.append({
                    u'type': u'history',
                    u'number': binding[u'phone_number'],
                    u'phone_id': binding[u'phone_id'],
                    u'uid': binding[u'uid'],
                    u'bound': binding[u'bound'],
                })
        if bindings_type in ['all', 'current']:
            for binding in self._current_bindings:
                if binding[u'phone_number'] not in phone_numbers:
                    continue
                bindings.append({
                    u'type': u'current',
                    u'number': binding[u'phone_number'],
                    u'phone_id': binding[u'phone_id'],
                    u'uid': binding[u'uid'],
                    u'bound': binding[u'bound'],
                })

        return mock.Mock(
            name=u'phone_bindings',
            status_code=200,
            content=blackbox_phone_bindings_response(bindings),
        )


__all__ = (
    'blackbox_build_badauth_counts',
    'blackbox_check_device_signature_response',
    'blackbox_check_rfc_totp_response',
    'blackbox_check_sign_response',
    'blackbox_create_oauth_token_response',
    'blackbox_create_pwd_hash_response',
    'blackbox_createsession_response',
    'blackbox_edit_totp_response',
    'blackbox_editsession_delete_empty_response',
    'blackbox_editsession_response',
    'blackbox_family_info_response',
    'blackbox_find_pdd_accounts_response',
    'blackbox_generate_public_id_response',
    'blackbox_get_device_public_key_response',
    'blackbox_get_hosts_response',
    'blackbox_get_recovery_keys_response',
    'blackbox_get_track_response',
    'blackbox_get_oauth_tokens_response',
    'blackbox_hosted_domains_response',
    'blackbox_json_error_response',
    'blackbox_login_response',
    'blackbox_loginoccupation_response',
    'blackbox_lrandoms_response',
    'blackbox_oauth_response',
    'blackbox_phone_bindings_response',
    'blackbox_pwdhistory_response',
    'blackbox_sessionid_multi_append_invalid_session',
    'blackbox_sessionid_multi_append_user',
    'blackbox_sessionid_multi_response',
    'blackbox_sessionid_response',
    'blackbox_sign_response',
    'blackbox_test_pwd_hashes_response',
    'blackbox_userinfo_response',
    'blackbox_userinfo_response_multiple',
    'blackbox_webauthn_credentials_response',
    'BlackboxYasmsConfigurator',
    'build_blackbox_oauth_token',
    'FakeBlackbox',
    'get_parsed_blackbox_response',
)
