# -*- coding: utf-8 -*-
import logging

# Список полей, которые являются числами но передаются из ЧЯ строками - надо парсить
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_LOGIN_DISABLED_STATUS,
    BLACKBOX_LOGIN_V1_VALID_STATUS,
    BLACKBOX_LOGIN_VALID_STATUS,
    BLACKBOX_OAUTH_VALID_STATUS,
    BLACKBOX_PASSWORD_BAD_STATUS,
    BLACKBOX_PASSWORD_COMPROMISED_STATUS,
    BLACKBOX_PASSWORD_SECOND_STEP_REQUIRED_STATUS,
    BLACKBOX_PASSWORD_VALID_STATUS,
    BLACKBOX_PWDHISTORY_FOUND_STATUS,
    BLACKBOX_YAKEY_BACKUP_EXISTS_STATUS,
    BLACKBOX_YAKEY_BACKUP_NOT_FOUND_STATUS,
)
from passport.backend.core.builders.blackbox.exceptions import (
    AccessDenied,
    BaseBlackboxError,
    BlackboxInvalidDeviceSignature,
    BlackboxInvalidParamsError,
    BlackboxInvalidResponseError,
    BlackboxTemporaryError,
    BlackboxUnknownError,
)
from passport.backend.core.eav_type_mapping import (
    ATTRIBUTE_NAME_TO_TYPE as AT,
    EXTENDED_ATTRIBUTES_EMAIL_TYPE_TO_NAME_MAPPING,
    EXTENDED_ATTRIBUTES_PHONE_TYPE_TO_NAME_MAPPING,
    EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE_TO_NAME_MAPPING,
    get_attr_name,
    SUBSCRIPTION_ATTR_TO_SID,
)
from passport.backend.core.types.bit_vector.bit_vector import (
    PhoneBindingsFlags,
    PhoneOperationFlags,
)
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.utils.time import parse_datetime
from six import (
    iteritems,
    iterkeys,
    itervalues,
)
from six.moves import map


# Список полей, которые являются числами но передаются из ЧЯ строками - надо парсить
BLACKBOX_DBFIELDS_INTEGER_FIELDNAMES = {
    'userinfo.sex.uid',
    'accounts.ena.uid',
    'password_quality.quality.uid',
    'password_quality.version.uid',
}

# Таблица преобразования имён атрибутов
BLACKBOX_ATTRIBUTE_NAMES_TO_KEY = {
    'account.browser_key': 'browser_key',
    'account.default_email': 'default_email',
    'account.enable_app_password': 'enable_app_password',
    'account.global_logout_datetime': 'global_logout_datetime',
    'account.is_employee': 'is_employee',
    'account.is_maillist': 'is_maillist',
    'account.is_shared': 'is_shared',
    'account.is_disabled': 'is_disabled',
    'account.auth_email_datetime': 'auth_email_datetime',
    'account.failed_auth_challenge_checks_counter': 'failed_auth_challenge_checks_counter',
    'account.have_organization_name': 'have_organization_name',

    'account.2fa_on': '2fa_on',
    'account.audience_on': 'audience_on',
    'account.totp.secret': 'totp.secret',
    'account.totp.failed_pin_checks_count': 'totp.failed_pin_checks_count',
    'account.totp.update_datetime': 'totp.update_datetime',
    'account.totp.secret_ids': 'totp.secret_ids',
    'account.totp.pin_length': 'totp.pin_length',
    'account.show_2fa_promo': 'show_2fa_promo',
}

BLACKBOX_ATTRIBUTES_DEFAULT_VALUES = {
    AT['person.country']: '',
    AT['person.city']: '',
    AT['person.timezone']: '',
    AT['person.language']: '',
    AT['person.firstname']: '',
    AT['person.firstname_global']: '',
    AT['person.lastname']: '',
    AT['person.lastname_global']: '',
    AT['person.birthday']: '',
    AT['account.normalized_login']: '',
    AT['account.is_disabled']: '0',
    AT['account.global_logout_datetime']: '1',
    AT['revoker.tokens']: '1',
    AT['revoker.app_passwords']: '1',
    AT['revoker.web_sessions']: '1',
}

BLACKBOX_ATTRIBUTES_INTEGER_FIELDNAMES = {
    'account.global_logout_datetime',
    'account.is_disabled',
    'account.totp.failed_pin_checks_count',
    'account.totp.update_datetime',
    'account.totp.pin_length',
    'account.additional_data_ask_next_datetime',
}

PHONE_OP_TYPE_IDX = {
    u'bind': 1,
    u'remove': 2,
    u'securify': 3,
    u'replace': 4,
    u'mark': 5,
    u'aliasify': 6,
    u'dealiasify': 7,
}

PHONE_OP_TYPE_NAME = {PHONE_OP_TYPE_IDX[name]: name for name in PHONE_OP_TYPE_IDX}


def parse_phone_operation_unixtime(unixtime):
    unixtime = int(unixtime)
    return unixtime or None


PHONE_OP_FIELDS = (
    (u'id', int),
    (u'uid', int),
    (u'phone_id', lambda id_: int(id_) or None),
    (u'security_identity', int),
    (u'type', lambda type_: PHONE_OP_TYPE_NAME[int(type_)]),
    (u'started', parse_phone_operation_unixtime),
    (u'finished', parse_phone_operation_unixtime),
    (u'code_value', lambda val_: val_ or None),
    (u'code_checks_count', int),
    (u'code_send_count', int),
    (u'code_last_sent', parse_phone_operation_unixtime),
    (u'code_confirmed', parse_phone_operation_unixtime),
    (u'password_verified', parse_phone_operation_unixtime),
    (u'flags', int),
    (u'phone_id2', lambda id_: int(id_) or None),
)

PHONE_OP_DEFAULT_VALUES = {
    u'code_checks_count': 0,
    u'code_send_count': 0,
    u'flags': PhoneOperationFlags,
}

log = logging.getLogger('passport.blackbox.parsers')


def errors_in_blackbox_response(response, raw_response):
    failures = [x for x in response.get('users', []) + [response] if 'exception' in x]

    if failures:
        value, error = failures[0]['exception']['value'], failures[0]['error']
    else:
        return False

    if value == 'ACCESS_DENIED':
        raise AccessDenied(error)
    elif value in ['DB_FETCHFAILED', 'DB_EXCEPTION']:
        raise BlackboxTemporaryError(error)
    elif value == 'INVALID_PARAMS':
        raise BlackboxInvalidParamsError(error)
    elif value == 'UNKNOWN':
        raise BlackboxUnknownError(error)
    else:
        raise BaseBlackboxError(error)


def parse_blackbox_userinfo_response(response, request=None):
    data = [
        parse_blackbox_user(item, request)
        for item in response['users']
    ]

    if len(data) == 1:
        data = data[0]

    return data


def parse_get_single_email_response(response, request=None):
    if 'users' not in response:
        raise BlackboxInvalidResponseError('Wrong userinfo response format')

    data = [
        parse_blackbox_user(item, request)
        for item in response['users']
    ]

    if len(data) > 1:
        raise BlackboxInvalidResponseError('More than one user in emails=testone userinfo response')

    data = data[0]
    addresses = data.get('address-list')

    if not addresses:
        return data['uid'], None
    elif len(addresses) > 1:
        raise BlackboxInvalidResponseError('More than one email returned')

    return data['uid'], addresses[0]


def parse_blackbox_pwdhistory_response(response):
    return {
        'found_in_history': response['password_history_result'] == BLACKBOX_PWDHISTORY_FOUND_STATUS,
    }


def parse_blackbox_test_pwd_hashes_response(response):
    return response['hashes']


def parse_blackbox_create_pwd_hash_response(response):
    return response['hash']


def parse_blackbox_user(item, request):
    response = dict(item)

    response.update(
        item['uid'],
        uid=int(item['uid']['value']) if 'value' in item['uid'] else None,
        domid=int(item['uid']['domid']) if item['uid'].get('domid') else None,
        domain=item['uid'].get('domain') or None,
        display_login=item['login'] if 'login' in item else None,
    )

    karma_status = response.pop('karma_status', {})
    response.update(karma=int(karma_status['value']) if 'value' in karma_status else None)

    if 'dbfields' in item:
        subscriptions = {}
        for key in list(iterkeys(item['dbfields'])):
            # TODO: отрефакторить на if с двумя ветками: специальные поля и подписки

            table, field, sid = key.split('.')
            if key in BLACKBOX_DBFIELDS_INTEGER_FIELDNAMES:
                item['dbfields'][key] = int(item['dbfields'][key]) if item['dbfields'][key] else None

            if table != 'subscription':
                continue
            sid = int(sid)
            if sid not in subscriptions:
                subscriptions[sid] = {'sid': sid}
            value = item['dbfields'][key]
            if field not in ['login', 'born_date']:
                value = int(value) if value else None
            subscriptions[sid][field] = value

            del item['dbfields'][key]

        if subscriptions:
            response['subscriptions'] = dict((key, val) for key, val in iteritems(subscriptions) if val['suid'])

        response.update(item['dbfields'])

    # Проставляем значения по умолчанию для атрибутов, которые были запрошены у
    # ЧЯ, но не получены от него.
    if request:
        request_args = request.post_args or request.get_args
        if 'attributes' in request_args:
            item.setdefault('attributes', {})
            requested_attr_types = request_args['attributes'].split(',')
            for attr_type_str in requested_attr_types:
                attr_type_num = int(attr_type_str)
                if attr_type_str not in item['attributes'] and attr_type_num in BLACKBOX_ATTRIBUTES_DEFAULT_VALUES:
                    item['attributes'][attr_type_str] = BLACKBOX_ATTRIBUTES_DEFAULT_VALUES[attr_type_num]

    # Разбираем приехавшие от ЧЯ атрибуты
    if 'attributes' in item:
        for attr_type in list(iterkeys(item['attributes'])):
            try:
                attr_name = get_attr_name(attr_type)
            except KeyError:
                log.error('Blackbox provided unknown attribute type: %s = %s' % (attr_type, value))
                continue

            response_key = BLACKBOX_ATTRIBUTE_NAMES_TO_KEY.get(attr_name, attr_name)

            value = item['attributes'][attr_type]
            if attr_name in BLACKBOX_ATTRIBUTES_INTEGER_FIELDNAMES:
                value = int(value) if value else None

            if attr_name == 'account.normalized_login':
                response.update(login=value)

            response[response_key] = value

        # Хак для преобразования атрибутов в подписки
        response.setdefault('subscriptions', {})
        for attr_type, value in iteritems(item['attributes']):
            service_sid = SUBSCRIPTION_ATTR_TO_SID.get(int(attr_type))
            if service_sid:
                response['subscriptions'][service_sid] = {'sid': service_sid}
        if not response['subscriptions']:
            del response['subscriptions']

    if 'emails' in response:
        response['emails'] = [
            parse_email(email)
            for email in response['emails']
        ]

    if u'phones' in response:
        response[u'phones'] = {int(p[u'id']): parse_phone(p) for p in response[u'phones']}

    if u'phone_operations' in response:
        unparsed_ops = response[u'phone_operations']
        parsed_ops = {}
        for op_id in unparsed_ops:
            op_id_str = op_id
            op_id_int = int(op_id)
            parsed_ops[op_id_int] = {u'id': op_id_int}
            parsed_op = parse_phone_operation(unparsed_ops[op_id_str])
            parsed_ops[op_id_int].update(parsed_op)
        response[u'phone_operations'] = parsed_ops

    # Прицепим операции к телефонам
    if u'phones' in response and u'phone_operations' in response:
        # Когда не запрашивали операции, у телефона нет ключа operation,
        # а когда запрашивали, такой ключ есть и его значение None (нет
        # операции) или операционный словарь.
        for phone in list(itervalues(response[u'phones'])):
            phone[u'operation'] = None
        for op in list(itervalues(response[u'phone_operations'])):
            primary_phone = response[u'phones'].get(op[u'phone_id'])
            if op[u'phone_id2'] is not None:
                secondary_phone = response[u'phones'].get(op[u'phone_id2'])
            else:
                secondary_phone = primary_phone
            if primary_phone is not None and secondary_phone is not None:
                # Связываем операцию с телефоном, только когда все телефоны
                # объявленные в этой операции есть в ответе, другие операции
                # считаем битыми.
                primary_phone[u'operation'] = op
            else:
                log.debug(
                    'Orphaned phone operation found: uid=%s, phone_operation_id=%s' % (
                        op.get('uid'),
                        op.get('id'),
                    ),
                )

    if u'phone_bindings' in response:
        response[u'phone_bindings'] = parse_phone_bindings(
            response[u'phone_bindings'],
            need_current=True,
            need_unbound=True,
            need_history=False,
        )

    # Прицепим связки к телефонам
    if u'phones' in response and u'phone_bindings' in response:
        for phone in list(itervalues(response[u'phones'])):
            phone[u'binding'] = None
        for binding in response[u'phone_bindings']:
            phone = response[u'phones'].get(binding[u'phone_id'])
            if phone is not None:
                phone[u'binding'] = binding

    if 'webauthn_credentials' in response:
        response['webauthn_credentials'] = [
            parse_webauthn_credential(webauthn_credential)
            for webauthn_credential in response['webauthn_credentials']
        ]

    response.update(parse_account_deletion_operation(response))

    return response


def try_parse_blackbox_user(item, request):
    if 'uid' in item:
        return parse_blackbox_user(item, request)
    else:
        # Специально для случая двуногого Oauth-токена
        return dict(item, uid=None)


def parse_phone_operation(op_str):
    idx_op_attr = [(idx, val) for idx, val in enumerate(op_str.split(u','))]
    operation = {}
    for idx, op_attr in idx_op_attr:
        field_name, parse = PHONE_OP_FIELDS[idx]
        operation[field_name] = parse(op_attr)
    return operation


def parse_phone(phone):
    attributes = {
        EXTENDED_ATTRIBUTES_PHONE_TYPE_TO_NAME_MAPPING[int(_id)]: phone[u'attributes'][_id]
        for _id in phone[u'attributes']
    }
    if u'number' in attributes:
        # модель Phone ждёт что номер будет с +
        attributes[u'number'] = u'+' + attributes[u'number']

    time_attr_types = (
        u'created',
        u'bound',
        u'confirmed',
        u'admitted',
        u'secured',
    )
    for time_attr_type in time_attr_types:
        if time_attr_type in attributes:
            attributes[time_attr_type] = int(attributes[time_attr_type])

    bool_attr_types = (
        u'is_default',
        u'is_bank',
    )
    for bool_attr_type in bool_attr_types:
        if bool_attr_type in attributes:
            attributes[bool_attr_type] = bool(int(attributes[bool_attr_type]))

    return {
        u'id': int(phone[u'id']),
        u'attributes': attributes,
    }


def parse_email(email):
    name_translation = {
        'is_rpop': 'rpop',
        'is_unsafe': 'unsafe',
        'is_silent': 'silent',
    }
    attributes = {}

    for _id, value in iteritems(email['attributes']):
        attribute_name = EXTENDED_ATTRIBUTES_EMAIL_TYPE_TO_NAME_MAPPING.get(int(_id))
        attributes[name_translation.get(attribute_name, attribute_name)] = value

    attributes['id'] = email['id']
    return attributes


def parse_webauthn_credential(webauthn_credential):
    attributes = {
        EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE_TO_NAME_MAPPING[int(id_)]: webauthn_credential[u'attributes'][id_]
        for id_ in webauthn_credential[u'attributes']
    }

    int_attr_types = (
        'sign_count',
        'created',
    )
    for time_attr_type in int_attr_types:
        if time_attr_type in attributes:
            attributes[time_attr_type] = int(attributes[time_attr_type])

    attributes['id'] = webauthn_credential['id']
    return attributes


def parse_blackbox_loginoccupation_response(response):
    return response.get('logins', {})


def parse_blackbox_create_oauth_token_response(response):
    return {
        'access_token': response['oauth_token'],
        'token_id': int(response['token_id']),
    }


def parse_blackbox_oauth_response(response, request=None):
    status = response['status']['value']
    response.update(status=status)

    if status != BLACKBOX_OAUTH_VALID_STATUS:
        return response

    response = try_parse_blackbox_user(response, request)
    oauth = response['oauth']
    uid = oauth['uid']
    oauth.update(
        uid=int(uid) if uid is not None else uid,
        scope=oauth['scope'].split(' '),
    )

    return response


def parse_blackbox_lrandoms_response(response):
    lines = response.decode('utf-8').split('\n')
    randoms = []
    for line in lines:
        line = line.strip()
        if not line:
            continue
        items = [x.strip() for x in line.split(';')]
        if len(items) < 3:
            raise BlackboxInvalidResponseError('Wrong lrandoms format')
        try:
            randoms.append({
                'id': items[0],
                'body': items[1],
                'created_timestamp': int(items[2]),
            })
        except ValueError as ex:
            log.error('Error parsing lrandoms: %s', ex)
            raise BlackboxInvalidResponseError('Wrong lrandoms format')
    return randoms


def can_parse_login_v1_response(response):
    """
    Отвечает на вопрос:
    Есть ли в ответе ЧЯ данные об аккаунте - можно/нужно ли парсить ответ ЧЯ?
    Предназначена для первой версии метода.
    """
    return (
        response['status'] in [BLACKBOX_LOGIN_V1_VALID_STATUS, BLACKBOX_PASSWORD_SECOND_STEP_REQUIRED_STATUS] and
        'uid' in response
    )


def parse_blackbox_login_v1_response(response, request=None):
    response.update(
        status=response['status']['value'],
    )

    # Не будем разбирать ответ ЧЯ если там нет полезной информации
    if not can_parse_login_v1_response(response):
        return response

    if response['status'] == BLACKBOX_PASSWORD_SECOND_STEP_REQUIRED_STATUS and 'allowed_second_steps' in response:
        response['allowed_second_steps'] = response['allowed_second_steps'].split(',')

    response = parse_blackbox_user(response, request)
    return response


def can_parse_login_v2_response(response):
    """
    Отвечает на вопрос:
    Есть ли в ответе ЧЯ данные об аккаунте - можно/нужно ли парсить ответ ЧЯ?
    Предназначена для второй версии метода.
    """
    return (
        response['login_status'] in [
            BLACKBOX_LOGIN_VALID_STATUS,
            BLACKBOX_LOGIN_DISABLED_STATUS,
        ] and
        response['password_status'] in [
            BLACKBOX_PASSWORD_VALID_STATUS,
            BLACKBOX_PASSWORD_BAD_STATUS,
            BLACKBOX_PASSWORD_COMPROMISED_STATUS,
            BLACKBOX_PASSWORD_SECOND_STEP_REQUIRED_STATUS,
        ] and
        'uid' in response  # У аккаунта должен быть хотя бы uid
    )


def parse_blackbox_login_v2_response(response, request=None):
    login_status = response['login_status']['value']
    password_status = response['password_status']['value']
    bruteforce_status = response.get('bruteforce_policy', {}).get('value')
    response.update(
        login_status=login_status,
        password_status=password_status,
        bruteforce_status=bruteforce_status,
    )

    # Не будем разбирать ответ ЧЯ если там нет полезной информации
    if not can_parse_login_v2_response(response):
        return response

    if response['password_status'] == BLACKBOX_PASSWORD_SECOND_STEP_REQUIRED_STATUS and 'allowed_second_steps' in response:
        response['allowed_second_steps'] = response['allowed_second_steps'].split(',')

    response = parse_blackbox_user(response, request)
    return response


def parse_user(user_response, request=None):
    """
    Парсит информацию о пользователе из ответа метода sessionid ЧЯ
    :user_response: Либо ответ метода ЧЯ sessionid(multisession=no),
    либо элемент коллекции users ответа метода ЧЯ sessionid(multisession=yes)
    """
    if isinstance(user_response['status'], dict):
        status = user_response['status']['value']
        user_response.update(status=status)
    # Идентификатор элемента коллекции пользователей.
    # Присутствует всегда, в отличае от уида. Если уид есть, то равен ему.
    item_id = int(user_response['id']) if 'id' in user_response else None
    auth = user_response.get('auth')
    if auth:
        auth.update(
            password_verification_age=int(auth['password_verification_age']),
        )
        if 'social' in list(iterkeys(auth)):
            auth['social']['profile_id'] = int(auth['social']['profile_id'])
    if 'uid' in user_response:
        # Информация о пользователе в ответе есть
        user = parse_blackbox_user(user_response, request)
        user['is_lite_session'] = user_response['uid'].get('lite', False)

        # Нам не нужно иметь и lite, и is_lite_session (одно и то же)
        user.pop('lite', None)
        return user['uid'], user

    # Информации о пользователе в ответе нет
    return item_id, user_response


def parse_blackbox_sessionid_response(response, request=None):
    status = response['status']['value']
    response.update(status=status)
    response['cookie_status'] = status

    if 'age' in response:
        response['age'] = int(response['age'])
    if 'ttl' in response:
        response['ttl'] = int(response['ttl'])

    if 'users' in response:
        # мультисессионный ответ - несколько пользователей
        default_uid = int(response['default_uid'])
        response['default_uid'] = default_uid
        response['users'] = dict(parse_user(user, request) for user in response['users'])
        default_user = response['users'].get(default_uid)
        if default_user:
            response.update(default_user)
        return response

    _, user = parse_user(response, request=request)
    return user


def parse_blackbox_phone_bindings_response(response, need_current, need_history, need_unbound):
    return parse_phone_bindings(
        phone_bindings=response[u'phone_bindings'],
        need_current=need_current,
        need_history=need_history,
        need_unbound=need_unbound,
    )


def parse_phone_bindings(phone_bindings, need_current, need_history, need_unbound):
    filtered = []
    for binding in phone_bindings:
        _type = binding[u'type']
        if (_type == 'current' and need_current or
            _type == 'history' and need_history or
                _type == 'unbound' and need_unbound):
            filtered.append(binding)
    return [parse_binding(binding) for binding in filtered]


def parse_binding(binding):
    # Адаптируем ЧЯшный номер к требованиям PhoneNumber
    plus_number = u'+' + binding[u'number']
    # парсим даже неправильные номера, потому что они прилетели из ЧЯ и мы должны их парсить
    phone_number = PhoneNumber.parse(plus_number, allow_impossible=True)
    if binding[u'phone_id']:
        phone_id = int(binding[u'phone_id'])
    else:
        phone_id = None
    uid = int(binding[u'uid'])
    binding_time = int(binding[u'bound'])
    flags = PhoneBindingsFlags(int(binding[u'flags']))

    return {
        u'type': binding['type'],
        u'phone_number': phone_number,
        u'phone_id': phone_id,
        u'uid': uid,
        u'binding_time': binding_time,
        u'should_ignore_binding_limit': flags.should_ignore_binding_limit,
    }


def parse_find_pdd_accounts_response(resp):
    return {
        'uids': list(map(int, resp['uids'])),
        'total_count': int(resp['total_count']),
        'count': int(resp['count']),
    }


def parse_blackbox_phone_operations_response(response):
    return list(map(parse_phone_operation, response['phone_operations']))


def parse_blackbox_get_all_tracks_response(response):
    return response['track']


def parse_yakey_backup_response(response):
    if not response['yakey_backups']:
        return dict(
            status=BLACKBOX_YAKEY_BACKUP_NOT_FOUND_STATUS,
        )

    yakey_backup = response['yakey_backups'][0]

    phone_number = yakey_backup['phone_number']
    # парсим даже неправильные номера, потому что они прилетели из ЧЯ и мы должны их парсить
    phone_number = PhoneNumber.parse('+' + str(phone_number), allow_impossible=True)
    yakey_backup['phone_number'] = phone_number

    response_backup = {
        'status': BLACKBOX_YAKEY_BACKUP_EXISTS_STATUS,
        'yakey_backup': yakey_backup,
    }

    return response_backup


def parse_account_deletion_operation(response):
    started_at = response.get('account.deletion_operation_started_at')
    if started_at is not None:
        return {
            'account_deletion_operation': {
                'started_at': int(started_at),
            },
        }
    else:
        return {}


def parse_deletion_operations_response(response):
    ops = response['deletion_operations']
    ops = [{'uid': int(op['uid'])} for op in ops]
    return {'deletion_operations': ops}


def parse_blackbox_get_recovery_keys_response(response):
    return response['recovery_key'] or None


def parse_blackbox_check_rfc_totp_response(response):
    rv = {
        'status': response['status'],
    }
    if 'time' in response:
        rv.update(time=response['time'])
    return rv


def parse_blackbox_check_device_signature_response(response):
    status = response['status']
    if status == 'OK':
        return
    # В error находятся связанные с безопасностью подробности, которые нельзя
    # показывать потребителям или пользователям.
    security_info = response.get('error')
    raise BlackboxInvalidDeviceSignature(status, security_info)


def parse_blackbox_family_info_response(family_id, response):
    # Доверяем структуре ответа от ЧЯ
    family = response['family'][family_id]
    # Если пришёл пустой dict в качестве семьи - вернуть None
    if not family:
        return
    # Добавляем family_id
    family['family_id'] = family_id
    # Меняем список юзеров на словарь по uid
    family['users'] = {int(u['uid']): u for u in family['users']}
    return family


def parse_blackbox_generate_public_id_response(response):
    return response['public_id']


def parse_blackbox_webauthn_credentials_response(response):
    return {
        cred_id: {
            'uid': int(info['uid']) if info['uid'] else None,
        }
        for cred_id, info in response.items()
    }


def parse_blackbox_get_oauth_tokens_response(response):
    for token in response['tokens']:
        for field in ('issue_time', 'ctime', 'expire_time'):
            if token['oauth'].get(field):
                token['oauth'][field] = parse_datetime(token['oauth'][field])

    return response['tokens']


__all__ = globals().keys()
