# -*- coding: utf-8 -*-

from datetime import datetime
import logging
from time import time

from passport.backend.core.builders.base.base import RequestInfo
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_EDITSESSION_OP_ADD,
    BLACKBOX_EDITSESSION_OP_DELETE,
    BLACKBOX_EDITSESSION_OP_SELECT,
)
from passport.backend.core.conf import settings
from passport.backend.core.eav_type_mapping import (
    EXTENDED_ATTRIBUTES_EMAIL_NAME_TO_TYPE_MAPPING,
    EXTENDED_ATTRIBUTES_PHONE_NAME_TO_TYPE_MAPPING,
    EXTENDED_ATTRIBUTES_WEBAUTHN_NAME_TO_TYPE_MAPPING,
    get_attr_type,
)
from passport.backend.core.host.host import get_current_host
from passport.backend.core.tvm import get_tvm_credentials_manager
from passport.backend.utils.string import smart_unicode
from passport.backend.utils.time import datetime_to_integer_unixtime
from six.moves.urllib.parse import urljoin


log = logging.getLogger('passport.blackbox.url_builders')


def serialize_blackbox_attributes_to_url(attributes_by_names):
    return ','.join(
        map(
            lambda name: str(get_attr_type(name)),
            attributes_by_names,
        ),
    )


def serialize_blackbox_phone_attributes_to_url(phone_attr_names):
    phone_attr_types = [
        EXTENDED_ATTRIBUTES_PHONE_NAME_TO_TYPE_MAPPING[attr_name]
        for attr_name in phone_attr_names
    ]
    return u','.join(map(str, phone_attr_types))


def add_phone_operations_to_request_args(need_phone_operations, args):
    if need_phone_operations:
        args.update({'getphoneoperations': u'1'})


def add_phones_and_phone_attributes_to_request_args(phones, phone_attrs, args):
    if phones:
        args.update({'getphones': phones})
        if phone_attrs is None:
            phone_attrs = settings.BLACKBOX_PHONE_EXTENDED_ATTRIBUTES
        if phone_attrs:
            args.update({
                u'phone_attributes': serialize_blackbox_phone_attributes_to_url(phone_attrs),
            })


def add_phone_bindings_to_request_args(need_current, need_unbound, args):
    """
    Строит параметры для получения телефонных связок из ручек userinfo, login,
    sessionid, oauth.
    """
    if need_current and need_unbound:
        _type = u'all'
    elif need_current:
        _type = u'current'
    elif need_unbound:
        _type = u'unbound'
    else:
        _type = None
    if _type is not None:
        args.update({u'getphonebindings': _type})


def add_email_attributes_to_request_args(email_attributes, args):
    if email_attributes is not None:
        args['getemails'] = 'all'

        if email_attributes == 'all':
            args['email_attributes'] = 'all'
        else:
            args['email_attributes'] = ','.join([
                str(EXTENDED_ATTRIBUTES_EMAIL_NAME_TO_TYPE_MAPPING[attr])
                for attr in email_attributes
            ])


def add_webauthn_credential_attributes_to_request_args(get_webauthn_credentials, webauthn_credential_attributes, args):
    if get_webauthn_credentials:
        args['get_webauthn_credentials'] = 'all'

        if webauthn_credential_attributes is None:
            webauthn_credential_attributes = 'all'

        if webauthn_credential_attributes == 'all':
            args['webauthn_credential_attributes'] = 'all'
        else:
            args['webauthn_credential_attributes'] = ','.join([
                str(EXTENDED_ATTRIBUTES_WEBAUTHN_NAME_TO_TYPE_MAPPING[attr])
                for attr in webauthn_credential_attributes
            ])


def add_get_user_ticket_to_request_args(args, headers):
    # Параметр get_user_ticket работает только при переданном сервис-тикете. Поэтому явно добавляем хедер с
    # сервис-тикетом, так как использование TVM может быть отключено настройкой билдера.
    headers.update({
        'X-Ya-Service-Ticket': get_tvm_credentials_manager().get_ticket_by_alias('blackbox'),
    })
    args['get_user_ticket'] = 'yes'


def phone_bindings_args_to_type(need_current, need_history, need_unbound):
    """
    Строит параметры для получения телефонных связок из ручки phone_bindings.
    """
    if need_current and not need_history and not need_unbound:
        _type = u'current'
    elif need_history and not need_current and not need_unbound:
        _type = u'history'
    elif need_history or need_current or need_unbound:
        _type = u'all'
    else:
        raise ValueError(u"Need current, need history, need_unbound can't be False at the same time")
    return _type


class BlackboxRequestBuilder(object):

    def _request_info_builder(self, get_args, post_args, headers=None):
        return RequestInfo(urljoin(self.url, '/blackbox/'), get_args, post_args, headers)

    def build_userinfo_request(
        self,
        uid=None,
        login=None,
        public_id=None,
        uids=None,
        ip='127.0.0.1',
        sid=None,
        suid=None,
        dbfields=None,
        attributes=None,
        emails=False,
        email_attributes=None,
        need_display_name=True,
        need_public_name=True,
        need_aliases=True,
        get_hidden_aliases=None,
        pin_to_test=None,
        phones=None,
        phone_attributes=None,
        need_phone_operations=False,
        need_current_phone_bindings=False,
        need_unbound_phone_bindings=False,
        find_by_phone_alias=None,
        country=None,
        get_public_id=False,
        get_billing_features=False,
        get_family_info=False,
        force_show_mail_subscription=False,
        get_webauthn_credentials=False,
        webauthn_credential_attributes=None,
        allow_scholar=None,
    ):
        dbfields = settings.BLACKBOX_FIELDS if dbfields is None else dbfields
        attributes = settings.BLACKBOX_ATTRIBUTES if attributes is None else attributes
        get_hidden_aliases = settings.BLACKBOX_GET_HIDDEN_ALIASES if get_hidden_aliases is None else get_hidden_aliases

        post_args = {
            'method': 'userinfo',
            'userip': ip,
            'format': 'json',
        }

        if dbfields:
            post_args['dbfields'] = ','.join(dbfields)

        if attributes:
            post_args['attributes'] = serialize_blackbox_attributes_to_url(attributes)

        if sid is not None:
            post_args['sid'] = sid

        if uid is not None:
            post_args['uid'] = uid
        elif uids is not None:
            if isinstance(uids, (list, tuple, set)):
                post_args['uid'] = ','.join(map(str, uids))
            else:
                raise ValueError('`uids` parameter should be list, tuple, set or string')
        elif login is not None:
            post_args['login'] = smart_unicode(login)
        elif suid is not None and sid is not None:
            post_args['suid'] = suid
        elif public_id is not None:
            post_args['public_id'] = public_id

        else:
            raise ValueError('uid, uids, login, public_id or suid and sid is required')

        if emails:
            post_args['emails'] = 'getall'

        if need_display_name:
            # в ответе приходит информация про registration_name и display_name
            post_args.update(
                regname='yes',
                is_display_name_empty='yes',
            )
            if need_public_name:
                post_args.update(
                    get_public_name='yes',
                )

        if need_aliases:
            post_args['aliases'] = 'all_with_hidden' if get_hidden_aliases else 'all'

        if pin_to_test:
            post_args['pintotest'] = pin_to_test

        if find_by_phone_alias is not None:
            post_args['find_by_phone_alias'] = find_by_phone_alias

        if country is not None:
            post_args['country'] = country  # используется для поиска по ЦА без доменной части

        if allow_scholar:
            post_args['allow_scholar'] = 'yes'

        add_email_attributes_to_request_args(email_attributes, post_args)
        add_phones_and_phone_attributes_to_request_args(
            phones,
            phone_attributes,
            post_args,
        )
        add_phone_operations_to_request_args(need_phone_operations, post_args)
        add_phone_bindings_to_request_args(
            need_current_phone_bindings,
            need_unbound_phone_bindings,
            post_args,
        )
        add_webauthn_credential_attributes_to_request_args(
            get_webauthn_credentials,
            webauthn_credential_attributes,
            post_args,
        )

        if get_billing_features:
            post_args['get_billing_features'] = 'all'

        if get_family_info:
            post_args['get_family_info'] = 'yes'

        if get_public_id:
            post_args['get_public_id'] = 'yes'

        if force_show_mail_subscription:
            post_args['force_show_mail_subscription'] = 'yes'

        return self._request_info_builder({}, post_args)

    def build_pwdhistory_request(self, uid, password, depth, reason=None):
        post_args = {
            'method': 'pwdhistory',
            'uid': uid,
            'depth': depth,
            'password': password,
            'format': 'json',
        }
        if reason is not None:
            post_args['reason'] = ','.join(map(str, reason))
        return self._request_info_builder({}, post_args)

    def build_get_single_email_request(self, userip, address, uid=None, login=None):
        post_args = {
            'method': 'userinfo',
            'emails': 'testone',
            'addrtotest': address,
            'userip': userip,
            'format': 'json',
        }

        if uid:
            post_args['uid'] = uid
        elif login:
            post_args['login'] = login
        else:
            raise ValueError('Specify either uid or login')

        return self._request_info_builder({}, post_args)

    def build_test_pwd_hashes_request(self, password, hashes, uid=None):
        post_args = {
            'method': 'test_pwd_hashes',
            'password': password,
            'hashes': ','.join(hashes),
            'format': 'json',
        }
        if uid is not None:
            post_args['uid'] = uid
        return self._request_info_builder({}, post_args)

    def build_create_pwd_hash_request(self, version, uid=None, password=None, md5crypt_hash=None, rawmd5_hash=None):
        post_args = {}
        if password is not None:
            post_args['password'] = password
        if md5crypt_hash is not None:
            post_args['md5crypt'] = md5crypt_hash
        if rawmd5_hash is not None:
            post_args['rawmd5'] = rawmd5_hash

        if len(post_args) != 1:
            raise ValueError('Either password or one of hashes must be passed')

        post_args.update({
            'method': 'create_pwd_hash',
            'ver': str(version),
            'format': 'json',
        })
        if uid is not None:
            post_args['uid'] = str(uid)
        return self._request_info_builder({}, post_args)

    def build_hosted_domains_request(self, domain_id=None, domain=None,
                                     domain_admin=None, aliases=False):
        if domain_id is None and domain is None and domain_admin is None:
            raise ValueError('Specify domain, domain_id or domain_admin argument')

        request = {
            'method': 'hosted_domains',
            'format': 'json',
        }
        if domain_id is not None:
            request['domain_id'] = domain_id
        if domain is not None:
            # Переводим в punycode домен, до тех пор пока ЧЯ не научится принимать as is
            try:
                request['domain'] = domain.encode('idna').decode('utf-8')
            except UnicodeError:
                request['domain'] = domain.encode('utf-8').decode('utf-8')
        if domain_admin is not None:
            request['domain_admin'] = domain_admin
        if aliases:
            request['aliases'] = True

        return self._request_info_builder(request, None)

    def build_loginoccupation_request(
        self,
        logins,
        ignore_stoplist=None,
        is_pdd=None,
    ):
        if not logins:
            raise ValueError('empty logins list')

        request = {
            'method': 'loginoccupation',
            'logins': ','.join(smart_unicode(login) for login in logins),
            'format': 'json',
        }

        if ignore_stoplist:
            request['ignore_stoplist'] = '1'
        if is_pdd:
            request['is_pdd'] = '1'

        return self._request_info_builder(request, None)

    def build_createsession_request(
        self,
        uid,
        ip,
        keyspace,
        ttl,
        is_yastaff=False,
        social_id=None,
        is_lite=False,
        is_betatester=False,
        ver=None,
        lang=None,
        password_check_time=None,
        have_password=False,
        yateam_auth=None,
        guard_hosts=None,
        request_id=None,
        login_id=None,
        is_scholar=None,
        get_login_id=False,
    ):
        now = time()
        request = {
            'method': 'createsession',
            'format': 'json',
            'uid': uid,
            'userip': ip,
            'host_id': '%x' % get_current_host().get_id(),
            'ver': settings.BLACKBOX_SESSION_VERSION,
            'ttl': ttl,
            'keyspace': keyspace,
            'create_time': int(now),
            'auth_time': int(now * 1000),
        }
        if ver is not None:
            request['ver'] = ver
        if lang is not None:
            request['lang'] = lang
        if password_check_time is not None:
            request['password_check_time'] = int(password_check_time)
        if is_yastaff:
            request['is_yastaff'] = '1'
        if is_betatester:
            request['is_betatester'] = '1'
        if social_id:
            request['social_id'] = social_id
        if yateam_auth is not None:
            request['yateam_auth'] = int(yateam_auth)
        if guard_hosts:
            guard_hosts = [x for i, x in enumerate(guard_hosts) if guard_hosts.index(x) == i]
            request['guard_hosts'] = ','.join(guard_hosts)
        if login_id is not None:
            request['login_id'] = login_id
        if get_login_id:
            request['get_login_id'] = 'yes'

        request['have_password'] = int(have_password)
        request['is_lite'] = int(is_lite)

        if request_id is not None:
            request['request_id'] = request_id

        if is_scholar:
            request['is_scholar'] = '1'

        return self._request_info_builder(request, None)

    def build_editsession_request(
        self,
        op,
        sessionid,
        uid,
        ip,
        host,
        sslsessionid=None,
        new_default=None,
        is_yastaff=False,
        is_betatester=False,
        social_id=None,
        lang=None,
        have_password=None,
        password_check_time=None,
        keyspace=None,
        is_lite=False,
        yateam_auth=None,
        sessguard=None,
        guard_hosts=None,
        request_id=None,
        is_scholar=None,
        get_login_id=False,
    ):
        if op not in [
            BLACKBOX_EDITSESSION_OP_ADD,
            BLACKBOX_EDITSESSION_OP_DELETE,
            BLACKBOX_EDITSESSION_OP_SELECT,
        ]:
            raise ValueError('Invalid operation: "%s"' % op)
        request = {
            'method': 'editsession',
            'format': 'json',
            'sessionid': sessionid,
            'op': op,
            'uid': uid,
            'userip': ip,
            'host': host,
            'create_time': int(time()),
        }
        if sslsessionid:
            request['sslsessionid'] = sslsessionid
        if new_default:
            request['new_default'] = new_default
        if is_yastaff:
            request['is_yastaff'] = '1'
        if is_betatester:
            request['is_betatester'] = '1'
        if social_id:
            request['social_id'] = social_id
        if lang:
            request['lang'] = lang
        if password_check_time is not None:
            request['password_check_time'] = int(password_check_time)
        if have_password:
            request['have_password'] = have_password
        if keyspace:
            request['keyspace'] = keyspace
        if is_lite:
            request['is_lite'] = '1'
        if yateam_auth is not None:
            request['yateam_auth'] = int(yateam_auth)
        if sessguard is not None:
            request['sessguard'] = sessguard
        if guard_hosts:
            guard_hosts = [x for i, x in enumerate(guard_hosts) if guard_hosts.index(x) == i]
            request['guard_hosts'] = ','.join(guard_hosts)

        if request_id is not None:
            request['request_id'] = request_id

        if is_scholar:
            request['is_scholar'] = '1'
        if get_login_id:
            request['get_login_id'] = 'yes'

        return self._request_info_builder(request, None)

    def build_create_oauth_token_request(self, uid, ip, client_id, expire_time, create_time=None, scope_ids=None,
                                         device_id=None, token_id=None, xtoken_id=None, xtoken_shard=None, meta=None,
                                         **kwargs):
        request = dict(
            method='create_oauth_token',
            format='json',
            uid=uid,
            userip=ip,
            client_id=client_id,
            expire_time=expire_time,
            create_time=create_time or int(time()),
            **kwargs
        )
        if scope_ids:
            request['scopes'] = ','.join(map(str, sorted(set(scope_ids))))
        if device_id is not None:
            request['device_id'] = device_id
        if token_id is not None:
            request['token_id'] = token_id
        if meta is not None:
            request['meta'] = meta
        if xtoken_id is not None:
            if xtoken_shard is None:
                raise ValueError('`xtoken_shard` is required if `xtoken_id` is specified')
            request.update(
                xtoken_id=xtoken_id,
                xtoken_shard=xtoken_shard,
            )

        return self._request_info_builder(request, None)

    def build_oauth_request(self, oauth_token=None, ip='127.0.0.1', dbfields=None, attributes=None,
                            emails=False, email_attributes=None, need_display_name=True, need_public_name=True,
                            need_aliases=True, get_hidden_aliases=None, phones=None,
                            phone_attributes=None, need_phone_operations=False,
                            need_current_phone_bindings=False,
                            need_unbound_phone_bindings=False,
                            get_user_ticket=False,
                            need_token_attributes=False, need_client_attributes=False, headers=None,
                            get_login_id=False, get_billing_features=False,
                            get_family_info=False, get_public_id=False, request_id=None,
                            force_show_mail_subscription=False,
                            get_webauthn_credentials=False, webauthn_credential_attributes=None):
        dbfields = settings.BLACKBOX_FIELDS if dbfields is None else dbfields
        attributes = settings.BLACKBOX_ATTRIBUTES if attributes is None else attributes
        get_hidden_aliases = settings.BLACKBOX_GET_HIDDEN_ALIASES if get_hidden_aliases is None else get_hidden_aliases

        request = {
            'method': 'oauth',
            'userip': ip,
            'format': 'json',
        }

        if oauth_token:
            request.update({'oauth_token': oauth_token})

        if dbfields:
            request['dbfields'] = ','.join(dbfields)

        if attributes:
            request['attributes'] = serialize_blackbox_attributes_to_url(attributes)

        if emails:
            request['emails'] = 'getall'

        if need_display_name:
            # в ответе приходит информация про registration_name и display_name
            request.update(
                regname='yes',
                is_display_name_empty='yes',
            )
            if need_public_name:
                request.update(
                    get_public_name='yes',
                )

        if need_aliases:
            request['aliases'] = 'all_with_hidden' if get_hidden_aliases else 'all'

        if need_token_attributes:
            request['oauth_token_attributes'] = 'all'

        if need_client_attributes:
            request['oauth_client_attributes'] = 'all'

        if get_login_id:
            request['get_login_id'] = 'yes'

        if get_billing_features:
            request['get_billing_features'] = 'all'

        if get_family_info:
            request['get_family_info'] = 'yes'

        if get_public_id:
            request['get_public_id'] = 'yes'

        if request_id is not None:
            request['request_id'] = request_id

        if force_show_mail_subscription:
            request['force_show_mail_subscription'] = 'yes'

        add_email_attributes_to_request_args(email_attributes, request)
        add_phones_and_phone_attributes_to_request_args(
            phones,
            phone_attributes,
            request,
        )
        add_phone_operations_to_request_args(need_phone_operations, request)
        add_phone_bindings_to_request_args(
            need_current_phone_bindings,
            need_unbound_phone_bindings,
            request,
        )
        add_webauthn_credential_attributes_to_request_args(
            get_webauthn_credentials,
            webauthn_credential_attributes,
            request,
        )

        headers = headers or {}
        if get_user_ticket:
            add_get_user_ticket_to_request_args(request, headers)

        return self._request_info_builder(request, None, headers)

    def build_lrandoms_request(self):
        return RequestInfo(urljoin(self.url, self.LRANDOMS_PATH), None, None)

    def build_find_pdd_accounts_request(self, domain_id=None, domain=None, login=None, offset=None,
                                        limit=None, sort_by=None, descending_order=False):

        request = {
            'method': 'find_pdd_accounts',
            'format': 'json',
        }

        if domain and domain_id:
            raise ValueError('You should specify only domain name or domain ID, not both.')
        elif domain:
            request['domain'] = domain
        elif domain_id:
            request['domain_id'] = domain_id
        else:
            raise ValueError('You should specify either domain name or domain ID.')

        if login:
            request['login'] = login

        request['sort'] = '%s.%s' % (sort_by or 'uid', 'desc' if descending_order else 'asc')

        if offset:
            request['offset'] = offset

        if limit is not None:
            # Если limit будет больше определенного порога (100 на момент
            # написания), то ЧЯ тихо установит limit равным порогу.
            request['limit'] = limit

        return self._request_info_builder(request, None)

    def build_edit_totp_request(self, operation, *args, **kwargs):
        # Здесь на одном mode живут несколько непохожих операций.
        # Не будем смешивать мух с котлетами - используем несколько урл-билдеров.
        if operation == 'create':
            return self.build_create_totp_secret_request(*args, **kwargs)
        elif operation == 'replace':
            return self.build_replace_totp_secret_request(*args, **kwargs)
        else:
            raise ValueError('Unknown operation: %s' % operation)

    def build_create_totp_secret_request(self, uid, pin, secret_id, secret, otp):
        return self._request_info_builder(
            {},
            {
                'method': 'edit_totp',
                'op': 'create',
                'uid': uid,
                'secret': secret,
                'secret_id': secret_id,
                'pin': pin,
                'password': otp,
                'format': 'json',
            },
        )

    def build_replace_totp_secret_request(self, uid, old_secret_id, new_secret_id, secret, otp):
        return self._request_info_builder(
            {},
            {
                'method': 'edit_totp',
                'op': 'replace',
                'uid': uid,
                'old_secret_id': old_secret_id,
                'secret_id': new_secret_id,
                'secret': secret,
                'password': otp,
                'format': 'json',
            },
        )

    def build_login_request(
        self,
        version,
        password,
        ip,
        uid=None,
        login=None,
        sid=None,
        from_service=None,
        authtype=None,
        emails=False,
        dbfields=None,
        attributes=None,
        need_aliases=True,
        get_hidden_aliases=None,
        need_display_name=True,
        need_public_name=True,
        useragent=None,
        referer=None,
        retpath=None,
        yandexuid=None,
        secret=None,
        phones=None,
        phone_attributes=None,
        email_attributes=None,
        need_phone_operations=False,
        captcha_already_checked=False,
        need_current_phone_bindings=False,
        need_unbound_phone_bindings=False,
        get_user_ticket=False,
        find_by_phone_alias=None,
        country=None,
        get_family_info=False,
        get_public_id=False,
        force_show_mail_subscription=False,
        get_badauth_counts=True,
        get_webauthn_credentials=False,
        webauthn_credential_attributes=None,
        allow_scholar=None,
    ):
        dbfields = settings.BLACKBOX_FIELDS if dbfields is None else dbfields
        attributes = settings.BLACKBOX_ATTRIBUTES if attributes is None else attributes
        get_hidden_aliases = settings.BLACKBOX_GET_HIDDEN_ALIASES if get_hidden_aliases is None else get_hidden_aliases

        request = {
            'method': 'login',
            'password': password,
            'userip': ip,
            'full_info': 'yes',
            'ver': str(version),
            'format': 'json',
        }

        if version == 1 and captcha_already_checked:
            request['captcha'] = 'no'

        if dbfields:
            request['dbfields'] = ','.join(dbfields)

        if attributes:
            request['attributes'] = serialize_blackbox_attributes_to_url(attributes)

        if uid is not None:
            request['uid'] = uid
        elif login is not None:
            request['login'] = smart_unicode(login)
        else:
            raise ValueError('uid or login is required')

        # Необязательные параметры ниже
        if sid is not None:
            request['sid'] = sid

        if from_service:
            request['from'] = from_service

        if authtype:
            request['authtype'] = authtype

        if emails:
            request['emails'] = 'getall'

        # В доке не указано, но ЧЯ хотел бы видеть и эти параметры тоже
        if useragent:
            request['useragent'] = useragent
        if referer:
            request['referer'] = referer
        if retpath:
            request['xretpath'] = retpath
        if yandexuid:
            request['yandexuid'] = yandexuid

        # Получить дополнительные поля в ответе ЧЯ
        if need_aliases:
            request['aliases'] = 'all_with_hidden' if get_hidden_aliases else 'all'
        if need_display_name:
            # добавить в ответ поля regname и display_name
            request.update(
                regname='yes',
                is_display_name_empty='yes',
            )
            if need_public_name:
                request.update(
                    get_public_name='yes',
                )
        if secret:
            request['secret'] = secret
        if find_by_phone_alias is not None:
            request['find_by_phone_alias'] = find_by_phone_alias
        if country is not None:
            request['country'] = country  # используется для поиска по ЦА без доменной части
        if get_family_info:
            request['get_family_info'] = 'yes'
        if get_public_id:
            request['get_public_id'] = 'yes'
        if force_show_mail_subscription:
            request['force_show_mail_subscription'] = 'yes'
        if get_badauth_counts:
            request['get_badauth_counts'] = 'yes'
        if allow_scholar:
            request['allow_scholar'] = 'yes'

        add_email_attributes_to_request_args(email_attributes, request)
        add_phones_and_phone_attributes_to_request_args(
            phones,
            phone_attributes,
            request,
        )
        add_phone_operations_to_request_args(need_phone_operations, request)
        add_phone_bindings_to_request_args(
            need_current_phone_bindings,
            need_unbound_phone_bindings,
            request,
        )
        add_webauthn_credential_attributes_to_request_args(
            get_webauthn_credentials,
            webauthn_credential_attributes,
            request,
        )

        headers = {}
        if get_user_ticket:
            add_get_user_ticket_to_request_args(request, headers)

        return self._request_info_builder({}, request, headers)

    def build_sessionid_request(
        self,
        sessionid,
        ip,
        host,
        sslsessionid=None,
        attributes=None,
        dbfields=None,
        emails=False,
        prolong_cookies=False,
        force_prolong=False,
        resign_for_domains=None,
        authid=True,
        email_attributes=None,
        need_display_name=True,
        need_public_name=True,
        multisession=False,
        need_aliases=True,
        get_hidden_aliases=None,
        phones=None,
        phone_attributes=None,
        need_phone_operations=False,
        need_current_phone_bindings=False,
        need_unbound_phone_bindings=False,
        useragent=None,
        yandexuid=None,
        get_user_ticket=False,
        sessguard=None,
        guard_hosts=None,
        request_id=None,
        get_login_id=False,
        get_billing_features=False,
        get_family_info=False,
        get_public_id=False,
        force_show_mail_subscription=False,
        get_webauthn_credentials=False,
        webauthn_credential_attributes=None,
        allow_scholar=None,
    ):
        dbfields = settings.BLACKBOX_FIELDS if dbfields is None else dbfields
        attributes = settings.BLACKBOX_ATTRIBUTES if attributes is None else attributes
        get_hidden_aliases = settings.BLACKBOX_GET_HIDDEN_ALIASES if get_hidden_aliases is None else get_hidden_aliases

        request = {
            'method': 'sessionid',
            'sessionid': sessionid,
            'userip': ip,
            'host': host,
            'full_info': 'yes',
            'format': 'json',
        }

        if dbfields:
            request['dbfields'] = ','.join(dbfields)

        if attributes:
            request['attributes'] = serialize_blackbox_attributes_to_url(attributes)

        if sslsessionid:
            request['sslsessionid'] = sslsessionid

        if emails:
            request['emails'] = 'getall'

        if prolong_cookies:
            request['resign'] = 'yes'
            if force_prolong:
                request['force_resign'] = 'yes'

        if resign_for_domains:
            request['resign_for_domains'] = ','.join(sorted(resign_for_domains))

        if authid:
            request['authid'] = 'yes'

        if multisession:
            request['multisession'] = 'yes'

        if need_display_name:
            # в ответе приходит информация про registration_name и display_name
            request.update(
                regname='yes',
                is_display_name_empty='yes',
            )
            if need_public_name:
                request.update(
                    get_public_name='yes',
                )

        if need_aliases:
            request['aliases'] = 'all_with_hidden' if get_hidden_aliases else 'all'

        if get_login_id:
            request['get_login_id'] = 'yes'

        if get_billing_features:
            request['get_billing_features'] = 'all'

        if useragent:
            request['useragent'] = useragent

        if yandexuid:
            request['yandexuid'] = yandexuid

        if sessguard is not None:
            request['sessguard'] = sessguard
        if guard_hosts:
            guard_hosts = [x for i, x in enumerate(guard_hosts) if guard_hosts.index(x) == i]
            request['guard_hosts'] = ','.join(guard_hosts)

        if request_id is not None:
            request['request_id'] = request_id

        if get_family_info:
            request['get_family_info'] = 'yes'

        if get_public_id:
            request['get_public_id'] = 'yes'

        if force_show_mail_subscription:
            request['force_show_mail_subscription'] = 'yes'

        if allow_scholar:
            request['allow_scholar'] = 'yes'

        add_email_attributes_to_request_args(email_attributes, request)
        add_phones_and_phone_attributes_to_request_args(
            phones,
            phone_attributes,
            request,
        )
        add_phone_operations_to_request_args(need_phone_operations, request)
        add_phone_bindings_to_request_args(
            need_current_phone_bindings,
            need_unbound_phone_bindings,
            request,
        )
        add_webauthn_credential_attributes_to_request_args(
            get_webauthn_credentials,
            webauthn_credential_attributes,
            request,
        )

        headers = {}
        if get_user_ticket:
            add_get_user_ticket_to_request_args(request, headers)

        return self._request_info_builder(request, None, headers)

    def build_phone_bindings_request(self, need_current, need_history, need_unbound,
                                     phone_numbers, phone_ids, bound_after_time,
                                     should_ignore_binding_limit):
        _type = phone_bindings_args_to_type(need_current, need_history, need_unbound)
        query = {
            u'method': u'phone_bindings',
            u'type': _type,
            u'format': u'json',
        }
        if phone_numbers:
            phone_numbers = u','.join(phone_numbers)
            query.update({u'numbers': phone_numbers})
        if phone_ids:
            phone_ids = u','.join(map(str, phone_ids))
            query.update({u'phoneids': phone_ids})
        if bound_after_time is not None:
            query.update({u'bound_after': bound_after_time})
        if should_ignore_binding_limit is not None:
            ignorebindlimit = str(int(should_ignore_binding_limit))
            query.update({u'ignorebindlimit': ignorebindlimit})

        return self._request_info_builder(query, None)

    def build_get_hosts_request(self, is_pdd=False, sid=None):
        query = {
            'method': 'get_hosts',
            'is_pdd': is_pdd,
            'format': 'json',
        }
        if sid is not None:
            query['sid'] = sid
        return self._request_info_builder(query, None)

    def build_get_track_request(self, uid, track_id):
        query = {
            'method': 'get_track',
            'uid': uid,
            'track_id': track_id,
            'format': 'json',
        }
        return self._request_info_builder(query, None)

    def build_get_all_tracks_request(self, uid):
        query = {
            'method': 'get_all_tracks',
            'uid': uid,
            'format': 'json',
        }
        return self._request_info_builder(query, None)

    def build_phone_operations_request(self, finished_before):
        query = {
            'method': 'phone_operations',
            'finished_before': datetime_to_integer_unixtime(finished_before),
            'format': 'json',
        }
        return self._request_info_builder(query, None)

    def build_yakey_backup_request(self, phone_number, meta=False):
        query = {
            'method': 'yakey_backup',
            'phone_number': phone_number,
            'meta': meta,
            'format': 'json',
        }
        return self._request_info_builder(query, None)

    def build_deletion_operations_request(self, deleted_after, deleted_before, chunk_count, chunk_no):
        query = {
            'method': 'deletion_operations',
            'deleted_after': datetime_to_integer_unixtime(deleted_after),
            'deleted_before': datetime_to_integer_unixtime(deleted_before),
            'chunk_count': chunk_count,
            'chunk_no': chunk_no,
            'format': 'json',
        }
        return self._request_info_builder(query, None)

    def build_get_recovery_keys_request(self, uid, key_id_b64):
        query = {
            'method': 'get_recovery_keys',
            'uid': uid,
            'key_id': key_id_b64,
            'format': 'json',
        }
        return self._request_info_builder(query, None)

    def build_check_rfc_totp_request(self, uid, totp):
        query = {
            'method': 'check_rfc_totp',
            'uid': uid,
            'totp': totp,
            'format': 'json',
        }
        return self._request_info_builder(query, None)

    def build_sign_request(self, value, ttl, sign_space):
        query = {
            'method': 'sign',
            'value': value,
            'ttl': ttl,
            'sign_space': sign_space,
            'format': 'json',
        }
        return self._request_info_builder(query, None)

    def build_check_sign_request(self, signed_value, sign_space):
        query = {
            'method': 'check_sign',
            'signed_value': signed_value,
            'sign_space': sign_space,
            'format': 'json',
        }
        return self._request_info_builder(query, None)

    def build_check_device_signature_request(
        self,
        nonce,
        nonce_sign_space,
        device_id,
        signature,
        timestamp,
        public_key,
        version,
    ):
        data = {
            'method': 'check_device_signature',
            'format': 'json',
            'nonce': nonce,
            'nonce_sign_space': nonce_sign_space,
            'device_id': device_id,
            'signature': signature,
        }
        if timestamp is not None:
            if isinstance(timestamp, datetime):
                timestamp = datetime_to_integer_unixtime(timestamp)
            data['timestamp'] = str(int(timestamp))
        if public_key is not None:
            data['public_key'] = public_key
        if version is not None:
            data['version'] = str(version)
        return self._request_info_builder(None, data)

    def build_get_device_public_key_request(self, device_id):
        args = {
            'method': 'get_device_public_key',
            'format': 'json',
            'device_id': device_id,
        }
        return self._request_info_builder(args, None)

    def build_family_info_request(self, family_id, get_place, get_members_info):
        args = {
            'method': 'family_info',
            'format': 'json',
            'family_id': family_id,
        }
        if get_place:
            args['get_place'] = 'yes'
        if get_members_info:
            args['get_members_info'] = get_members_info
        return self._request_info_builder(args, None)

    def build_generate_public_id_request(self, uid):
        args = {
            'method': 'generate_public_id',
            'format': 'json',
            'uid': uid,
        }
        return self._request_info_builder(args, None)

    def build_webauthn_credentials_request(self, credential_external_id):
        args = {
            'method': 'webauthn_credentials',
            'format': 'json',
            'credential_id': credential_external_id,
        }
        return self._request_info_builder(args, None)

    def build_get_oauth_tokens_request(
        self,
        uid,
        full_info,
        xtoken_only,
        get_is_xtoken_trusted,
        get_login_id,
        device_id,
        client_id,
    ):
        args = {
            'method': 'get_oauth_tokens',
            'format': 'json',
            'uid': uid,
        }
        if full_info:
            args['full_info'] = 'yes'
        if xtoken_only:
            args['xtoken_only'] = 'yes'
        if get_is_xtoken_trusted:
            args['get_is_xtoken_trusted'] = 'yes'
        if get_login_id:
            args['get_login_id'] = 'yes'
        if device_id is not None:
            args['device_id'] = device_id
        if client_id is not None:
            args['client_id'] = client_id

        return self._request_info_builder(args, None)

    def build_check_ip_request(self, ip):
        args = {
            'method': 'checkip',
            'format': 'json',
            'ip': ip,
            'nets': 'yandexusers',
        }
        return self._request_info_builder(args, None)
