# -*- coding: utf-8 -*-
import json
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger

from .exceptions import (
    BaseBotApiError,
    BotApiAuthError,
    BotApiInvalidRequestError,
    BotApiPermanentError,
    BotApiTemporaryError,
)


log = logging.getLogger('passport.bot_api')


def http_error_handler(response):
    if response.status_code == 401:
        raise BotApiAuthError(u'Request failed with with response=%s code=%s' % (
            trim_message(response.content.decode('utf-8')),
            response.status_code,
        ))
    if response.status_code == 400:
        raise BotApiInvalidRequestError(u'Request failed with with response=%s code=%s' % (
            trim_message(response.content.decode('utf-8')),
            response.status_code,
        ))

    if 500 <= response.status_code <= 599:
        raise BotApiTemporaryError('Request failed with code=%s' % response.status_code)

    if response.status_code != 200:
        raise BotApiPermanentError(u'Request failed with with response=%s code=%s' % (
            trim_message(response.content.decode('utf-8')),
            response.status_code,
        ))


class BotApi(BaseBuilder, JsonParserMixin):
    """
    https://bp.mssngr.yandex.net/docs/api/bot/
    """

    base_error_class = BaseBotApiError
    temporary_error_class = BotApiTemporaryError
    parser_error_class = BotApiPermanentError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='bot_api')
        super(BotApi, self).__init__(
            url=url or settings.BOT_API_URL,
            timeout=timeout or settings.BOT_API_TIMEOUT,
            retries=retries or settings.BOT_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            **kwargs
        )

    def _make_request(self, url_suffix, request_id, data):
        headers = {
            'Authorization': 'OAuth %s' % settings.BOT_API_TOKEN,
            'Content-Type': 'application/json',
        }
        if request_id:
            headers.update({'X-Request-Id': request_id})

        return self._request_with_retries_simple(
            url_suffix=url_suffix,
            method='POST',
            data=json.dumps(data),
            headers=headers,
            http_error_handler=http_error_handler,
            parser=self.parse_json,
            error_detector=None,
        )

    def send_message(self, uid, text, card=None, notification_text=None, request_id=None):
        data = {
            'user_uid': uid,
            'text': text,
        }
        if card:
            data.update(card=card)
        if notification_text:
            data.update(notification=notification_text)

        return self._make_request(
            url_suffix='/bot/sendMessage/',
            request_id=request_id,
            data=data,
        )

    def edit_message(self, uid, message_id, text, card=None, notification_text=None, request_id=None):
        data = {
            'user_uid': uid,
            'message_id': message_id,
            'text': text,
        }
        if card:
            data.update(card=card)
        if notification_text:
            data.update(notification=notification_text)

        return self._make_request(
            url_suffix='/bot/editMessage/',
            request_id=request_id,
            data=data,
        )


def get_bot_api():
    return BotApi()  # pragma: no cover
