# -*- coding: utf-8 -*-

from collections import namedtuple
import logging

from passport.backend.core.builders.base.base import (
    BaseBuilder,
    RequestInfo,
)
from passport.backend.core.builders.captcha.exceptions import (
    CaptchaAccessError,
    CaptchaError,
    CaptchaLocateError,
    CaptchaServerError,
    CaptchaTypeCheckError,
    CaptchaXmlParseError,
)
from passport.backend.core.builders.mixins.xml_parser.xml_parser import XmlParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.utils.string import smart_unicode


log = logging.getLogger(u'passport.captcha')

image_captcha = namedtuple(u'image_captcha', [u'url', u'type'])
voice_captcha = namedtuple(u'voice_captcha', [u'url', u'intro_url', u'type'])
captcha_response = namedtuple(
    'captcha_response',
    ['key', 'image_captcha', 'voice_captcha'],
)


def _language_or_country_or_type(caller, data_type_by_language, data_type_by_country,
                                 wave_type_by_language, wave_type_by_country,
                                 type_=None, language=None, country=None):
    if type_ == 'wave':
        type_by_country = wave_type_by_country
        type_by_language = wave_type_by_language
        if country is not None:
            type_ = type_by_country.get(country, type_by_country[''])

        if language is not None:
            type_ = type_by_language.get(language, type_by_language[''])

        language = country = None  # больше не нужны

    param_count = len([param for param in [type_, language, country] if param is not None])

    if not param_count:
        error_text = u'''Method %s was called with none of ['type_', 'language', 'country']
            parameters but should take exactly one of them''' % caller
        raise ValueError(error_text)

    if param_count > 1:
        error_text = u'''Method %s was called with too many parameters
            but should take only one of ['type_', 'language', 'country']''' % caller
        raise ValueError(error_text)

    if country is not None:
        type_ = data_type_by_country.get(country, data_type_by_country[''])

    if language is not None:
        type_ = data_type_by_language.get(language, data_type_by_language[''])

    return type_


def _get_scaled_image_type(image_type, scale_factor):
    """
    Возвращаем очередь капчи увеличенного размера, если такая есть
    :param image_type: имя оригинальной очереди капчи
    :param scale_factor: фактор масштабирования
    :return: результирующее имя очереди
    """
    scaled = settings.CAPTCHA_SCALE_MATCHING
    if image_type in scaled and scale_factor in scaled[image_type]:
        return scaled[image_type][scale_factor]

    return image_type


class Captcha(BaseBuilder, XmlParserMixin):
    temporary_error_class = CaptchaServerError
    base_error_class = CaptchaError
    parser_error_class = CaptchaXmlParseError

    specified_errors = {
        403: CaptchaAccessError(u'Access permissions are missing'),
        500: CaptchaServerError(u'A wild ERROR appeared! Unknown captcha server error'),
        502: CaptchaServerError(u'Request haven\'t been processed'),
    }

    def __init__(self, useragent=None, timeout=None, url=None, retries=None,
                 graphite_logger=None, consumer='passport-api'):
        timeout = timeout or settings.CAPTCHA_TIMEOUT
        if graphite_logger is None:
            graphite_logger = GraphiteLogger(service='captcha')
        self.consumer = consumer  # используется не для проверки грантов, а просто для разделения запросов в логах
        super(Captcha, self).__init__(
            url=url or settings.CAPTCHA_URL,
            timeout=timeout,
            retries=retries or settings.CAPTCHA_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
        )

    def _handle_http_error(self, raw_response):
        if raw_response.status_code in self.specified_errors:
            try:
                raise self.specified_errors[raw_response.status_code]
            except CaptchaAccessError as e:
                self.logger.warning(
                    u'Captcha request access error: "%s" for %s',
                    e,
                    raw_response.request.url,
                )
                raise

    def _get_request(self, method, parameters):
        return RequestInfo(u'%s%s' % (self.url, method), parameters, None)

    def generate(self, image_type=None, voice_type=None, language=None, country=None,
                 checks=None, https=False, voice=False, scale_factor=None, request_id=None):
        u"""Генерация капчи
        Captcha-server всегда отдаёт url на картиночную капчу.

        Args:
            image_type: тип картиночной капчи - std/lite/rus.
            voice_type: тип голосовой капчи, пока только rus.
            language: язык капчи (если не задан тип).
            country: страна пользователя (если не задан тип).
            checks (optional): количество разрешенных проверок для 1й капчи.
            При первом же неправильном ответе на капчу делает ее бесполезной
            для дальнейшего использования.
            https: параметр для включения https протокола для ссылки на картинку.
            voice: параметр для включения головой капчи.
            scale_factor: указание желаемого размера картинки - обычный, увеличенный, двойной
            request_id - id запроса
        Returns:
            'captcha_object' namedtuple: url адрес изображения 'image_url',
            идентификатор капчи 'key', url адрес голосовой капчи 'voice_url',
            url адрес голосового приглашения 'voice_intro_url'.
        """
        # Определяем тип картиночной капчи
        image_type = _language_or_country_or_type(
            type_=image_type,
            language=language,
            country=country,
            caller='generate',
            data_type_by_language=settings.CAPTCHA_LANGUAGE_MATCHING,
            data_type_by_country=settings.CAPTCHA_COUNTRY_MATCHING,
            wave_type_by_language=settings.WAVE_CAPTCHA_LANGUAGE_MATCHING,
            wave_type_by_country=settings.WAVE_CAPTCHA_COUNTRY_MATCHING,
        )
        image_type = _get_scaled_image_type(image_type, scale_factor)

        parameters = {
            'checks': checks or settings.CAPTCHA_CHECKS,
            'type': image_type,
            'client': self.consumer,
        }
        if https:
            parameters.update(https='on')
        if request_id:
            parameters.update(request_id=request_id)

        # Определяем тип голосовой капчи
        if voice:
            voice_type = _language_or_country_or_type(
                type_=voice_type,
                language=language,
                country=country,
                caller='generate',
                data_type_by_language=settings.CAPTCHA_VOICE_LANGUAGE_MATCHING,
                data_type_by_country=settings.CAPTCHA_VOICE_COUNTRY_MATCHING,
                wave_type_by_language={},
                wave_type_by_country={},
            )
            parameters.update(vtype=voice_type)
        else:
            voice_type = None

        tree = self._request_with_retries(
            'GET',
            self._get_request(u'generate', parameters),
            self.parse_xml,
            http_error_handler=self._handle_http_error,
        )
        image_captcha_result = image_captcha(tree.attrib[u'url'], image_type)
        voice_captcha_result = voice_captcha(
            tree.attrib.get(u'voiceurl'),
            tree.attrib.get(u'voiceintrourl'),
            voice_type,
        )
        response = captcha_response(tree.text, image_captcha_result, voice_captcha_result)
        log.info('Captcha with a new key=%s successfully generated, captcha image_type=%s or voice_type=%s, language=%s, country=%s',
                 response.key, image_type, voice_type, language, country)
        return response

    def check(self, answer, key, request_id=None):
        u"""Проверка капчи.

        Args:
            answer: пользовательский ввод капчи,
            key: идентификатор капчи для капча сервера
            request_id - id запроса
        Returns:
            True при правильно введенной капче, иначе False.
        """

        parameters = {
            'rep': smart_unicode(answer),
            'key': smart_unicode(key),
            'client': self.consumer,
        }
        if request_id:
            parameters.update(request_id=request_id)

        tree = self._request_with_retries(
            'GET',
            self._get_request(u'check', parameters),
            self.parse_xml,
            http_error_handler=self._handle_http_error,
        )

        specified_errors = {
            u'not found': CaptchaLocateError(
                u'Wrong captcha key, picture hasn\'t been opened or time out has expired'),
            u'inconsistent type': CaptchaTypeCheckError(
                u'The type arguments specified for the generation and check methods do not match')
        }

        if u'error' in tree.attrib:
            if tree.attrib[u'error'].lower() in specified_errors:
                self.logger.debug(
                    u'Captcha server returned "%s": "%s" error on %s',
                    tree.attrib[u'error'],
                    tree.attrib.get(u'error_desc', ''),
                    self.url,
                )
                raise specified_errors.get(tree.attrib[u'error'])
            elif tree.attrib[u'error'].lower() == u'invalid characters in the user answer':
                self.logger.debug(u'Invalid characters in the user answer')
                return False
            else:
                self.logger.warning(
                    u'Captcha server returned undocumented error as an "error" parameter: %s: %s',
                    tree.attrib[u'error'],
                    tree.attrib.get(u'error_desc', ''),
                )
                raise CaptchaError(u'Received an undocumented parameter from server')

        if tree.text:
            if tree.text.lower() == u'failed':
                return False

            elif tree.text.lower() == u'ok':
                return True

        self.logger.warning(u'Haven\'t found neither "ok" nor "failed" in the xml from server on %s', self.url)
        raise CaptchaError(u'Haven\'t found neither "ok" nor "failed" in xml')


def get_captcha():
    return Captcha()  # pragma: no cover
