# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.clean_web_api')


METHOD_CHECK_MEDIA = 'tmu.check_media'
METHOD_CHECK_MEDIA_BUNCH = 'tmu.check_all_media'
METHOD_CHECK_TEXT = 'tmu.check'
METHOD_CHECK_TEXT_BUNCH = 'tmu.check_all'


class BaseCleanWebError(BaseCoreError):
    pass


class CleanWebPermanentError(BaseCleanWebError):
    """Пятисотка или иная непредвиденная ошибка"""


class CleanWebTemporaryError(BaseCleanWebError):
    """Временная ошибка CleanWeb - стоит поретраиться"""


class CleanWebInvalidResponse(CleanWebPermanentError):
    """Ошибка в ответе"""


def clean_web_api_http_error_handler(response):
    if response.status_code in (502, 503):
        raise CleanWebTemporaryError(u'Request failed with code=%s; response=%s' % (
            response.status_code,
            trim_message(response.content.decode('utf-8')),
        ))
    if response.status_code != 200:
        raise CleanWebPermanentError(u'Request failed with code=%s; response=%s' % (
            response.status_code,
            trim_message(response.content.decode('utf-8')),
        ))


def _error_detector(response, raw_response):
    if 'result' in response:
        result = response['result']
        if type(result) == dict:  # v2
            if 'verdicts' not in result:
                raise CleanWebInvalidResponse(u'Wrong result type in response')
            result = result['verdicts']
        if type(result) != list:
            raise CleanWebInvalidResponse(u'Wrong result type in response')
        for d in result:
            if type(d) != dict:
                raise CleanWebInvalidResponse(u'Wrong result type in response')
            if 'value' not in d:
                raise CleanWebInvalidResponse(u'No value in result of response')
            if type(d['value']) != bool:
                raise CleanWebInvalidResponse(u'Wrong value type in result of response')
    elif 'error' not in response:
        raise CleanWebInvalidResponse(u'No result and no error in response')


def _make_bunch_error_detector(size_of_bunch):
    def error_detector(response, raw_response):
        if type(response) != list:
            raise CleanWebInvalidResponse(u'Wrong result type in bunch response')
        if len(response) != size_of_bunch:
            raise CleanWebInvalidResponse(u'Wrong number of subresponses in bunch response')
        for r in response:
            _error_detector(r, None)
    return error_detector


def get_entities_with_bad_verdicts_from_response(response):
    if 'error' in response:
        return None

    if type(response['result']) == list:
        verdicts = response['result']
    else:  # v2
        if 'errors' in response['result']:
            return None
        verdicts = response['result']['verdicts']

    log.debug(verdicts)

    whitelisted_entities = {
        v['entity']
        for v in verdicts
        if v['value'] is True and v['name'] == settings.CLEAN_WEB_EMERGENCY_APPROVE_CASE and not v['name'] == settings.CLEAN_WEB_EMERGENCY_REJECT_CASE
    }
    bad_fields = {
        v['entity']
        for v in verdicts
        if v['value'] is True and v['name'] in settings.CLEAN_WEB_REJECTED_CAUSES
    }
    return sorted(bad_fields - whitelisted_entities)


def response_simplifier(response):
    """
        Для одного ответа возвращает
        - если все хорошо - True
        - если случилась ошибка - None
        - если клинвеб нашел что-то плохое - False
    """
    if type(response) == dict:
        bad_entities = get_entities_with_bad_verdicts_from_response(response)
        if bad_entities is None:
            return None
        return len(bad_entities) == 0
    elif type(response) == list:
        return [response_simplifier(r) for r in response]


def response_simplifier_user_data(response):
    """
        Для одного ответа возвращает
        - если все хорошо - True
        - если случилась ошибка - None
        - если клинвеб нашел чтото плохое - список полей
    """
    if type(response) == dict:
        bad_entities = get_entities_with_bad_verdicts_from_response(response)
        if bad_entities is None:
            return None
        return bad_entities if bad_entities else True
    elif type(response) == list:
        return [response_simplifier(r) for r in response]


class CleanWebAPI(BaseBuilder, JsonParserMixin):
    """
    API для проверки текстов на нежелательное содержимое
    https://st.yandex-team.ru/CLEANWEB-72
    """
    base_error_class = BaseCleanWebError
    temporary_error_class = CleanWebTemporaryError
    parser_error_class = CleanWebPermanentError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None,
                 use_tvm=False, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='clean_web')
        super(CleanWebAPI, self).__init__(
            url=url or settings.CLEAN_WEB_API_URL,
            timeout=timeout or settings.CLEAN_WEB_API_TIMEOUT,
            retries=retries or settings.CLEAN_WEB_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='clean_web' if use_tvm else None,
            **kwargs
        )

    def _make_request(self, data, simplifier=None, error_detector=_error_detector):
        response = self._request_with_retries_simple(
            error_detector=error_detector,
            parser=self.parse_json,
            method='POST',
            json_data=data,
            http_error_handler=clean_web_api_http_error_handler,
            response_processor=simplifier,
        )
        return response

    def _make_request_data(self, type, fields, key, auto_only=True, verdict_data=None, uid=None):
        data = {
            'jsonrpc': '2.0',
            'method': 'process',
            'id': 1234,
            'params': {
                'service': 'passport',
                'type': type,
                'body': dict(
                    fields,
                    auto_only=auto_only,
                ),
                'key': key,
            },
        }
        if verdict_data is not None:
            data['params']['body']['verdict_data'] = verdict_data
        if uid is not None:
            data['params']['puid'] = str(uid)
        return data

    def _check(self, type, fields, key, simplifier=None, **kwargs):
        return self._make_request(self._make_request_data(type, fields, key, **kwargs), simplifier)

    def _check_bunch(self, type, fields, keys, simplifier=None, **kwargs):
        assert len(fields) == len(keys)
        return self._make_request(
            [
                self._make_request_data(type, fs, key, **kwargs)
                for fs, key in zip(fields, keys)
            ],
            simplifier,
            error_detector=_make_bunch_error_detector(len(fields)),
        )

    def check_text(self, text, key, simple_response=False, **kwargs):
        return self._check('text', {'text': text}, key, simplifier=response_simplifier if simple_response else None, **kwargs)

    def check_text_bunch(self, texts, keys, simple_response=False, **kwargs):
        return self._check_bunch('text', [{'text': t} for t in texts], keys, simplifier=response_simplifier if simple_response else None, **kwargs)

    def check_media(self, url, key, simple_response=False, **kwargs):
        return self._check('image', {'image_url': url}, key, simplifier=response_simplifier if simple_response else None, **kwargs)

    def check_media_bunch(self, urls, keys, simple_response=False, **kwargs):
        return self._check_bunch('image', [{'image_url': t} for t in urls], keys, simplifier=response_simplifier if simple_response else None, **kwargs)

    def check_user_data(self, key, first_name=None, last_name=None, full_name=None, display_name=None, public_id=None, simple_response=False, **kwargs):
        fields = {
            k: v
            for k, v in dict(
                first_name=first_name,
                last_name=last_name,
                full_name=full_name,
                display_name=display_name,
                public_id=public_id,
            ).items()
            if v is not None
        }
        return self._check('user_data', fields, key, simplifier=response_simplifier_user_data if simple_response else None, **kwargs)


def get_clean_web_api():
    return CleanWebAPI()  # pragma: no cover
