# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.collections.exceptions import (
    BaseCollectionsApiError,
    CollectionsApiInvalidResponseError,
    CollectionsApiNotFoundError,
    CollectionsApiPermanentError,
    CollectionsApiTemporaryError,
)
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.core.utils.domains import get_keyspace_by_host
from passport.backend.utils.common import remove_none_values


log = logging.getLogger('passport.collections_api')


def collections_api_error_detector(response, raw_response):
    if raw_response.status_code == 404:
        raise CollectionsApiNotFoundError()

    # TODO: Специально отдельно, чтобы легче было выпиливать
    # после влития https://st.yandex-team.ru/PODB-8986
    if raw_response.status_code == 400 and response.get('description', '') == 'login is not valid':
        raise CollectionsApiNotFoundError()

    if raw_response.status_code != 200 or 'results' not in response:
        log.warning(
            u'CollectionsApi bad response (status=%s code=%s response=%s)',
            response.get('status'),
            raw_response.status_code,
            trim_message(raw_response.content.decode('utf-8')),
        )
        raise CollectionsApiInvalidResponseError()


def collections_api_http_error_handler(raw_response):
    if raw_response.status_code >= 500:
        message = u'Request failed with response=%s code=%s' % (
            trim_message(raw_response.content.decode('utf-8')),
            raw_response.status_code,
        )
        if raw_response.status_code == 504:
            raise CollectionsApiTemporaryError(message)
        else:
            raise CollectionsApiPermanentError(message)


def collections_api_response_processor(response):
    return response['results']


class CollectionsApi(BaseBuilder, JsonParserMixin):
    base_error_class = BaseCollectionsApiError
    temporary_error_class = CollectionsApiTemporaryError
    parser_error_class = CollectionsApiInvalidResponseError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='collections_api')
        super(CollectionsApi, self).__init__(
            url=url or settings.COLLECTIONS_API_URL,
            timeout=timeout or settings.COLLECTIONS_API_TIMEOUT,
            retries=retries or settings.COLLECTIONS_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            **kwargs
        )

    def _make_request(self, url_suffix='', method='GET', params=None,
                      page_size=None, page=None, sessionid=None, host=None):
        if sessionid and not host:
            raise ValueError('Host is required when sessionid is specified')
        elif host:
            # TODO: перед включением sessguard для этого сервиса переделать передачу кук и хоста (PASSP-30037)
            host = get_keyspace_by_host(host)

        params = params or {}
        params.update({
            'page_size': page_size,
            'page': page,
        })
        params = remove_none_values(params)
        return self._request_with_retries_simple(
            url_suffix=url_suffix,
            method=method,
            params=params,
            headers={
                'Cookie': 'Session_id=%s' % sessionid,
                'X-Forwarded-Host': host,
                'Accept': 'application/json',
            },
            error_detector=collections_api_error_detector,
            parser=self.parse_json,
            http_error_handler=collections_api_http_error_handler,
            response_processor=collections_api_response_processor,
        )

    def collections(self, public_id, previews_count=6, page_size=10, page=1,
                    sessionid=None, host=None):
        params = {
            'owner.public_id': public_id,
            'previews_count': previews_count,
            'sort': '-service.updated_at',
        }
        try:
            return self._make_request(
                url_suffix='api/boards/',
                params=params,
                sessionid=sessionid,
                host=host,
                page_size=page_size,
                page=page,
            )
        except CollectionsApiNotFoundError:
            return []

    def pictures(self, public_id, page_size=10, page=1, sessionid=None, host=None):
        params = {
            'board.slug': 'izbrannoe-iz-yandex-kartinok',
            'board.owner.public_id': public_id,
            # автоинкрементное поле, аналог сортировки по времени добавления
            'sort': '-id',
        }
        try:
            return self._make_request(
                url_suffix='api/cards/',
                params=params,
                sessionid=sessionid,
                host=host,
                page_size=page_size,
                page=page,
            )
        except CollectionsApiNotFoundError:
            return []


def get_collections_api():
    return CollectionsApi()  # pragma: no cover
