# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.datasync_api.exceptions import (
    BaseDatasyncApiError,
    DatasyncAccountInvalidTypeError,
    DatasyncApiAuthorizationInvalidError,
    DatasyncApiObjectNotFoundError,
    DatasyncApiPermanentError,
    DatasyncApiTemporaryError,
    DatasyncUserBlockedError,
    DatasyncUserNotFound,
)
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.datasync')


def datasync_api_error_detector(response, raw_response, success_statuses=(200,)):
    if raw_response.status_code in success_statuses:
        return

    error = response.get('error', 'unknown_error')
    if raw_response.status_code >= 500:
        log.warning(
            'Request failed with error=%s code=%s',
            error,
            raw_response.status_code,
        )
        if error == 'DiskServiceUnavailableError' or error == 'InternalServerError':
            raise DatasyncApiTemporaryError(error)
        else:
            raise DatasyncApiPermanentError(error)
    elif raw_response.status_code in (401, 403):
        if error == 'DiskUnsupportedUserAccountTypeError':
            raise DatasyncAccountInvalidTypeError(error)
        elif error == 'DiskUserBlockedError':
            raise DatasyncUserBlockedError(error)
        elif error == 'UserNotFoundInPassportError':
            raise DatasyncUserNotFound(error)
        else:
            raise DatasyncApiAuthorizationInvalidError(error)
    elif raw_response.status_code == 404:
        raise DatasyncApiObjectNotFoundError(error)
    else:
        raise DatasyncApiPermanentError('Bad response: error=%s code=%s' % (error, raw_response.status_code))


class DatasyncApi(BaseBuilder, JsonParserMixin):
    base_error_class = BaseDatasyncApiError
    temporary_error_class = DatasyncApiTemporaryError
    parser_error_class = DatasyncApiPermanentError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='datasync_api')
        super(DatasyncApi, self).__init__(
            url=url or settings.DATASYNC_API_URL,
            timeout=timeout or settings.DATASYNC_API_TIMEOUT,
            retries=retries or settings.DATASYNC_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            tvm_dst_alias='datasync_api',
            **kwargs
        )

    def _make_request(self, uid, method='GET', url_suffix='', params=None, data=None, headers=None, parser=None, error_detector=None, **kwargs):
        headers = headers or {}
        headers.update({
            'X-Uid': str(uid),
        })
        return self._request_with_retries_simple(
            url_suffix=url_suffix,
            method=method,
            params=params,
            data=data,
            parser=parser or self.parse_json,
            error_detector=error_detector or datasync_api_error_detector,
            headers=headers,
            **kwargs
        )
