# -*- coding: utf-8 -*-
import functools

from passport.backend.core.builders.datasync_api.base import (
    datasync_api_error_detector,
    DatasyncApi,
)
from passport.backend.core.builders.datasync_api.exceptions import DatasyncApiObjectNotFoundError
from passport.backend.core.logging_utils.loggers import GraphiteLogger


class DiskApi(DatasyncApi):

    """
    Дока к апи: https://tech.yandex.ru/disk/poligon/#!//v1/disk/GetDisk
    """

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='disk_api')
        super(DiskApi, self).__init__(
            url=url,
            timeout=timeout,
            retries=retries,
            useragent=useragent,
            graphite_logger=graphite_logger,
            **kwargs
        )

    def disk_info(self, uid, fields=None):
        return self._make_request(
            uid=uid,
            url_suffix='/v1/disk',
            method='GET',
            params=dict(
                fields=','.join(fields or ['used_space', 'total_space', 'trash_size', 'is_paid']),
            ),
        )

    def _billing_subscriptions_response_processor(self, response):
        return response.get('items', [])

    def billing_subscriptions(self, uid, user_ip, offset=0, limit=100):
        try:
            return self._make_request(
                uid=uid,
                url_suffix='/v1/disk/billing/subscriptions',
                method='GET',
                params=dict(
                    offset=offset,
                    limit=limit,
                ),
                headers={
                    'X-Real-Ip': user_ip,
                },
                response_processor=self._billing_subscriptions_response_processor,
            )
        except DatasyncApiObjectNotFoundError:
            return []

    def _plus_parse_response(self, raw_response):
        if raw_response.status_code == 204:
            # При 204 статусе не смотрим в тело ответа (RFC)
            return

        return self.parse_json(raw_response)

    def plus_subscribe(self, uid, partner_id, product_id):
        self._make_request(
            uid=uid,
            url_suffix='/v1/disk/partners/{partner_id}/services'.format(partner_id=partner_id),
            method='POST',
            params=dict(
                product_id=product_id,
            ),
            parser=self._plus_parse_response,
            error_detector=functools.partial(datasync_api_error_detector, success_statuses=(201, 409,)),
        )

    def plus_unsubscribe(self, uid, partner_id, product_id):
        self._make_request(
            uid=uid,
            url_suffix='/v1/disk/partners/{partner_id}/services/remove_by_product'.format(partner_id=partner_id),
            method='DELETE',
            params=dict(
                product_id=product_id,
            ),
            parser=self._plus_parse_response,
            error_detector=functools.partial(datasync_api_error_detector, success_statuses=(204, 404,)),
        )


def get_disk_api():
    return DiskApi()  # pragma: no cover
