# -*- coding: utf-8 -*-
from collections import namedtuple
import json
from time import time

from passport.backend.core.builders.base.base import RequestInfo
from passport.backend.core.builders.datasync_api.base import (
    datasync_api_error_detector,
    DatasyncApi,
)
from passport.backend.core.builders.datasync_api.exceptions import (
    BaseDatasyncApiError,
    DatasyncApiPermanentError,
)
from passport.backend.core.logging_utils.loggers import GraphiteLogger


def _make_fake_raw_response(status_code, content):
    fake_raw_response_fields = {
        'status_code': status_code,
        'content': content,
    }
    fake_response_class = namedtuple('FakeResponseClass', fake_raw_response_fields.keys())
    return fake_response_class(**fake_raw_response_fields)


def _make_batch_request_item(url, get_args=None):
    return {
        'method': 'GET',
        'relative_url': RequestInfo(url, get_args, None).url,
    }


class PersonalityApi(DatasyncApi):

    """
    Дока к апи: https://st.yandex-team.ru/VIDEOUI-3551
    Дока к батчевой ручке: https://wiki.yandex-team.ru/disk/mpfs/PlatformAPIBatchRequests/
    """

    _URLS_FOR_COLLECTION_PROCESSOR = [
        '/v1/personality/profile/passport/external_data',
    ]

    def __init__(self, url=None, useragent=None, timeout=None, retries=None, graphite_logger=None, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='personality_api')
        super(PersonalityApi, self).__init__(
            url=url,
            timeout=timeout,
            retries=retries,
            useragent=useragent,
            graphite_logger=graphite_logger,
            **kwargs
        )

    def _collection_processor(self, response):
        return response.get('items', [])

    def video_favorites(self, uid, offset=0, limit=20):
        return self._make_request(
            uid=uid,
            url_suffix='/v1/personality/profile/videosearch/likes',
            method='GET',
            params={
                'query.order.asc': False,
                'offset': offset,
                'limit': limit,
            },
            response_processor=self._collection_processor,
        )

    def maps_bookmarks(self, uid, offset=0, limit=20):
        return self._make_request(
            uid=uid,
            url_suffix='/v1/personality/profile/maps_common/bookmarks',
            method='GET',
            params={
                'offset': offset,
                'limit': limit,
            },
            response_processor=self._collection_processor,
        )

    def _single_object_error_detector(self, response, raw_response):
        datasync_api_error_detector(response, raw_response)
        if 'data' not in response:
            raise DatasyncApiPermanentError('`data` not in response')

    def _passport_external_data_single_object_processor(self, response):
        response['data'] = json.loads(response['data'])
        if response.get('meta'):
            response['meta'] = json.loads(response['meta'])
        return response

    def _passport_external_data_collection_processor(self, response):
        return [
            self._passport_external_data_single_object_processor(obj)
            for obj in self._collection_processor(response)
        ]

    def _get_response_processor_by_url(self, url):
        if url == '/v1/personality/profile/passport/external_data':
            return self._passport_external_data_collection_processor
        elif url.startswith('/v1/personality/profile/passport/external_data/'):
            return self._passport_external_data_single_object_processor
        else:
            return self._collection_processor

    def _make_batch_processor(self, single_processors):
        def response_processor(response):
            items = response.get('items', [])
            assert len(single_processors) == len(items)
            result = []
            for processor, item in zip(single_processors, items):
                fake_raw_response = _make_fake_raw_response(status_code=item['code'], content=item['body'])
                body = self.parse_json(fake_raw_response)
                try:
                    datasync_api_error_detector(body, fake_raw_response)
                    result.append(processor(body))
                except BaseDatasyncApiError as e:
                    result.append(e)
            return result

        return response_processor

    def batch_get(self, uid, request_array):
        """
        Получает список запросов в формате [{'url': ..., 'params': {...}}, ...]
        Возвращает список, в котором для каждого запроса либо ответ на него, либо исключение, которое должно возникнуть
        """
        return self._make_request(
            uid=uid,
            url_suffix='/v1/batch/request',
            method='POST',
            data={
                'items': [_make_batch_request_item(**item) for item in request_array],
            },
            response_processor=self._make_batch_processor([self._get_response_processor_by_url(item['url']) for item in request_array]),
        )

    def passport_external_data_get_all(self, uid, offset=0, limit=20):
        return self._make_request(
            uid=uid,
            url_suffix='/v1/personality/profile/passport/external_data',
            method='GET',
            params={
                'query.order.asc': False,
                'offset': offset,
                'limit': limit,
            },
            response_processor=self._passport_external_data_collection_processor,
        )

    def passport_external_data_get(self, uid, object_id):
        return self._make_request(
            uid=uid,
            url_suffix='/v1/personality/profile/passport/external_data/%s' % object_id,
            method='GET',
            error_detector=self._single_object_error_detector,
            response_processor=self._passport_external_data_single_object_processor,
        )

    def passport_external_data_update(self, uid, object_id, data, meta=None, modified_at=None):
        object_data = {
            'modified_at': int(modified_at or time()),
            'data': json.dumps(data),
        }
        if meta:
            object_data.update(meta=json.dumps(meta))
        self._make_request(
            uid=uid,
            url_suffix='/v1/personality/profile/passport/external_data/%s' % object_id,
            method='PUT',
            json_data=object_data,
        )

    def passport_external_data_delete(self, uid, object_id):
        self._make_request(
            uid=uid,
            url_suffix='/v1/personality/profile/passport/external_data/%s' % object_id,
            method='DELETE',
        )


def get_personality_api():
    return PersonalityApi()  # pragma: no cover
