# -*- coding: utf-8 -*-

import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger(__name__)


def get_drive_api():
    return DriveApi()


class BaseDriveApiError(BaseCoreError):
    pass


class DriveApiTemporaryError(BaseDriveApiError):
    pass


class DriveApiPermanentError(BaseDriveApiError):
    pass


class DriveApi(BaseBuilder, JsonParserMixin):
    base_error_class = BaseDriveApiError
    temporary_error_class = DriveApiTemporaryError
    parser_error_class = DriveApiTemporaryError

    def __init__(
        self,
        url=None,
        useragent=None,
        timeout=None,
        retries=None,
        graphite_logger=None,
        tvm_dst_alias='drive_api',
        **kwargs
    ):
        graphite_logger = graphite_logger or GraphiteLogger(service='drive_api')
        super(DriveApi, self).__init__(
            graphite_logger=graphite_logger,
            logger=log,
            retries=retries or settings.DRIVE_API_RETRIES,
            timeout=timeout or settings.DRIVE_API_TIMEOUT,
            tvm_dst_alias=tvm_dst_alias,
            url=url or settings.DRIVE_API_URL,
            useragent=useragent,
            **kwargs
        )

    def find_drive_session_id(self, drive_device_id):
        params = {'device_id': drive_device_id}
        return self._request_with_retries_simple(
            method='GET',
            url_suffix='api/staff/head/session',
            params=params,
            parser=self.parse_find_drive_session_response,
            error_detector=None,
        )

    def parse_find_drive_session_response(self, raw_response):
        if raw_response.status_code in {400, 404}:
            return
        if raw_response.status_code == 401:
            raise DriveApiPermanentError(self.format_response(raw_response))
        if raw_response.status_code != 200:
            raise self.temporary_error_class(self.format_response(raw_response))
        response = self.parse_json(raw_response)
        try:
            return response['session_id']
        except (KeyError, TypeError):
            raise self.temporary_error_class(self.format_response(raw_response))
