# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import (
    BaseBuilder,
    parser_trivial,
)
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers import GraphiteLogger


log = logging.getLogger('passport.edadeal')


class BaseEdadealError(BaseCoreError):
    pass


class EdadealPermanentError(BaseEdadealError):
    """Пятисотка или иная непредвиденная ошибка"""


class EdadealTemporaryError(BaseEdadealError):
    """Временная ошибка Edadeal - стоит поретраиться"""


def edadeal_error_detector(response, raw_response):
    if raw_response.status_code != 200:
        raise EdadealPermanentError(u'Request failed with code=%s; response=%s' % (
            raw_response.status_code,
            trim_message(raw_response.content.decode('utf-8')),
        ))


class EdadealApi(BaseBuilder):
    """
    Дока: https://st.yandex-team.ru/EDADEALBACKEND-870
    """
    base_error_class = BaseEdadealError
    temporary_error_class = EdadealTemporaryError
    parser_error_class = EdadealPermanentError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None,
                 graphite_logger=None, token=None, **kwargs):
        graphite_logger = graphite_logger or GraphiteLogger(service='edadeal')
        super(EdadealApi, self).__init__(
            url=url or settings.EDADEAL_URL,
            timeout=timeout or settings.EDADEAL_TIMEOUT,
            retries=retries or settings.EDADEAL_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger,
            **kwargs
        )
        self.token = token or settings.EDADEAL_TOKEN

    def update_plus_status(self, uid, is_active):
        return self._request_with_retries_simple(
            error_detector=edadeal_error_detector,
            parser=parser_trivial,
            url_suffix='/auth/v1/plus',
            method='POST',
            json_data={'yuid': uid, 'is_active': is_active},
            headers={
                'Authorization': 'Bearer %s' % self.token,
            },
        )


def get_edadeal():
    return EdadealApi()  # pragma: no cover
