# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.federal_configs_api.exceptions import (
    BaseFederalConfigsApiError,
    FederalConfigsApiInvalidResponseError,
    FederalConfigsApiNotFoundError,
    FederalConfigsApiPermanentError,
    FederalConfigsApiTemporaryError,
)
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from six.moves.urllib.parse import quote_plus


log = logging.getLogger('passport.federal_configs_api')


def federal_configs_api_http_error_handler(response):
    if response.status_code in (502, 503):
        raise FederalConfigsApiTemporaryError(u'Request failed with code=%s' % response.status_code)

    if response.status_code == 404:
        raise FederalConfigsApiNotFoundError()

    if response.status_code != 200:
        raise FederalConfigsApiPermanentError(u'Request failed with code=%s' % response.status_code)


class FederalConfigsApi(BaseBuilder, JsonParserMixin):
    base_error_class = BaseFederalConfigsApiError
    temporary_error_class = FederalConfigsApiTemporaryError
    parser_error_class = FederalConfigsApiInvalidResponseError

    def __init__(self, url=None, useragent=None, timeout=None, retries=None,
                 graphite_logger=None):
        super(FederalConfigsApi, self).__init__(
            url=url or settings.FEDERAL_CONFIGS_API_URL,
            timeout=timeout or settings.FEDERAL_CONFIGS_API_TIMEOUT,
            retries=retries or settings.FEDERAL_CONFIGS_API_RETRIES,
            logger=log,
            useragent=useragent,
            graphite_logger=graphite_logger or GraphiteLogger(service='federal_configs_api'),
            tvm_dst_alias='federal_configs_api',
        )

    def _make_request(self, method='GET', url_suffix='', params=None, data=None, **kwargs):
        return self._request_with_retries_simple(
            url_suffix=url_suffix,
            method=method,
            params=params,
            data=data,
            parser=self.parse_json,
            http_error_handler=federal_configs_api_http_error_handler,
            error_detector=None,
            **kwargs
        )

    def get_config_by_domain_id(self, domain_id):
        return self._make_request(
            url_suffix='/1/config/by_domain_id/%s/' % domain_id,
            params={'namespace': settings.FEDERAL_CONFIGS_API_NAMESPACE},
        )

    def get_config_by_entity_id(self, entity_id):
        return self._make_request(
            url_suffix='/1/config/by_entity_id/%s/' % quote_plus(entity_id),
            params={'namespace': settings.FEDERAL_CONFIGS_API_NAMESPACE},
        )


def get_federal_configs_api():
    return FederalConfigsApi()  # pragma: no cover


__all__ = [
    'get_federal_configs_api',
    'FederalConfigsApi',
]
